# coding: utf-8

import logging

from datetime import datetime
from dateutil.parser import parse as date_parse
from collections import namedtuple

from mail.pypg.pypg.common import transaction, qexec
from mail.pypg.pypg.query_conf import load_from_package
from mail.husky.husky.types import Task, ResultData, Errors
from ora2pg.sharpei import get_pg_dsn_from_sharpei
from pymdb.operations import UpdateMessages
from pymdb.queries import Queries

from .base import BaseTask

QUERIES = load_from_package(__package__, __file__)

log = logging.getLogger(__name__)


Label = namedtuple('Label', ['type', 'name'])


class UnlabelMessages(BaseTask):
    CHUNK_SIZE = 500

    name = Task.UnlabelMessages
    required_args = ['label']

    @property
    def label(self):
        return Label(**(self.task_args.get('label')))

    def _get_date_arg(self, name):
        dt = self.get_arg(name)
        return dt and date_parse(dt)

    @property
    def from_date(self):
        return self._get_date_arg('from_date') or datetime.min

    @property
    def to_date(self):
        return self._get_date_arg('to_date') or datetime.max

    @property
    def chunk_size(self):
        return self.task_args.get('chunk_size', self.CHUNK_SIZE)

    def run(self):
        dsn = self._get_dsn_by_uid()
        with transaction(dsn) as conn:
            queries = Queries(conn, self.uid)
            user = queries.user()
            if not user.is_here:
                return ResultData(
                    transfer_id=self.transfer_id,
                    error=Errors.NotSupported,
                    error_message='Try process user which is not here',
                    task_output=None,
                )
            label = next((l for l in queries.labels() if l.type == self.label.type and l.name == self.label.name), None)

        if label is None:
            log.info('No label found, should not proceed')
            return

        if label.message_count == 0:
            log.info('No messages with label, should not proceed')
            return

        log.info('Will remove label "{label}" in range [{from_date}, {to_date})'.format(
            label=label,
            from_date=self.from_date,
            to_date=self.to_date)
        )
        for chunk in self._process_cunk(dsn, label):
            log.info('Unlabel {} messages'.format(len(chunk)))

    def _get_dsn_by_uid(self):
        return get_pg_dsn_from_sharpei(
            sharpei=self.config.sharpei,
            uid=self.uid,
            dsn_suffix=self.config.maildb_dsn_suffix,
        )

    def _process_cunk(self, dsn, label):
        if self.from_date >= self.to_date:
            return
        while True:
            with transaction(dsn) as conn:
                cur = qexec(
                    conn=conn,
                    query=QUERIES.mids_by_label,
                    uid=self.uid,
                    lids=[label.lid],
                    from_date=self.from_date,
                    to_date=self.to_date,
                    limit=self.chunk_size,
                )
                mids = [int(*row) for row in cur]
                if len(mids) == 0:
                    return
                UpdateMessages(conn, self.uid)(
                    mids=mids,
                    seen=None,
                    recent=None,
                    deleted=None,
                    lids_del=[label.lid]
                )
                yield mids
