import logging
from sys import exc_info
from traceback import format_tb

from psycopg2 import IntegrityError
from werkzeug.wrappers import Request, Response

from mail.husky.husky.sharddb import GetUserShardIdError

from .json_helpers import jdumps
from mail.python.tvm_requests import Tvm, TvmCheckError, TvmInternalError  # noqa

log = logging.getLogger(__name__)


class TvmMiddleware(object):
    def __init__(self, app, tvm):
        """
        :type app:
        :type tvm: Tvm
        """
        self.app = app
        self.tvm = tvm

    def __call__(self, environ, start_response):
        request = Request(environ)
        ticket_hdr = request.headers.get('X-Ya-Service-Ticket', None)
        try:
            ticket_info = ticket_hdr and self.tvm.parse(ticket_hdr)
            environ['tvm_ticket'] = ticket_info
            return self.app(environ, start_response)
        except TvmCheckError as e:
            response = Response(str(e), mimetype='text/plain')
            response.status_code = 401
            return response(environ, start_response)


class ExceptionMiddleware(object):
    """The middleware we use."""

    def __init__(self, app):
        self.app = app

    def __call__(self, environ, start_response):
        """Call the application can catch exceptions."""
        appiter = None
        # just call the application and send the output back
        # unchanged but catch exceptions
        try:
            appiter = self.app(environ, start_response)
            for item in appiter:
                yield item
        # if an exception occours we get the exception information
        # and prepare a traceback we can render
        except Exception as exception:
            e_type, e_value, tb = exc_info()
            traceback = ['Traceback (most recent call last):']
            traceback += format_tb(tb)
            traceback.append('%s: %s' % (e_type.__name__, e_value))
            # we might have not a stated response by now. try
            # to start one with the status code 500 or ignore an
            # raised exception if the application already started one.
            try:
                start_response('500 INTERNAL SERVER ERROR', [('Content-Type', 'application/json')])
            except:
                pass
            yield jdumps(make_error(exception=exception, traceback=traceback)).encode('utf-8')


def make_error(exception, traceback):
    return dict(
        status='error',
        error=dict(
            code=ERROR_CODES.get(exception.__class__) or 1,
            message=str(exception),
            traceback=traceback,
        ),
    )


def make_custom_error(code, message, traceback=None):
    return dict(
        status='error',
        error=dict(
            code=code,
            message=message,
            traceback=traceback,
        ),
    )


ERROR_CODES = {
    Exception: 1,
    GetUserShardIdError: 2,
    IntegrityError: 3,
    TvmInternalError: 4,
}
