from aiopg.sa import Engine
from mail.husky.stages.worker.interactions.db.huskydb import HuskydbTransfer
from mail.husky.stages.worker.interactions.yasm_adaptor import YasmAdaptor
from mail.husky.stages.worker.settings.shard_processor import ShardProcessorSettings
from mail.python.theatre.roles import Director
from ora2pg.app.transfer_app import TransferApp

from .load_regulator import LoadRegulator
from .task_dispatcher import TaskDispatcher
from .shard_workload_updater import ShardWorkloadUpdater


class ShardProcessor(Director):
    def __init__(self,
                 huskydb_transfer: HuskydbTransfer,
                 task_dispatcher: TaskDispatcher,
                 load_regulator: LoadRegulator,
                 shard_workload_updater: ShardWorkloadUpdater) -> None:
        self.huskydb_transfer = huskydb_transfer
        self.task_dispatcher = task_dispatcher
        self.load_regulator = load_regulator
        self.shard_workload_updater = shard_workload_updater
        super().__init__(tasks=(
            self.task_dispatcher,
            self.load_regulator,
            self.shard_workload_updater,
        ))

    @classmethod
    async def async_init(cls, app: TransferApp, pg: Engine, shard_id: int, husky_cluster: str, settings: ShardProcessorSettings):
        huskydb_transfer = HuskydbTransfer(app.args, pg=pg._pool, shard_id=shard_id, husky_cluster=husky_cluster)
        task_dispatcher = TaskDispatcher(app, shard_id, huskydb_transfer, settings.task_dispatcher)
        shard_workload_updater = ShardWorkloadUpdater(
            shard_id=shard_id,
            default_max_workers=settings.task_dispatcher.worker_cnt,
            huskydb=huskydb_transfer,
            settings=settings.shard_workload_updater,
            set_max_workers=task_dispatcher.update_max_workers,
        )
        task_dispatcher.update_max_workers(await shard_workload_updater.get_max_workers())
        try:
            # TODO :: remove that query from HuskydbTransfer
            cluster_id = await huskydb_transfer.get_cluster_id(shard_id)
            is_in_cloud = True
        except ValueError:
            cluster_id = None
            is_in_cloud = False
        yasm_adaptor = YasmAdaptor(cluster_id, is_in_cloud)
        load_regulator = LoadRegulator(yasm_adaptor, task_dispatcher, settings.load_regulator, shard_id)
        return cls(
            huskydb_transfer=huskydb_transfer,
            task_dispatcher=task_dispatcher,
            load_regulator=load_regulator,
            shard_workload_updater=shard_workload_updater,
        )
