import asyncio
import logging
import os
import socket
import json
from functools import reduce
from operator import methodcaller
from typing import Dict, Any, Optional, Callable, TypeVar

import click
import uvloop
from aiopg.sa import create_engine
from psycopg2.extras import DictConnection

from mail.python.theatre.stages.base import Stage
from mail.python.tvm_requests import Tvm
from ora2pg.app import make_app_from_env, mkdir_p
from ora2pg.app.config_file import env_to_config_file, load_configs

from .interactions.db.huskydb import DEFAULT_HUSKY_CLUSTER
from .interactions.db.meta import get_buckets_metainfo
from .roles.buckets_provider import BucketsProvider
from .roles.director import Director
from .settings.app import Settings
from mail.husky.husky.types import Task


QLOUD_META_PATH = '/etc/qloud/meta.json'

T = TypeVar('T')


def and_then(init, *funcs, default: Callable[[], T] = lambda: None) -> T:
    return reduce(lambda v, f: v and f(v), funcs, init) or default()


def read_json_config(file_path) -> Optional[Dict]:
    if os.path.exists(file_path):
        with open(file_path) as fd:
            return json.load(fd)
    return None


def gethostname() -> str:
    """Get host FQDN"""
    def compose_fqdn(data) -> Optional[str]:
        try:
            return '.'.join((
                data["qloud_instance"],
                data['user_environment']['QLOUD_COMPONENT'],
                data["qloud_environment"],
                data["qloud_application"],
                data["qloud_project"],
                "stable.qloud-d.yandex.net",
            ))
        except KeyError:
            return None

    return and_then(
        read_json_config(QLOUD_META_PATH),
        compose_fqdn,
        default=socket.gethostname,
    )


def get_qloud_component() -> Optional[str]:
    return and_then(
        read_json_config(QLOUD_META_PATH),
        methodcaller('get', 'user_environment'),
        methodcaller('get', 'QLOUD_COMPONENT'),
    )


def get_tvm_from_conf(config):
    return Tvm(
        tvm_daemon_url=config['tvm'].get('daemon_url', 'http://localhost:1'),
        client_id=config['tvm']['client_id'],
        local_token=config['tvm'].get('local_token', None),
    )


def make_max_tries(config):
    max_tries = config.get('max_tries', {'default': 3})
    return {k: max_tries.get(k, max_tries['default']) for k in Task.__dict__.values() if type(k) == str}


class WorkerStage(Stage):
    def __init__(self, env: str, conf: Dict[str, Any], worker_name: str = None, husky_cluster: str = None):
        self.worker_name = worker_name or gethostname()
        self.husky_cluster = husky_cluster or get_qloud_component() or DEFAULT_HUSKY_CLUSTER
        self.conf = conf
        super(WorkerStage, self).__init__(env=env, name='husky-worker', settings=Settings)

    @staticmethod
    def make_transfer_app(config):
        app_kwargs = dict(
            log_filepath=config.get('log_dir') and os.path.join(config['log_dir'], 'husky.log'),
            log_level=config.get('log_level', logging.INFO),
            user_log_level=config.get('user_log_level', logging.DEBUG),
            need_user_log=config.get('need_user_log', True),
            verbose=config.get('verbose', False),
            tvm=get_tvm_from_conf(config),
            sync_subscriptions_wait_progression=config.get('sync_subscriptions_wait_progression'),
            max_tries=make_max_tries(config),
            fail_retry_wait_progression=config.get('fail_retry_wait_progression', None),
            husky_name=config.get('husky_name'),
            mds_id=config['mds_id'],
            bb_tvm_id=config.get('bb_tvm_id', None),
            tabs_mapping=config.get('tabs_mapping', []),
            message_count_limit=config.get('message_count_limit'),
        )
        return make_app_from_env(config['env'], **app_kwargs)

    async def create_app(self):
        db_meta = get_buckets_metainfo()
        pg_ro_mngr = create_engine(
            dsn=self.settings.db.pg_dsn(ro=True),
            maxsize=4,
            connection_factory=DictConnection,
            timeout=600,
        )
        pg_ro = await pg_ro_mngr.__aenter__()
        pg_mngr = create_engine(
            dsn=self.settings.db.pg_dsn(),
            maxsize=4,
            connection_factory=DictConnection,
        )
        pg = await pg_mngr.__aenter__()
        buckets_provider = BucketsProvider(huskydb_pg=pg, huskydb_ro=pg_ro, meta=db_meta, settings=self.settings.buckets_provider)
        director = Director(
            pg,
            self.settings,
            self.worker_name,
            self.husky_cluster,
            transfer_app=self.make_transfer_app(self.conf),
            db_meta=db_meta,
        )
        self.app.update(settings=self.settings, director=director, buckets_provider=buckets_provider)
        await director.start()
        await buckets_provider.start()

        async def shutdown(_):
            await buckets_provider.stop()
            await director.stop()
            await pg_mngr.__aexit__(None, None, None)
            await pg_ro_mngr.__aexit__(None, None, None)

        self.app.on_shutdown.append(shutdown)
        self.setup_routes(director=director)
        return self.app

    def setup_routes(self, director: Director):
        self.app.router.add_get('/unistat', director.unistat_handler, name='unistat')


@click.command()
@click.option('--host', default='::0', help='IP addr to bind to')
@click.option('--port', default=8080, help='HTTP port to listen')
@click.option('--env', help='Environment name')
@click.option('--app-config-path', help='Path to env config file')
@click.option('--env-config-path', help='Path to env config file')
@click.option('--worker-name', help='Worker name to be used for shard acquisition')
@click.option('--husky-cluster', help='Husky cluster name to be used for task grouping')
def run_main(host, port, app_config_path, env=None, env_config_path=None, worker_name=None, husky_cluster=None):
    asyncio.set_event_loop_policy(uvloop.EventLoopPolicy())

    if env_config_path is None:
        env_config_path = env_to_config_file(env)
    conf = load_configs([app_config_path, env_config_path])
    conf.get('log_dir') and mkdir_p(conf['log_dir'], mode=0o755)
    stage = WorkerStage(env=env, conf=conf, worker_name=worker_name, husky_cluster=husky_cluster)
    stage.run(host=host, port=port)
