package ru.yandex.gate.mail;

import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import ru.yandex.function.ByteArrayProcessable;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.uri.CgiParams;

public class ForwardedMessage {
    private static final CollectionParser<
        String,
        Set<String>,
        Exception>
        SET_PARSER = new CollectionParser<>(
            s -> s,
            LinkedHashSet::new);

    private final ByteArrayProcessable message;
    private final String subject;
    private final String from;
    private final String to;
    private final String msgTo;
    private final String date;
    private final String contentType;
    private final List<String> headers;
    private final Set<String>  types;
    private final Set<String> labels;
    private final boolean raw;

    public ForwardedMessage(
        final CgiParams params,
        final ByteArrayProcessable message)
        throws BadRequestException, IllegalArgumentException
    {
        from = params.getString("from");
        to = params.getString("to");
        msgTo = params.getString("msg-to", null);
        date = params.getString("date", null);
        subject = params.getString("subject");
        contentType = params.getString("content-type", null);
        raw = params.getBoolean("raw", false);
        types = params.getAll(
            "types",
            Collections.emptySet(),
            // Allow empty types:
            t -> t.isEmpty() ? Collections.emptySet() : SET_PARSER.apply(t));
        labels = params.getAll(
            "labels",
            Collections.emptySet(),
            SET_PARSER);
        checkTypes(types);
        headers = params.getAll("header");
        this.message = message;
    }

    public ByteArrayProcessable getMessage() {
        return message;
    }

    public String getSubject() {
        return subject;
    }

    public Set<String> getTypes() {
        return types;
    }

    public Set<String> getLabels() {
        return labels;
    }

    public String getFrom() {
        return from;
    }

    public String getTo() {
        return to;
    }

    public String getDate() {
        return date;
    }

    public String getMsgTo() {
        return msgTo;
    }

    public List<String> getHeaders() {
        return headers;
    }

    public String getContentType() {
        return contentType;
    }

    public boolean raw() {
        return raw;
    }

    private void checkTypes(final Set<String> types)
        throws IllegalArgumentException
    {
        for (String type: types) {
            try {
                if (!Server.typesMap.containsKey(Integer.parseInt(type))) {
                    throw new IllegalArgumentException(
                        "type " + type + " not found");
                }
            } catch (NumberFormatException e) {
                throw new IllegalArgumentException(
                    "Type must be number, got \"" + type + '\"');
            }
        }
    }
}
