package ru.yandex.gate.mail;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.SequenceInputStream;
import java.io.StringWriter;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;

import com.google.common.base.Charsets;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.HttpEntityWrapper;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.collection.Pattern;
import ru.yandex.function.ByteArrayCopier;
import ru.yandex.function.ByteArrayProcessable;
import ru.yandex.function.OutputStreamProcessorAdapter;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.ByteArrayProcessableAsyncConsumer;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.email.types.MessageType;
import ru.yandex.parser.uri.QueryConstructor;

//  https://wiki.yandex-team.ru/ps/so/Gatemail/

public class Server
    extends HttpProxy<Config>
    implements HttpAsyncRequestHandler<ByteArrayProcessable>
{
    private static final String INBOX_FID = "1";
    private static final String HANDLE = "/mail/store?";

    static final Map<Integer, String> typesMap;

    static {
        typesMap = new HashMap<>();
        for (MessageType type :  MessageType.values()) {
            typesMap.put(type.typeNumber(), type.toString());
        }
    }

    private final AsyncClient smtpGate;
    private final HttpHost smtpGateHost;

    private final AsyncClient blackbox;
    private final HttpHost blackboxHost;

    private final Cache<String, String> loginToUid;

    public Server(final Config config) throws IOException {
        super(config);
        this.smtpGate = client("smtpgate", config.smtpGate());
        this.smtpGateHost = config.smtpGate().host();
        this.blackbox = client("blackbox", config.blackbox());
        this.blackboxHost = config.blackbox().host();

        register(
            new Pattern<>("/send-mail", false),
            this,
            RequestHandlerMapper.POST);
        this.loginToUid = CacheBuilder.newBuilder()
            .maximumSize(100)
            .build();
    }

    public static void main(final String... args)
        throws ConfigException, IOException
    {
        main(new ServerFactory(), args);
    }

    public void send(
        ForwardedMessage msg,
        String uid,
        ProxySession session,
        FutureCallback<? super HttpResponse> callback)
        throws HttpException
    {
        QueryConstructor qc =
            new QueryConstructor(HANDLE, false);
        qc.append("fid", INBOX_FID);
        qc.append("uid", uid);
        qc.append("service", "iex");

        StringWriter configWriter = new StringWriter();
        JsonWriter config = JsonType.NORMAL.create(configWriter);
        try {
            config.startObject();
            config.key("options");
            config.startObject();
            config.key("use_filters");
            config.value(true);
            config.endObject(); // options
            config.key("user_info");
            config.startObject();
            config.key("email");
            config.value(msg.getTo());
            config.endObject(); // user_info
            config.key("mail_info");
            config.startObject();
            config.key("received_date");
            config.value(System.currentTimeMillis() / 1000L);
            config.key("labels");
            config.startObject();
            config.key("symbol");
            config.startArray();
            for (String label: msg.getLabels()) {
                config.value(label);
            }
            config.endArray(); // symbol
            config.key("system");
            config.startArray();
            for (String type: msg.getTypes()) {
                int iType = Integer.parseInt(type);
                String label = typesMap.get(iType);
                config.value(label);
            }
            config.endArray(); // system
            config.endObject(); // labels
            config.endObject(); // user_info
            config.endObject();
        } catch (IOException unreachable) {
            throw new HttpException("error creating store config", unreachable);
        }

        BasicAsyncRequestProducerGenerator request;
        try {
            DecodableByteArrayOutputStream eml =
                new DecodableByteArrayOutputStream();
            try (Writer writer =
                    new OutputStreamWriter(eml, StandardCharsets.UTF_8))
            {
                writer.write("From: ");
                writer.write(msg.getFrom());
                writer.write('\n');
                if (msg.getMsgTo() != null) {
                    writer.write("To: ");
                    writer.write(msg.getMsgTo());
                    writer.write('\n');
                }
                if (msg.getDate() != null) {
                    writer.write("Date: ");
                    writer.write(msg.getDate());
                    writer.write('\n');
                }
                if (msg.getHeaders() != null) {
                    for (String header : msg.getHeaders()) {
                        logger.info("Sending header: " + header);
                        writer.write(header);
                        writer.write('\n');
                    }
                }
                if (msg.getContentType() != null) {
                    writer.write("Content-type: ");
                    writer.write(msg.getContentType());
                    writer.write('\n');
                } else {
                    writer.write("Content-Type: text/html;charset=UTF-8\n");
                    writer.write("Content-Transfer-Encoding: base64\n");
                }
                writer.write("Subject: ");
                writer.write(msg.getSubject());
                writer.write('\n');
                writer.write('\n');
            }

            if (msg.raw()) {
                msg.getMessage().processWith(
                    new OutputStreamProcessorAdapter(eml));
            } else {
                eml.write(
                    Base64.getEncoder().encode(
                        msg.getMessage().processWith(ByteArrayCopier.INSTANCE)));
            }
            eml.write('\n');

            MultipartEntityBuilder payload = MultipartEntityBuilder.create();
            payload.addBinaryBody(
                "config",
                configWriter.toString().getBytes(Charsets.UTF_8),
                ContentType.APPLICATION_JSON,
                null);
            payload.addBinaryBody(
                "message",
                eml.toByteArray(),
                ContentType.create("message/rfc822"),
                null);

            // cause empty lines required https://st.yandex-team.ru/RTEC-4966
            request = new BasicAsyncRequestProducerGenerator(
                qc.toString(),
                new HttpEntityWrapper(payload.build()) {
                    @Override
                    public void writeTo(OutputStream outStream)
                        throws IOException
                    {
                        outStream.write('\r');
                        outStream.write('\n');
                        super.writeTo(outStream);
                    }

                    @Override
                    public InputStream getContent() throws IOException {
                        return new SequenceInputStream(
                            new ByteArrayInputStream(
                                "\r\n".getBytes(Charsets.US_ASCII)),
                            super.getContent());
                    }

                    @Override
                    public long getContentLength() {
                        return super.getContentLength() + 2;
                    }
                });
        } catch (IOException unreachable) {
            throw new HttpException("error combining multipart", unreachable);
        }
        request.addHeader(HttpHeaders.USER_AGENT, "GateMail");
        request.addHeader(YandexHeaders.X_REQUEST_ID, String.valueOf(
            session.context().getAttribute(HttpServer.SESSION_ID)));

        AsyncClient client = smtpGate.adjust(session.context());
        client.execute(
            smtpGateHost,
            request,
            BasicAsyncResponseConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            callback);
    }

    @Override
    public HttpAsyncRequestConsumer<ByteArrayProcessable> processRequest(
        final HttpRequest request,
        final HttpContext context)
    {
        return new ByteArrayProcessableAsyncConsumer();
    }

    @Override
    public void handle(
       final ByteArrayProcessable body,
       final HttpAsyncExchange exchange,
       final HttpContext context)
       throws HttpException
    {
        ProxySession session = new BasicProxySession(this, exchange, context);
        ForwardedMessage msg = new ForwardedMessage(session.params(), body);
        new SendingCallback(
            msg,
            session,
            new ResponseSendingCallback(session)
        ).execute();
    }


    private class SendingCallback
        extends AbstractFilterFutureCallback<JsonObject, HttpResponse>
    {
        private final ForwardedMessage message;
        private final ProxySession session;

        protected SendingCallback(
            ForwardedMessage message,
            ProxySession session,
            FutureCallback<? super HttpResponse> callback)
        {
            super(callback);
            this.message = message;
            this.session = session;
        }

        public void execute() {
            String login = message.getTo();
            String uid = loginToUid.getIfPresent(login);
            if (uid != null) {
                sendImp(uid);
                return;
            }
            String uri =
                "/blackbox?method=userinfo&userip=127.0.0.1&login="
                + login + "&format=json";
            AsyncClient client = blackbox.adjust(session.context());
            client.execute(
                blackboxHost,
                new BasicAsyncRequestProducerGenerator(uri),
                JsonAsyncTypesafeDomConsumerFactory.OK,
                session.listener().createContextGeneratorFor(client),
                this);
        }

        @Override
        public void completed(JsonObject blackboxResponse) {
            try {
                String uid = blackboxResponse
                    .get("users")
                    .get(0)
                    .get("uid").asMap()
                    .getString("value", null);
                if (uid == null) {
                    session.logger().warning(
                        "Sending to non-existent login " + message.getTo());
                    failed(new HttpException(
                        "Login " + message.getTo() + " not found"));
                } else {
                    loginToUid.put(message.getTo(), uid);
                    sendImp(uid);
                }
            } catch (JsonException e) {
                failed(e);
            }
        }

        private void sendImp(String uid) {
            try {
                send(message, uid, session, callback);
            } catch (HttpException e) {
                failed(e);
            }
        }
    }

    private static class ResponseSendingCallback
        extends AbstractProxySessionCallback<HttpResponse> {

        public ResponseSendingCallback(ProxySession session) {
            super(session);
        }

        @Override
        public void completed(HttpResponse response) {
            session.logger().info(
                "SmtpGate returned "
                + response.getStatusLine().getStatusCode()
                + " with Y-Context: "
                + response.getFirstHeader("Y-Context"));
            session.response(response);
        }
    }
}
