package ru.yandex.iex.proxy;

import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;

import ru.yandex.blackbox.BlackboxDbfield;
import ru.yandex.blackbox.BlackboxUserIdType;
import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.server.async.BaseAsyncServer;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.parser.uri.QueryConstructor;

public abstract class AbstractContext implements AbstractHandlersContext {
    private static final double MILLIS = 1000d;
    protected final IexProxy iexProxy;
    protected final Map<?, ?> json;
    protected final ProxySession session;
    protected final double operationDate;
    protected final String mdb;
    protected final String pgShard;
    protected final boolean hasTransferTimestamp;
    protected BlackboxUserinfo userInfo;

    public AbstractContext(final IexProxy iexProxy, final ProxySession session, final Map<?, ?> json)
        throws JsonUnexpectedTokenException
    {
        this.iexProxy = iexProxy;
        this.session = session;
        this.json = json;
        operationDate = ValueUtils.asDouble(json.get("operation_date"));
        mdb = session.params().getString("mdb", null);
        pgShard = session.params().getString("pgshard", null);
        hasTransferTimestamp =
            session.params().getString("transfer-timestamp", null) != null;
    }

    protected AbstractContext(final IexProxy iexProxy, final ProxySession session) {
        this.iexProxy = iexProxy;
        this.session = session;
        json = Collections.emptyMap();
        operationDate = System.currentTimeMillis();
        mdb = null;
        pgShard = null;
        hasTransferTimestamp = false;
    }

    @Override
    public abstract long prefix();

    public abstract Long lcn();

    public abstract BlackboxUserIdType prefixType();

    public abstract boolean prefixUrl();

    public String mdb() {
        if (mdb == null && userInfo != null) {
            return userInfo.dbfields().get(BlackboxDbfield.MDB);
        } else {
            return mdb;
        }
    }

    public String pgShard() {
        return pgShard;
    }

    @Override
    public String uid() {
        final String uid;
        if (prefixType() == BlackboxUserIdType.UID) {
            uid = Long.toString(prefix());
        } else {
            if (userInfo != null) {
                uid = Long.toString(userInfo.uid());
            } else {
                uid = null;
            }
        }
        return uid;
    }

    public String suid() {
        final String suid;
        if (prefixType() == BlackboxUserIdType.SUID) {
            suid = Long.toString(prefix());
        } else {
            if (userInfo != null) {
                suid = userInfo.dbfields().get(BlackboxDbfield.SUID);
            } else {
                suid = null;
            }
        }
        return suid;
    }

    @Override
    public IexProxy iexProxy() {
        return iexProxy;
    }

    @Override
    public ProxySession session() {
        return this.session;
    }

    @Override
    public Map<?, ?> json() {
        return this.json;
    }

    public double operationDate() {
        return this.operationDate;
    }

    public long operationDateMillis() {
        return (long) (operationDate * MILLIS);
    }

    @SuppressWarnings("unused")
    public String userTag() {
        return prefixType().name().toLowerCase(Locale.ROOT) + '=' + prefix();
    }

    public boolean hasTransferTimestamp() {
        return hasTransferTimestamp;
    }

    public String sessionId() {
        return session.context().getAttribute(
            BaseAsyncServer.SESSION_ID,
            String.class);
    }

    public BlackboxUserinfo userInfo() {
        return userInfo;
    }

    public void setUserInfo(final BlackboxUserinfo userInfo) {
        this.userInfo = userInfo;
    }

    public String filterSearchUri(final Collection<String> mids) throws BadRequestException {
        if (mids == null) {
            return null;
        }
        QueryConstructor query = new QueryConstructor(filterSearchUri());
        query.append("uid", uid());
        /*if (suid() != null) {
            query.append("suid", suid());
        }*/
        if (mdb() != null) {
            query.append("mdb", mdb());
        }
        for (String mid : mids) {
            query.append("mids", mid);
        }
        return query.toString();
    }

    public String filterSearchUri(final Iterator<String> midsIterator, final int count) throws BadRequestException {
        if (midsIterator == null || !midsIterator.hasNext()) {
            return null;
        }
        QueryConstructor query = new QueryConstructor(filterSearchUri());
        query.append("uid", uid());
        if (mdb() != null) {
            query.append("mdb", mdb());
        }
        for (int i = 0; i < count && midsIterator.hasNext(); i++) {
            query.append("mids", midsIterator.next());
        }
        return query.toString();
    }
}
