package ru.yandex.iex.proxy;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.logging.Level;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.HttpExceptionConverter;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.iex.proxy.xutils.mailsender.BasicMailSender;
import ru.yandex.iex.proxy.xutils.mailsender.EmptyForwarder;
import ru.yandex.iex.proxy.xutils.mailsender.MultiIexForwarder;
import ru.yandex.json.async.consumer.JsonAsyncDomConsumer;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;

public abstract class AbstractEntityHandler<T extends AbstractEntityContext>
    implements HttpAsyncRequestHandler<Object>
{
    protected final IexProxy iexProxyInstance;
    protected final Consumer<AbstractEntityContext> forwarder;

    public AbstractEntityHandler(final IexProxy iexProxy) {
        iexProxyInstance = iexProxy;
        forwarder = EmptyForwarder.INSTANCE;
    }

    public AbstractEntityHandler(final IexProxy iexProxy, final String name) {
        iexProxyInstance = iexProxy;
        List<ImmutableIexForwarderConfig> configs =
            iexProxy.config().forwardersConfig().forwarder(name);
        if (configs != null) {
            forwarder = new MultiIexForwarder(new BasicMailSender(), configs);
            iexProxy.logger().info("Forwader configured " + name);
        } else {
            forwarder = EmptyForwarder.INSTANCE;
        }
    }

    @Override
    public HttpAsyncRequestConsumer<Object> processRequest(
            final HttpRequest request,
            final HttpContext context)
        throws HttpException
    {
        if (!(request instanceof HttpEntityEnclosingRequest)) {
            throw new BadRequestException("Payload expected");
        }
        return new JsonAsyncDomConsumer(
            ((HttpEntityEnclosingRequest) request).getEntity());
    }

    protected abstract void handle(final T context)
        throws JsonUnexpectedTokenException, BadRequestException;

    @Override
    public void handle(
        final Object payload,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        ProxySession session =
            new BasicProxySession(iexProxyInstance, exchange, context);
        try {
            if (payload instanceof Map) {
                Map<?, ?> json = ValueUtils.asMap(payload);
                T entityContext = createContext(iexProxyInstance, session, json);

                handle(entityContext);
                // forward depersonalized mail for iex monitoring reasons
                forwarder.accept(entityContext);
            } else {
                throw new BadRequestException("POST invalid JSON");
            }
        } catch (BadRequestException e) {
            Map<String, String> res = new HashMap<>();
            res.put("error", e.getMessage());
            String response = JsonType.NORMAL.toString(res);
            session.logger().log(
                Level.WARNING,
                "Bad request: " + response,
                e);
            session.response(YandexHttpStatus.SC_OK, response);
        } catch (JsonException | ClassCastException e) {
            final String message = "Failed to handle request: ";
            session.logger().log(Level.SEVERE, message + payload.toString(), e);
            session.handleException(HttpExceptionConverter.toHttpException(e));
        } catch (Exception e) {
            final String message = "Failed to handle request, undefined error: ";
            session.logger().log(Level.SEVERE, message, e);
            session.handleException(HttpExceptionConverter.toHttpException(e));
        }
    }

    protected abstract T createContext(
            IexProxy iexProxy,
            ProxySession session,
            Map<?, ?> json)
        throws HttpException, JsonUnexpectedTokenException;
}

