package ru.yandex.iex.proxy;

import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.message.BasicHeader;

import ru.yandex.client.wmi.FilterSearchErrorSuppressingFutureCallback;
import ru.yandex.client.wmi.FilterSearchResponseConsumerFactory;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.search.document.mail.FirstlineMailMetaInfo;
import ru.yandex.search.document.mail.MailMetaInfo;

public abstract class AbstractFilterSearchCallback<T>
    implements FutureCallback<List<FirstlineMailMetaInfo>>
{
    public static final int BATCH_MIDS_SIZE = 450;

    private static final String SPAM = "spam";

    protected final AbstractContext context;
    protected final Set<String> mids;
    protected final FutureCallback<? super List<T>> callback;
    protected final long start = System.currentTimeMillis();

    public AbstractFilterSearchCallback(
        final AbstractContext context,
        final FutureCallback<? super List<T>> callback,
        final Set<String> mids)
    {
        this.context = context;
        this.callback = callback;
        this.mids = mids;
    }

    public boolean skipSpam() {
        return true;
    }

    public boolean skipEmptyEntities() {
        return true;
    }

    public abstract AbstractCallback<T> subMessageCallback(final IndexationContext<T> context);

    public abstract void executeSubCallback(final AbstractCallback<T> callback);

    public void addMetasWithEmptyFacts(final List<FirstlineMailMetaInfo> metas) {
    }

    public void execute() {
        MultiFutureCallback<List<FirstlineMailMetaInfo>> multiCallback =
            new MultiFutureCallback<>(new AbstractFilterFutureCallback<>(this)
            {
                @Override
                public void completed(final List<List<FirstlineMailMetaInfo>> data) {
                    if (data == null) {
                        //context.session().logger().info("AbstractFilterFutureCallback.completed: data = null");
                        callback.completed(null);
                        return;
                    }
                    //context.session().logger().info("AbstractFilterFutureCallback.completed for " + data.size()
                    //    + " batches (" + mids.size() + " mids)");
                    if (data.size() == 1) {
                        callback.completed(data.get(0));
                    } else {
                        List<FirstlineMailMetaInfo> metaInfos = new ArrayList<>();
                        //int i = 0;
                        for (List<FirstlineMailMetaInfo> metaInfo : data) {
                            //context.session().logger().info("AbstractFilterFutureCallback.completed: batch " + ++i
                            //    + ", batchSize=" + metaInfo.size());
                            metaInfos.addAll(metaInfo);
                        }
                        callback.completed(metaInfos);
                    }
                }

                @Override
                public void failed(Exception e) {
                    context.session().logger().log(Level.SEVERE, "AbstractFilterFutureCallback failed for "
                        + mids.size() + "mids");
                }
            });
        try {
            final AsyncClient httpClient = context.filterSearchClient();
            final String tvmTicket = context.filterSearchTvm2Ticket();
            if (mids.size() > BATCH_MIDS_SIZE) {
                Iterator<String> it = mids.iterator();
                //int i = 0;
                for (int n = 0; n < mids.size(); n += BATCH_MIDS_SIZE) {
                    //i++;
                    executeFilterSearch(
                        context.filterSearchUri(it, BATCH_MIDS_SIZE),
                        httpClient,
                        tvmTicket,
                        multiCallback.newCallback());
                }
                //context.session().logger().info("AbstractFilterFutureCallback send " + i
                //    + " requests to /filter_search (" + mids.size() + " mids)");
            } else {
                executeFilterSearch(context.filterSearchUri(mids), httpClient, tvmTicket, multiCallback.newCallback());
                //context.session().logger().info("AbstractFilterFutureCallback send 1 request to /filter_search ("
                //    + mids.size() + " mids)");
            }
            multiCallback.done();
        } catch (URISyntaxException | BadRequestException e) {
            callback.failed(e);
        }
    }

    @Override
    public void cancelled() {
        callback.cancelled();
    }

    @Override
    public void failed(final Exception e) {
        callback.failed(e);
    }

    @Override
    public void completed(final List<FirstlineMailMetaInfo> metas) {
        context.iexProxy().filterSearchCompleted(System.currentTimeMillis() - start);
        try {
            if (metas == null || metas.isEmpty()) {
                context.session().logger().info("All mids from set " + mids + " have gone");
                callback.completed(Collections.emptyList());
            } else {
                MultiFutureCallback<T> callback = new MultiFutureCallback<>(this.callback);
                List<AbstractCallback<T>> callbacks = new ArrayList<>(metas.size());
                List<FirstlineMailMetaInfo> metasWithoutEntities = new ArrayList<FirstlineMailMetaInfo>();
                for (final FirstlineMailMetaInfo meta : metas) {
                    final IndexationContext<T> idxCtx = new IndexationContext<>(context, meta, null);
                    context.session().logger().info("Entities for mid <" + idxCtx.mid() + ">: " + idxCtx.entities());
                    if (idxCtx.entities().isEmpty() && skipEmptyEntities()) {
                        logSkippedMessage(meta, "no entities is associated");
                        metasWithoutEntities.add(meta);
                        continue;
                    }
                    boolean skipSpam = idxCtx.folderType().equals(SPAM) && skipSpam();
                    boolean skipByFromEmails = false;
                    for (Map.Entry<String, EntityOptions> entry : idxCtx.entities().entrySet()) {
                        if (skipSpam && !entry.getValue().skipSpam()) {
                            skipSpam = false;
                        }
                        List<?> emails = entry.getValue().fromEmails();
                        if (emails != null && emails.size() > 0) {
                            skipByFromEmails = true;
                        }
                        if (skipByFromEmails && emails != null && emails.contains(idxCtx.email())) {
                            skipByFromEmails = false;
                        }
                    }
                    if (skipSpam || skipByFromEmails) {
                        if (skipSpam) {
                            logSkippedMessage(meta, "folder-type is spam");
                        }
                        metasWithoutEntities.add(meta);
                        continue;
                    }
                    idxCtx.callback(callback.newCallback());
                    callbacks.add(subMessageCallback(idxCtx));
                }
                if (!metasWithoutEntities.isEmpty()) {
                    this.addMetasWithEmptyFacts(metasWithoutEntities);
                }
                if (callbacks.isEmpty()) {
                    context.session().logger().info(
                        "No entities is associated with any messages. Skipped all messages");
                    this.callback.completed(Collections.emptyList());
                } else {
                    callback.done();
                    for (AbstractCallback<T> fastCallback : callbacks) {
                        executeSubCallback(fastCallback);
                    }
                }
            }
        } catch (HttpException | JsonUnexpectedTokenException e) {
            failed(e);
        }
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    private void executeFilterSearch(
        final String url,
        final AsyncClient httpClient,
        final String tvmTicket,
        final FutureCallback<List<FirstlineMailMetaInfo>> callback)
        throws URISyntaxException
    {
        httpClient.execute(
            new HeaderAsyncRequestProducerSupplier(
                new AsyncGetURIRequestProducerSupplier(url),
                new BasicHeader(YandexHeaders.X_YA_SERVICE_TICKET, tvmTicket)),
            new StatusCheckAsyncResponseConsumerFactory<>(
                HttpStatusPredicates.OK,
                new FilterSearchResponseConsumerFactory(
                    context.prefix(),
                    Objects.toString(context.lcn(), null))),
            context.session().listener().createContextGeneratorFor(httpClient),
            new FilterSearchErrorSuppressingFutureCallback<>(
                callback,
                Collections.emptyList(),
                context.session().logger()));
    }

    private void logSkippedMessage(
        final FirstlineMailMetaInfo meta,
        final String message)
    {
        context.session().logger().info(
            "Skipping message: mid: " + meta.get(MailMetaInfo.MID)
            + ", types: " + meta.messageTypes().toString()
            + " : " + message);
    }
}

