package ru.yandex.iex.proxy;

import java.io.IOException;
import java.util.List;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.nio.protocol.BasicAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.blackbox.BlackboxDbfield;
import ru.yandex.blackbox.BlackboxUserIdType;
import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.blackbox.BlackboxUserinfoRequest;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.parser.uri.CgiParams;

public abstract class AbstractUseredGetHandler<T extends AbstractContext>
    implements HttpAsyncRequestHandler<HttpRequest>
{
    private static final String MDB = "mdb";
    private static final String UID = "uid";
    protected final IexProxy iexProxy;

    AbstractUseredGetHandler(final IexProxy iexProxy) {
        this.iexProxy = iexProxy;
    }

    protected abstract T createContext(
        final ProxySession session,
        final IexProxy iexProxy)
        throws HttpException;

    protected abstract void completed(final T context) throws HttpException;

    public IexProxy iexProxy() {
        return iexProxy;
    }

    @Override
    public BasicAsyncRequestConsumer processRequest(
        final HttpRequest request,
        final HttpContext context)
    {
        return new BasicAsyncRequestConsumer();
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException, IOException
    {
        handle(new BasicProxySession(iexProxy, exchange, context));
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    private void handle(final ProxySession session) throws HttpException {
        final CgiParams cgiParams = session.params();
        final Logger logger = session.logger();
        final long uid = cgiParams.getLong(UID);
        final String mdb = cgiParams.getString(MDB, null);
        final Long suid = cgiParams.getLong("suid", null);
        boolean blackboxRequired = false;

        if (mdb == null) {
            logger.fine("Missing &mdb parameter. Will blackbox.");
            blackboxRequired = true;
        } else {
            if (!mdb.equalsIgnoreCase("pg")) {
                if (suid == null) {
                    logger.fine("Missing &suid parameter. Will blackbox.");
                    blackboxRequired = true;
                }
            }
        }

        final T context =
            createContext(session, iexProxy);

        final BlackboxUserinfoRequest bbRequest =
            new BlackboxUserinfoRequest(BlackboxUserIdType.UID, uid)
                .requiredDbfields(BlackboxDbfield.MDB, BlackboxDbfield.SUID);

        if (blackboxRequired) {
            BlackboxClient client;
            if (BlackboxUserinfo.corp(uid)) {
                client = iexProxy.corpBlackboxClient();
                bbRequest.addHeader(
                    YandexHeaders.X_YA_SERVICE_TICKET,
                    iexProxy.corpBlackboxTvm2Ticket());
            } else {
                client = iexProxy.blackboxClient();
                bbRequest.addHeader(
                    YandexHeaders.X_YA_SERVICE_TICKET,
                    iexProxy.blackboxTvm2Ticket());
            }
            client = client.adjust(session.context());
            client.userinfo(
                bbRequest,
                session.listener().createContextGeneratorFor(client),
                new UserInfoProxyCallback<T>(context, this));
        } else {
            final BlackboxUserinfo bbInfo = new BlackboxUserinfo(uid);
            if (suid != null) {
                bbInfo.dbfields().put(
                    BlackboxDbfield.SUID,
                    Long.toString(suid));
            }
            if (mdb != null) {
                bbInfo.dbfields().put(BlackboxDbfield.MDB, mdb);
            }
            context.setUserInfo(bbInfo);
            completed(context);
        }
    }

    private static class UserInfoProxyCallback<T extends AbstractContext>
        extends AbstractProxySessionCallback<List<BlackboxUserinfo>>
    {
        private final T context;
        private final AbstractUseredGetHandler<T> handler;

        UserInfoProxyCallback(
            final T context,
            final AbstractUseredGetHandler<T> handler)
        {
            super(context.session());
            this.context = context;
            this.handler = handler;
        }

        @Override
        public void completed(final List<BlackboxUserinfo> infos) {
            final BlackboxUserinfo info = infos.get(0);
            context.setUserInfo(info);
            try {
                handler.completed(context);
            } catch (HttpException e) {
                failed(e);
            }
        }
    }
}
