package ru.yandex.iex.proxy;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;

import org.apache.http.HttpException;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.iex.proxy.xutils.HashUtils;
import ru.yandex.iex.proxy.xutils.StringUtils;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;

public class ActionContext extends AbstractEntityContext {
    public static final String TYPE = "@type";
    public static final String MICRO = "micro";
    public static final String UNKNOWN = "unknown";
    public static final String TYPE1 = "type";
    public static final String ORIGIN = "origin";
    public static final String WIDGET_SUBTYPE = "widget_subtype";
    public static final String CONFIRM_EMAIL = "confirm_email";
    public static final String CONFIRM_URL = "confirm_url";
    public static final String RESTORE_URL = "restore_url";
    public static final String REMAIN_URL = "remain_url";
    public static final String UNSUBSCRIBE_URL = "unsubscribe_url";
    public static final String UNDEFINED = "undefined";
    public static final String REGISTRATION = "registration";
    public static final String IEX_MSG = "iex_msg";
    public static final String UNSUBSCRIBE = "unsubscribe";
    public static final String URL = "url";
    public static final String POTENTIAL_ACTION = "potentialAction";
    public static final String HANDLER = "handler";
    public static final String NAME = "name";
    public static final String DESCRIPTION = "description";
    public static final String ACTION = "action";
    public static final String SUBJECT = "subject";
    public static final String PASSWORD_CHANGED = "password_changed";
    public static final String JSON_PARSING_ERROR = "Json parsing exception";
    private static final HashSet<String> ACCEPTABLE_TYPES = new HashSet<>();
    private static final String[] PASSWORD_CHANGED_STATUS = {
        "password has been changed",
        "Был произведен вход в вашу учетную запись в Твиттере",
        "Вход из нового браузера",
        "Пароль успешно изменён",
        "Вы изменили свой пароль",
        "Доступ к аккаунту восстановлен"
    };

    static {
        ACCEPTABLE_TYPES.add("ConfirmAction");
        ACCEPTABLE_TYPES.add("ViewAction");
        ACCEPTABLE_TYPES.add("SaveAction");
        ACCEPTABLE_TYPES.add("ReviewAction");
    }

    private Object actionJson = null;

    ActionContext(
        final IexProxy iexProxy,
        final ProxySession session,
        final Map<?, ?> json)
        throws HttpException, JsonUnexpectedTokenException
    {
        super(iexProxy, session, json);
        tryToParseAnyMicro();
        tryToParseRegistration();
        tryToParseUnsubscribe();
        tryToParseListUnsubscribe();
        changeWidgetSubtype(session.params().getOrNull(SUBJECT));
    }

    @Override
    public void response() {
        session.response(
            YandexHttpStatus.SC_OK,
            JsonType.NORMAL.toString(actionJson));
    }

    private void changeWidgetSubtype(final String subject) {
        if (subject != null) {
            for (String status : PASSWORD_CHANGED_STATUS) {
                if (subject.contains(status)) {
                    XJsonUtils.pushToMapElessAndValueNotNull(
                        actionJson,
                        WIDGET_SUBTYPE,
                        PASSWORD_CHANGED);
                }
            }
        }
    }

    private String renameTypeAction(final String b) {
        return XStrAlgo.fromCCtoSC(b);
        // ConfirmAction -> confirm_action
        // ViewAction -> view_action
        // SaveAction -> save_action
        // ReviewAction -> review_action
    }

    private boolean isEmailMessageAction(final Map<?, ?> json) {
        String type = "";
        if (json.containsKey(TYPE1)) {
            type = (String) json.get(TYPE1);
        } else if (json.containsKey(TYPE)) {
            type = (String) json.get(TYPE);
        } else if (json.containsKey(ACTION)) {
            try {
                Object innerType =
                    XJsonUtils.getNodeByPathOrNull(json, ACTION, TYPE1);
                if (innerType instanceof String) {
                    type = (String) innerType;
                }
            } catch (JsonUnexpectedTokenException e) {
            }
        }
        return type.equals("EmailMessage") || isAcceptableTypes(type);
    }

    private boolean isAcceptableTypes(final String type) {
        String cmlCaseType = type.replace("http://schema.org/", "");
        return ACCEPTABLE_TYPES.contains(cmlCaseType);
    }

    private void tryToParseListUnsubscribe() {
        if (actionJson == null) {
            actionJson = new HashMap<>();
        }
        Object tikaText = XJsonUtils.getNodeByPathOrNullEless(
            json,
            ActionEntityHandler.LIST_UNSUBSCRIBE,
            "getbody",
            "text");
        if (tikaText instanceof String) {
            String t = (String) tikaText;
            String rListu = XRegexpUtils.getListUnsubscribe(t);
            if (!rListu.isEmpty()) {
                pushListUnsubscribe(rListu);
            }
        }
    }

    private void tryToParseAnyMicro() {
        try {
            String microFormat = "json";
            Object rawData =
                XJsonUtils.getNodeByPathOrNull(json, MICRO, UNKNOWN);
            if (rawData == null) {
                rawData =
                    XJsonUtils.getNodeByPathOrNull(json, "microhtml", UNKNOWN);
                microFormat = "microdata";
            }
            if (rawData instanceof List) {
                for (Object x : ValueUtils.asList(rawData)) {
                    if (x instanceof Map && ((Map) x).containsKey(ACTION)) {
                        rawData = ValueUtils.asMap(x);
                        break;
                    }
                }
            }
            if (!(rawData instanceof Map)
                || !isEmailMessageAction(ValueUtils.asMap(rawData)))
            {
                rawData = null;
            }
            if (rawData != null) {
                actionJson = new HashMap<String, Object>();
                Object actionType = parseActionType(rawData);
                Object url = parseUrl(rawData);
                Object description = parseDescription(rawData);
                Object name = parseName(rawData);
                pushWidgetSubtype(actionType);
                pushOrigin(MICRO);
                pushMicroFormat(microFormat);
                pushRawData(rawData);
                pushUrl(url);
                pushDescription(description);
                pushName(name);
            }
        } catch (JsonUnexpectedTokenException e) {
            session.logger().log(Level.SEVERE, JSON_PARSING_ERROR, e);
        }
    }

    private boolean isUnsubscribe(final String url) {
        return url != null
            && url.toLowerCase(Locale.ROOT).contains(UNSUBSCRIBE);
    }

    private void tryToParseRegistration() {
        try {
            Object registration =
                XJsonUtils.getNodeByPathOrNull(json, REGISTRATION);
            if (actionJson != null || registration == null) {
                return;
            }
            Set<String> confirmUrls = new HashSet<>();
            Set<String> restoreUrls = new HashSet<>();
            Set<String> unsubscribeUrls = new HashSet<>();
            boolean licenseKeyFound = false;
            if (registration instanceof List) {
                for (Object x : ValueUtils.asList(registration)) {
                    if (!(x instanceof Map)) {
                        continue;
                    }
                    Map<?, ?> xm = ValueUtils.asMap(x);
                    String confirmUrl = (String) xm.get(CONFIRM_URL);
                    if (confirmUrl != null) {
                        if (isUnsubscribe(confirmUrl)) {
                            unsubscribeUrls.add(confirmUrl);
                        } else {
                            confirmUrls.add(confirmUrl);
                        }
                        continue;
                    }
                    String restoreUrl = (String) xm.get(RESTORE_URL);
                    if (restoreUrl != null) {
                        if (isUnsubscribe(restoreUrl)) {
                            unsubscribeUrls.add(restoreUrl);
                        } else {
                            restoreUrls.add(restoreUrl);
                        }
                        continue;
                    }
                    String unsubscribeUrl = (String) xm.get(UNSUBSCRIBE_URL);
                    if (unsubscribeUrl != null) {
                        unsubscribeUrls.add(unsubscribeUrl);
                        continue;
                    }
                    if (xm.containsKey("license_key")) {
                        licenseKeyFound = true;
                        continue;
                    }
                    String remainUrl = (String) xm.get(REMAIN_URL);
                    if (isUnsubscribe(remainUrl)) {
                        unsubscribeUrls.add(remainUrl);
                    }
                }
            }
            String type = UNDEFINED;
            Map<String, Object> result = new HashMap<>();
            if (licenseKeyFound) {
                type = "product_key";
            } else if (!restoreUrls.isEmpty()) {
                type = "restore_password";
                selectOneUrl(restoreUrls, result);
            } else if (!confirmUrls.isEmpty()) {
                type = CONFIRM_EMAIL;
                selectOneUrl(confirmUrls, result);
            } else if (!unsubscribeUrls.isEmpty()) {
                type = UNSUBSCRIBE;
                selectOneUrl(unsubscribeUrls, result);
            } else {
                result.put(IEX_MSG, "url not found");
            }
            result.put(WIDGET_SUBTYPE, type);
            result.put(ORIGIN, "regexp");
            actionJson = result;
        } catch (JsonUnexpectedTokenException e) {
            session.logger().log(Level.SEVERE, JSON_PARSING_ERROR, e);
        }
    }

    private void selectOneUrl(
        final Set<String> urls,
        final Map<String, Object> result)
    {
        if (urls.size() > 1) {
            String urlWithHash = getUrlWithHashOrNull(urls);
            if (urlWithHash != null) {
                urls.clear();
                urls.add(urlWithHash);
            } else {
                result.put(IEX_MSG, "too many urls");
            }
        }
        List<Object> urlsList = new ArrayList<>(urls);
        if (urlsList.size() == 1) {
            result.put(URL, urlsList.get(0));
        } else if (urlsList.size() > 1) {
            result.put("urls", urlsList);
        }
    }

    private String getUrlWithHashOrNull(final Set<String> urls) {
        String result = null;
        int cnt = 0;
        for (String x : urls) {
            if (HashUtils.isUrlContainsHash(x)) {
                cnt++;
                result = x;
            }
        }
        if (cnt != 1) {
            result = null;
        }
        return result;
    }

    private void tryToParseUnsubscribe() {
        try {
            // TODO: make chain of actors
            if (actionJson == null) {
                actionJson =
                    XJsonUtils.getNodeByPathOrNull(json, UNSUBSCRIBE);
                if (actionJson instanceof Map) {
                    Map<?, ?> m = XJsonUtils.asMap(actionJson);
                    Map<String, Object> result = new HashMap<>();
                    result.put(WIDGET_SUBTYPE, UNSUBSCRIBE);
                    result.put(ORIGIN, UNSUBSCRIBE);
                    if (m.containsKey(URL)) {
                        result.put(URL, m.get(URL));
                    }
                    actionJson = result;
                }
            }
        } catch (JsonUnexpectedTokenException e) {
        }
    }

    private void pushWidgetSubtype(final Object ws) {
        try {
            String type = UNDEFINED;
            if (ws instanceof String) {
                type = (String) ws;
            }
            XJsonUtils.pushToMap(
                actionJson,
                WIDGET_SUBTYPE,
                renameTypeAction(type));
        } catch (JsonUnexpectedTokenException e) {
        }
    }

    private void pushOrigin(final String origin) {
        try {
            XJsonUtils.pushToMap(
                actionJson,
                ORIGIN,
                origin);
        } catch (JsonUnexpectedTokenException e) {
        }
    }

    private void pushMicroFormat(final String format) {
        try {
            XJsonUtils.pushToMap(
                actionJson,
                "micro_format",
                format);
        } catch (JsonUnexpectedTokenException e) {
        }
    }

    private void pushRawData(final Object rawData) {
        try {
            XJsonUtils.pushToMap(
                actionJson,
                "rawdata",
                rawData);
        } catch (JsonUnexpectedTokenException e) {
        }
    }

    private void pushUrl(final Object url) {
        XJsonUtils.pushToMapElessAndValueNotNull(actionJson, URL, url);
    }

    private void pushDescription(final Object desc) {
        XJsonUtils.pushToMapElessAndValueNotNull(
            actionJson,
            DESCRIPTION,
            desc);
    }

    private void pushName(final Object name) {
        XJsonUtils.pushToMapElessAndValueNotNull(
            actionJson,
            NAME,
            name);
    }

    private void pushListUnsubscribe(final Object url) {
        XJsonUtils.pushToMapElessAndValueNotNull(
            actionJson,
            ActionEntityHandler.LIST_UNSUBSCRIBE,
            url);
        XJsonUtils.pushToMapElessAndValueNotNull(
            actionJson,
            ORIGIN,
            "header");
    }
    //parsers

    private Object parseUrl(final Object map) {
        Object url = XJsonUtils.getNodeByPathOrNullEless(
            map,
            POTENTIAL_ACTION,
            "target");
        if (url == null) {
            url = XJsonUtils.getNodeByPathOrNullEless(
                map,
                POTENTIAL_ACTION,
                HANDLER,
                URL);
        }
        if (url == null) {
            url = XJsonUtils.getNodeByPathOrNullEless(
                map,
                ACTION,
                URL);
        }
        if (url instanceof String && isUrlHtmlEscaped(url)) {
            url = StringUtils.unescapeHtml3((String) url);
        }
        return url;
    }

    private Object parseDescription(final Object map) {
        Object desc = XJsonUtils.getNodeByPathOrNullEless(map, DESCRIPTION);
        return desc;
    }

    private Object parseName(final Object map) {
        Object name = XJsonUtils.getNodeByPathOrNullEless(
            map,
            POTENTIAL_ACTION,
            NAME);
        if (name == null) {
            name = XJsonUtils.getNodeByPathOrNullEless(
                map,
                ACTION,
                NAME);
        }
        return name;
    }

    private Object parseActionType(final Object rawData) {
        Object actionType = XJsonUtils.getNodeByPathOrNullEless(
            rawData,
            POTENTIAL_ACTION,
            TYPE);
        if (actionType == null) {
            actionType =
                XJsonUtils.getNodeByPathOrNullEless(
                    rawData,
                    ACTION,
                    TYPE);
        }
        if (actionType == null) {
            actionType =
                XJsonUtils.getNodeByPathOrNullEless(
                    rawData,
                    ACTION,
                    TYPE1);
        }
        return actionType;
    }

    private boolean isUrlHtmlEscaped(final Object url) {
        // dropbox specific method
        if (url instanceof String) {
            return ((String) url).startsWith(
                "https&#58;&#47;&#47;www.dropbox.com");
        }
        return false;
    }
}
