package ru.yandex.iex.proxy;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.geocoder.GeocoderClient;
import ru.yandex.geocoder.GeocoderRequest;
import ru.yandex.geocoder.GeocoderResult;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;

public class AddrHandler extends AbstractEntityHandler<AddrContext> {
    AddrHandler(final IexProxy iexProxy) {
        super(iexProxy, "addr");
    }

    @Override
    protected AddrContext createContext(
        final IexProxy iexProxy,
        final ProxySession session,
        final Map<?, ?> json)
        throws HttpException, JsonUnexpectedTokenException
    {
        return new AddrContext(iexProxy, session, json);
    }

    @Override
    @SuppressWarnings("FutureReturnValueIgnored")
    protected void handle(final AddrContext context)
        throws JsonUnexpectedTokenException, BadRequestException
    {
        if (context.getCountOfValidAddress() == 0) {
            context.response();
            return;
        }
        final ArrayList<AddrEntry> addrs = context.getArrayOfAddress();
        final MultiFutureCallback<ActionWithResult> multiCallback =
            new MultiFutureCallback<>(new FinishMulticallbacks(context));

        final List<ResultCallback> subCallbacks =
            new ArrayList<>(context.getCountOfValidAddress());

        for (final AddrEntry x : addrs) {
            if (x.isEntryPerfect()) {
                final ResultCallback actionCallback =
                    new ResultCallback(x, multiCallback.newCallback());
                subCallbacks.add(actionCallback);
            }
        }
        multiCallback.done();
        GeocoderClient client =
            context.iexProxy().geoSearchClient().adjust(
                context.session().context());
        for (ResultCallback actionCallback : subCallbacks) {
            final String text =
                actionCallback.getAddrEntry().getAddressInGeocoderFormat();
            try {
                GeocoderRequest request = new GeocoderRequest(
                    context.iexProxy().config().geoSearchConfig());
                request.addHeader(
                    YandexHeaders.X_YA_SERVICE_TICKET,
                    context.iexProxy().geoTvm2Ticket());
                request.setText(text);
                client.execute(
                    request,
                    context.session().listener()
                        .createContextGeneratorFor(client),
                    actionCallback
                );
            } catch (Exception e) {
                context.session().logger().log(
                    Level.SEVERE,
                    "GeoCoder request encoding failed",
                    e);
                context.failed(e);
                return;
            }
        }
    }

    private static class ActionWithResult {
        private final Object result;
        private final AddrEntry addrEntry;

        ActionWithResult(final Object result, final AddrEntry entry) {
            this.result = result;
            this.addrEntry = entry; // my action
        }

        public Object result() {
            return result;
        }

        public AddrEntry getAddrEntry() {
            return addrEntry;
        }
    }

    private static class ResultCallback
        implements FutureCallback<GeocoderResult>
    {
        private final FutureCallback<ActionWithResult> callback;
        private final AddrEntry entry;

        ResultCallback(
            final AddrEntry entry,
            final FutureCallback<ActionWithResult> callback)
        {
            this.entry = entry;
            this.callback = callback;
        }

        public AddrEntry getAddrEntry() {
            return entry;
        }

        private Object exceptionToJson(final Exception e) {
            final StringBuilderWriter sbw = new StringBuilderWriter();
            e.printStackTrace(sbw);
            return Collections.singletonMap("error", sbw.toString());
        }

        @Override
        public void cancelled() {
            callback.cancelled();
        }

        @Override
        public void failed(final Exception e) {
            if (e instanceof ServerException) {
                final ServerException se = (ServerException) e;
                final int httpCode = se.statusCode();
                final boolean nonFatal =
                    HttpStatusPredicates.NON_FATAL.test(httpCode);
                if (nonFatal) {
                    callback.completed(
                        new ActionWithResult(exceptionToJson(e), entry));
                } else {
                    callback.failed(e);
                }
            } else {
                callback.failed(e);
            }
        }

        @Override
        public void completed(final GeocoderResult result) {
            callback.completed(new ActionWithResult((Object) result, entry));
        }
    }

    private static class FinishMulticallbacks
        implements FutureCallback<List<ActionWithResult>>
    {
        private final AddrContext context;

        FinishMulticallbacks(final AddrContext context) {
            this.context = context;
        }

        @Override
        public void completed(final List<ActionWithResult> results) {
            for (final ActionWithResult x : results) {
                if (x.result() instanceof GeocoderResult) {
                    GeocoderResult result = (GeocoderResult) x.result();
                    if (result.size() != 1) {
                        // ignore if more/less than a one result was found
                        x.getAddrEntry().pushToJson("ambiguity", "true");
                        continue;
                    }
                    int precision = 0;
                    double lat = (result.lowerLatitude(precision)
                        + result.upperLatitude(precision)) / 2.0;
                    double lon = (result.lowerLongitude(precision)
                        + result.upperLongitude(precision)) / 2.0;
                    x.getAddrEntry().pushToJson("geocoder_lat", lat);
                    x.getAddrEntry().pushToJson("geocoder_lon", lon);
                }
            }
            context.response();
        }

        @Override
        public void cancelled() {
            context.response();
        }

        @Override
        public void failed(final Exception e) {
            context.failed(e);
        }
    }
}
