package ru.yandex.iex.proxy;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonBoolean;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.mail.search.MailSearchParams;

public class BigImageHandler implements HttpAsyncRequestHandler<JsonObject> {
    public static final String WIDGET_INFO_HEADER = "x-mail-widget-info";
    public static final String WIDGET_CONTROLS_HEADER = "x-mail-widget-controls";
    public static final String DOUBLE = "double";

    private final IexProxy proxy;

    public BigImageHandler(final IexProxy proxy) {
        this.proxy = proxy;
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException
    {
        if (!(request instanceof HttpEntityEnclosingRequest)) {
            throw new BadRequestException("Payload expected");
        }
        return new JsonAsyncTypesafeDomConsumer(
            ((HttpEntityEnclosingRequest) request).getEntity(),
            StringCollectorsFactory.INSTANCE,
            BasicContainerFactory.INSTANCE);
    }

    @Override
    @SuppressWarnings("FutureReturnValueIgnored")
    public void handle(
        final JsonObject payload,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        ProxySession session = new BasicProxySession(proxy, exchange, context);
        String stid = session.params().getString(MailSearchParams.STID);
        BasicAsyncRequestProducerGenerator requestGenerator =
            new BasicAsyncRequestProducerGenerator(
                "/headers?json-type=dollar&stid=" + stid);
        requestGenerator.addHeader(
            YandexHeaders.X_YA_SERVICE_TICKET,
            proxy.tikaiteTvm2Ticket());
        requestGenerator.addHeader(
            YandexHeaders.X_SRW_SERVICE_TICKET,
            proxy.unistorageTvm2Ticket());
        AsyncClient client = proxy.tikaiteClient().adjust(session.context());
        client.execute(
            proxy.tikaiteHost(),
            requestGenerator,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            new TikaiteCallback(session));
    }

    private static class TikaiteCallback implements FutureCallback<JsonObject> {
        private final ProxySession session;
        private final Logger logger;

        private TikaiteCallback(final ProxySession session) {
            this.session = session;
            this.logger = session.logger();
        }

        @Override
        public void completed(final JsonObject result) {
            Map<String, String> widgetProps = parseHeaders(result);
            if (widgetProps.isEmpty()) {
                emptyResponse();
            } else {
                JsonMap response = generateResponse(widgetProps);
                session.response(
                    YandexHttpStatus.SC_OK,
                    JsonType.NORMAL.toString(response));
            }
        }

        @Override
        public void failed(final Exception e) {
            logger.info("Tikaite failed");
            logger.log(Level.WARNING, "Tikaite failed with", e);
            emptyResponse();
        }

        @Override
        public void cancelled() {
            logger.warning("Cancelled tikaite request");
            emptyResponse();
        }

        private void emptyResponse() {
            session.response(
                HttpStatus.SC_OK,
                JsonType.NORMAL.toString(Collections.emptyMap()));
        }

        @SuppressWarnings("StringSplitter")
        private Map<String, String> parseHeaders(final JsonObject result) {
            Map<String, String> widgetProps = new HashMap<>();
            try {
                JsonList headers = result.asMap().getList("headers");
                for (JsonObject header: headers) {
                    Map.Entry<String, JsonObject> entry =
                        header.asMap().entrySet().iterator().next();
                    String key = entry.getKey();
                    if (WIDGET_INFO_HEADER.equalsIgnoreCase(key)
                        || WIDGET_CONTROLS_HEADER.equalsIgnoreCase(key))
                    {
                        String value = entry.getValue().asStringOrNull();
                        if (value != null && value.startsWith("bigimage;")) {
                            for (String prop: value.split(";")) {
                                String[] splited = prop.trim().split("=", 2);
                                if (splited.length == 2) {
                                    widgetProps.put(splited[0], splited[1]);
                                }
                            }
                        }
                    }
                }
            } catch (JsonException e) {
                logger.warning(
                    "Json parsing exception for mail headers" + e.getMessage());
            }
            return widgetProps;
        }

        private JsonMap generateResponse(final Map<String, String> widgetProps) {
            JsonMap result = new JsonMap(BasicContainerFactory.INSTANCE);
            for (String key: widgetProps.keySet()) {
                result.put(key, new JsonString(widgetProps.get(key)));
            }
            if (widgetProps.containsKey(DOUBLE)) {
                boolean field = Boolean.parseBoolean(widgetProps.get(DOUBLE));
                result.put(DOUBLE, JsonBoolean.valueOf(field));
            }
            return result;
        }
    }
}
