package ru.yandex.iex.proxy;

import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncPostURIRequestProducerSupplier;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncDomConsumerFactory;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.parser.uri.QueryConstructor;

public final class Blackbox2CokemulatorCallback {
    public static final String EML = "eml";
    public static final String STID = "stid";
    public static final String UID = "uid";
    public static final String PDF = "pdf";
    public static final String BOUNCE = "bounce";
    public static final String LIST_UNSUBSCRIBE = "list_unsubscribe";

    private static final String UNEXPECTED_ERROR =
        "Unexpected unrecoverable fatal error occured trying to"
            + " execute post action for message: ";

    private Blackbox2CokemulatorCallback() {
    }

    private static class MessageBodySubCallback extends StorageDataSubRequest {
        MessageBodySubCallback(
            final IndexationContext<Solution> context,
            final FutureCallback<StorageRequestResult> callback)
        {
            super(context, callback);
        }

        @Override
        public String wrapKey() {
            return "";
        }

        @Override
        public String hid() {
            return "";
        }

        @Override
        public boolean partRequest() {
            return false;
        }

        @Override
        public void registerExecutionTime(final long time) {
            final IexProxy iexProxy = context.abstractContext().iexProxy();
            iexProxy.mainExtractTime(time);
        }

        @Override
        @SuppressWarnings("FutureReturnValueIgnored")
        public void execute() {
            String entitiesString = context.entitiesString();
            if (entitiesString.isEmpty()) {
                completed(new StorageRequestResult());
            } else {
                final IexProxy iexProxy = context.abstractContext().iexProxy();
                final AsyncClient client =
                    iexProxy.cokemulatorIexlibClient()
                        .adjust(context.abstractContext().session().context());
                client.execute(
                    iexProxy.cokemulatorIexlibHost(),
                    cokemulatorRequest(entitiesString),
                    JsonAsyncDomConsumerFactory.OK,
                    context.abstractContext().session().listener()
                        .createContextGeneratorFor(client),
                    this);
            }
        }
    }

    private static class IcsSubCallback extends StorageDataSubRequest {
        private final String hid;

        IcsSubCallback(
            final IndexationContext<Solution> context,
            final FutureCallback<StorageRequestResult> callback,
            final String hid)
        {
            super(context, callback);
            this.hid = hid;
        }

        @Override
        public String wrapKey() {
            return "";
        }

        @Override
        public String hid() {
            return "";
        }

        @Override
        public boolean partRequest() {
            return false;
        }

        @Override
        public void registerExecutionTime(final long time) {
        }

        @Override
        public void execute() {
            final Map<String, Object> subSolution = new HashMap<>();
            subSolution.put(UID, context.abstractContext().uid());
            subSolution.put(STID, context.stid());
            subSolution.put("hid", hid);
            subSolution.put("lang", "ru");
            final StorageRequestResult fakeIexSolution =
                new StorageRequestResult();
            fakeIexSolution.put("ics", subSolution);
            completed(fakeIexSolution);
        }
    }

    private static class PdfSubCallback extends TikaiteIexProxyRequest {
        private final String hid;

        PdfSubCallback(
            final IndexationContext<Solution> context,
            final FutureCallback<StorageRequestResult> callback,
            final String hid)
        {
            super(context, callback);
            this.hid = hid;
        }

        @Override
        public String wrapKey() {
            return PDF;
        }

        @Override
        public String hid() {
            return hid;
        }

        @Override
        public void registerExecutionTime(final long time) {
            final IexProxy iexProxy = context.abstractContext().iexProxy();
            iexProxy.attachExtractTime(time);
        }

        @Override
        public void execute() {
            // TODO: move to config
            final String entityToExtract = "ticket";
            if (context.entities().containsKey(entityToExtract)) {
                this.execute(entityToExtract);
            } else {
                this.completed(new HashMap<>());
            }
        }
    }

    private static class PkpassSubCallback extends TikaiteGetBodyRequest {
        private final String hid;

        PkpassSubCallback(
            final IndexationContext<Solution> context,
            final FutureCallback<StorageRequestResult> callback,
            final String hid)
        {
            super(context, callback);
            this.hid = hid;
        }

        @Override
        public String wrapKey() {
            return "pkpass";
        }

        @Override
        public String hid() {
            return hid;
        }

        @Override
        public boolean partRequest() {
            return true;
        }

        @Override
        public void registerExecutionTime(final long time) {
            final IexProxy iexProxy = context.abstractContext().iexProxy();
            iexProxy.attachExtractTime(time);
        }
    }

    private static class DmarcSubCallback extends TikaiteGetBodyRequest {
        private final String hid;

        DmarcSubCallback(
            final IndexationContext<Solution> context,
            final FutureCallback<StorageRequestResult> callback,
            final String hid)
        {
            super(context, callback);
            this.hid = hid;
        }

        @Override
        public void adjustUri(final StringBuilder uri) {
            uri.append("&parse-dmarc");
        }

        @Override
        public String wrapKey() {
            return "dmarc";
        }

        @Override
        public String hid() {
            return hid;
        }

        @Override
        public boolean partRequest() {
            return true;
        }

        @Override
        public void registerExecutionTime(final long time) {
            final IexProxy iexProxy = context.abstractContext().iexProxy();
            iexProxy.attachExtractTime(time);
        }
    }

    private static class BounceSubCallback extends TikaiteGetBodyRequest {
        private final String hid;

        BounceSubCallback(
            final IndexationContext<Solution> context,
            final FutureCallback<StorageRequestResult> callback,
            final String hid)
        {
            super(context, callback);
            this.hid = hid;
        }

        @Override
        public String wrapKey() {
            return BOUNCE + "_attach";
        }

        @Override
        public String hid() {
            return hid;
        }

        @Override
        public boolean partRequest() {
            return true;
        }

        @Override
        public void registerExecutionTime(final long time) {
            final IexProxy iexProxy = context.abstractContext().iexProxy();
            iexProxy.attachExtractTime(time);
        }
    }

    private static class ListUnsubscribeSubCallback
        extends TikaiteGetBodyRequest
    {
        private final String hid;

        ListUnsubscribeSubCallback(
            final IndexationContext<Solution> context,
            final FutureCallback<StorageRequestResult> callback,
            final String hid)
        {
            super(context, callback);
            this.hid = hid;
        }

        @Override
        public String wrapKey() {
            return ActionEntityHandler.LIST_UNSUBSCRIBE;
        }

        @Override
        public String hid() {
            return hid;
        }

        @Override
        public boolean partRequest() {
            return true;
        }

        @Override
        public void registerExecutionTime(final long time) {
            final IexProxy iexProxy = context.abstractContext().iexProxy();
            iexProxy.attachExtractTime(time);
        }
    }

    private static class HeadersMergingCallback
        extends AbstractFilterFutureCallback<
            Map.Entry<
                Map<String, String>, List<StorageRequestResult>>,
            List<StorageRequestResult>>
    {
        private final IndexationContext<Solution> context;

        HeadersMergingCallback(
            final IndexationContext<Solution> context,
            final FutureCallback<List<StorageRequestResult>> callback)
        {
            super(callback);
            this.context = context;
        }

        @Override
        public void completed(
            final Map.Entry<Map<String, String>, List<StorageRequestResult>>
                value)
        {
            context.headers(value.getKey());
            callback.completed(value.getValue());
        }
    }

    private static class MergeSolutionsCallback
        implements FutureCallback<List<StorageRequestResult>>
    {
        private final IndexationContext<Solution> context;
        private final AbstractCallback<Solution> callback;
        private final long startTime = System.currentTimeMillis();

        MergeSolutionsCallback(final AbstractCallback<Solution> callback) {
            this.callback = callback;
            this.context = callback.context();
        }

        @Override
        public void completed(final List<StorageRequestResult> results) {
            callback.context().abstractContext().iexProxy()
                .cokemulatorCompleted(System.currentTimeMillis() - startTime);
            Map<String, Object> merged = new HashMap<>();
            Map<String, Integer> nameConflictTable = new HashMap<>();
            for (final StorageRequestResult x : results) {
                XJsonUtils.putAllAndResolveNameConflicts(
                    merged,
                    x,
                    nameConflictTable);
            }
            if (!context.postActions().isEmpty()) {
                context.abstractContext().session().logger().info(
                    "message postActions: mid: " + context.mid()
                        + ", postActions: " + context.postActions());
                new CokemulatorToSolutionAdapter(
                        new PostActionsExecutingCallback(callback))
                    .completed(merged);
            } else {
                new CokemulatorToSolutionAdapter(callback).completed(merged);
            }
        }

        @Override
        public void cancelled() {
            context.callback().cancelled();
        }

        @Override
        public void failed(final Exception e) {
            context.callback().failed(e);
        }
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    public static void cokemulatorExecuter(
        final AbstractCallback<Solution> callback)
    {
        try {
            final DoubleFutureCallback<
                Map<String, String>,
                List<StorageRequestResult>>
                    headersAndDataCallback =
                        new DoubleFutureCallback<>(
                            new HeadersMergingCallback(
                                callback.context(),
                                new MergeSolutionsCallback(callback)));

            FutureCallback<Map<String, String>> headersCallback =
                headersAndDataCallback.first();
            if (callback.context().headersWanted().isEmpty()) {
                headersCallback.completed(Collections.emptyMap());
            } else {
                IndexationContext<Solution> context = callback.context();
                ProxySession session = context.abstractContext().session();
                session.logger().info(
                    "headers wanted for mid<" + context.mid()
                    + ">: " + context.headersWanted());
                IexProxy iexProxy = context.abstractContext().iexProxy();
                AsyncClient client =
                    iexProxy.tikaiteClient().adjust(session.context());
                BasicAsyncRequestProducerGenerator request =
                    new BasicAsyncRequestProducerGenerator(
                        "/headers?json-type=dollar&stid=" + context.stid());
                request.addHeader(
                    YandexHeaders.X_YA_SERVICE_TICKET,
                    iexProxy.tikaiteTvm2Ticket());
                request.addHeader(
                    YandexHeaders.X_SRW_SERVICE_TICKET,
                    iexProxy.unistorageTvm2Ticket());
                client.execute(
                    iexProxy.tikaiteHost(),
                    request,
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    session.listener().createContextGeneratorFor(client),
                    new TikaiteHeadersConvertingCallback(
                        headersCallback,
                        context.headersWanted()));
            }

            Map<String, EntityOptions> entities = callback.context().entities();
            final MultiFutureCallback<StorageRequestResult> multiCallback =
                new MultiFutureCallback<>(headersAndDataCallback.second());
            final List<StorageDataSubRequest> subCallbacks =
                new ArrayList<>(1);
            subCallbacks.add(
                new MessageBodySubCallback(
                    callback.context(),
                    new ErrorSuppressingFutureCallback<>(
                        multiCallback.newCallback(),
                        x -> RequestErrorType.HTTP,
                        new StorageRequestResult()
                    )
                )
            );
            if (entities.containsKey(EML)) {
                subCallbacks.add(
                    new MulcaGateDataRequest(
                        callback.context(),
                        new ErrorSuppressingFutureCallback<>(
                            multiCallback.newCallback(),
                            x -> RequestErrorType.HTTP,
                            new StorageRequestResult())));
            }
            //TODO: check an attach before creating an AbsSubCallback instance
            XMessageToLog.info(callback.context().abstractContext(),
                "found ics: "
                + callback.context().iscHids());
            for (final String iscHid : callback.context().iscHids()) {
                subCallbacks.add(
                    new IcsSubCallback(
                        callback.context(),
                        new ErrorSuppressingFutureCallback<>(
                            multiCallback.newCallback(),
                            x -> RequestErrorType.HTTP,
                            new StorageRequestResult()
                        ),
                        iscHid));
            }
            XMessageToLog.info(callback.context().abstractContext(),
                "found pdf: "
                    + callback.context().pdfHids());
            for (final String pdfHid : callback.context().pdfHids()) {
                subCallbacks.add(
                    new PdfSubCallback(
                        callback.context(),
                        new ErrorSuppressingFutureCallback<>(
                            multiCallback.newCallback(),
                            x -> RequestErrorType.HTTP,
                            new StorageRequestResult()
                        ),
                        pdfHid));
            }
            XMessageToLog.info(callback.context().abstractContext(),
                "found pkpass: "
                    + callback.context().pkpassHids());
            for (final String pkpassHid : callback.context().pkpassHids()) {
                subCallbacks.add(
                    new PkpassSubCallback(
                        callback.context(),
                        new ErrorSuppressingFutureCallback<>(
                            multiCallback.newCallback(),
                            x -> RequestErrorType.HTTP,
                            new StorageRequestResult()
                        ),
                        pkpassHid));
            }
            for (String dmarcHid: callback.context().dmarcHids()) {
                subCallbacks.add(
                    new DmarcSubCallback(
                        callback.context(),
                        new ErrorSuppressingFutureCallback<>(
                            multiCallback.newCallback(),
                            x -> RequestErrorType.HTTP,
                            new StorageRequestResult()),
                        dmarcHid));
            }
            final String oneHid = "1";
            if (entities.containsKey(BOUNCE)) {
                XMessageToLog.info(callback.context().abstractContext(),
                    "bounce type 8 found");
                subCallbacks.add(
                    new BounceSubCallback(
                        callback.context(),
                        multiCallback.newCallback(),
                        oneHid));
            }
            if (entities.containsKey(LIST_UNSUBSCRIBE)) {
                XMessageToLog.info(
                    callback.context().abstractContext(),
                    "found mail with news and firstmail's "
                    + "type for the post process action");
                subCallbacks.add(
                    new ListUnsubscribeSubCallback(
                        callback.context(),
                        new ErrorSuppressingFutureCallback<>(
                            multiCallback.newCallback(),
                            x -> RequestErrorType.HTTP,
                            new StorageRequestResult()
                        ),
                        oneHid));
            }
            multiCallback.done();
            for (final StorageDataSubRequest sdc : subCallbacks) {
                sdc.execute();
            }
        } catch (NullPointerException e) {
            callback.context().abstractContext().session().logger()
                .warning("Some ptr is empty.");
            callback.failed(e);
        } catch (Exception e) {
            callback.context().abstractContext().session().logger()
                .log(
                    Level.SEVERE,
                    "Unhandled error",
                    e);
            callback.failed(e);
        }
    }

    private static class PostActionsExecutingCallback
        extends AbstractCallback<Solution>
    {
        private final AbstractCallback<Solution> callback;
        private final Logger logger;

        PostActionsExecutingCallback(
            final AbstractCallback<Solution> callback)
        {
            super(callback.context());
            this.callback = callback;
            this.logger = context().abstractContext().session().logger();
        }

        public AbstractCallback<Solution> changeCallback() {
            return callback;
        }

        //CSOFF: ReturnCount
        @Override
        @SuppressWarnings("FutureReturnValueIgnored")
        public void completed(final Solution solution) {
            final Set<PostProcessAction> actions = context().postActions();
            Set<String> actionsStrings = new HashSet<>();
            try {
                for (PostProcessAction act: actions) {
                    actionsStrings.add(act.url().substring(act.url().lastIndexOf('/') + 1));
                }
            } catch (Exception e) {
                final StringBuilderWriter sbw = new StringBuilderWriter();
                e.printStackTrace(sbw);
                logger.info("Exception while constructions postprocess names: " + sbw);
            }
            logger.info("Postprocess actions for message: " + context().mid()
                + ": " + String.join(", ", actionsStrings));
            if (actions.isEmpty()) {
                callback.completed(solution);
                return;
            }

            if (solution.isEmpty() && !actionsStrings.contains("corovaneer")) {
                logger.severe("Cokemulator returned empty solution "
                    + "for message: " + context().mid()
                    + ", can't proceed with post actions");
                callback.completed(solution);
                return;
            }

            final AsyncClient client =
                callback.context().abstractContext()
                    .iexProxy().postProcessClient()
                        .adjust(context().abstractContext().
                                session().context());

            final String postEntity =
                JsonType.NORMAL.toString(solution.cokemulatorSolutions());
            final QueryConstructor query = new QueryConstructor("");
            try {
                query.append("subject", context.subject());
                query.append("email", context.email());
                query.append("user_email", context.userEmail());
                query.append(
                    "received_date",
                    String.valueOf(context.receivedDate()));
                query.append(UID, context.abstractContext().uid());
                if (context.abstractContext().suid() != null) {
                    query.append("suid", context.abstractContext().suid());
                }
                if (context.abstractContext().mdb() != null) {
                    query.append("mdb", context.abstractContext().mdb());
                }
                if (context.abstractContext().pgShard() != null) {
                    query.append(
                            "pgshard",
                            context.abstractContext().pgShard());
                }
                query.append("mid", context.mid());
                query.append(STID, context.stid());
                if (context.firstline() != null) {
                    query.append("firstline", context.firstline());
                }
                if (context.meta() != null) {
                    String folderName = context.meta().get("folder_name");
                    if (folderName != null) {
                        query.append("folder_name", folderName);
                    }
                    query.append("folder_type", context.folderType());
                }
                if (context.types() != null && !context.types().isEmpty()) {
                    // 2 for type + 1 for comma
                    StringBuilder sb =
                        new StringBuilder((2 + 1) * context.types().size());
                    for (Integer type: context.types()) {
                        sb.append(type);
                        sb.append(',');
                    }

                    sb.setLength(sb.length() - 1);
                    query.append("types", sb.toString());
                }
            } catch (BadRequestException | NullPointerException e) {
                logger.log(
                    Level.SEVERE,
                    "Can't constuct request to post actions. Cannot continue.",
                    e);
                callback.completed(null);
                return;
            }
            final String cgiParams = query.toString();

            final MultiFutureCallback<PostActionWithResult> multiCallback =
                new MultiFutureCallback<>(
                    new PostActionsJsonMeringAdapter(solution, callback));

            final List<PostActionResultCallback> subCallbacks =
                new ArrayList<>(actions.size());
            for (PostProcessAction action : actions) {
                try {
                    String url = action.url();
                    if (url.indexOf('?') == -1) {
                        url += '?' + cgiParams;
                    } else {
                        url += '&' + cgiParams;
                    }
                    final AsyncPostURIRequestProducerSupplier post =
                        new AsyncPostURIRequestProducerSupplier(
                            url,
                            postEntity,
                            ContentType.APPLICATION_JSON);
                    final PostActionResultCallback actionCallback =
                        new PostActionResultCallback(
                            post,
                            action,
                            multiCallback.newCallback(),
                            callback.context().abstractContext()
                                .iexProxy());
                    subCallbacks.add(actionCallback);
                } catch (URISyntaxException e) {
                    logger.log(
                        Level.SEVERE,
                        "Error creating postprocessing request for message: "
                            + context().mid(),
                        e);
                    callback.failed(e);
                    return;
                }
            }
            multiCallback.done();
            for (PostActionResultCallback actionCallback : subCallbacks) {
                final boolean voidAction = actionCallback.action().isVoid();
                try {
                    if (voidAction) {
                        client.execute(
                            actionCallback.post(),
                            EmptyAsyncConsumerFactory.NON_FATAL,
                            callback.context().abstractContext().session()
                                .listener().createContextGeneratorFor(client),
                            actionCallback);
                    } else {
                        client.execute(
                            actionCallback.post(),
                            JsonAsyncDomConsumerFactory.OK,
                            callback.context().abstractContext().session()
                                .listener().createContextGeneratorFor(client),
                            actionCallback);
                    }
                } catch (Exception e) {
                    logger.log(
                        Level.INFO,
                        UNEXPECTED_ERROR
                            + context().mid() + ", action: "
                            + actionCallback.action(),
                        e);
                    callback.failed(e);
                    return;
                }
            }
        }
        //CSON: ReturnCount
    }

    private static class PostActionResultCallback
        extends AbstractFilterFutureCallback<Object, PostActionWithResult>
    {
        private final AsyncPostURIRequestProducerSupplier post;
        private final PostProcessAction action;
        private final long startTime;
        private final IexProxy iexProxy;

        //CSOFF: ParameterNumber
        PostActionResultCallback(
            final AsyncPostURIRequestProducerSupplier post,
            final PostProcessAction action,
            final FutureCallback<PostActionWithResult> callback,
            final IexProxy iexProxy)
        {
            super(callback);
            this.post = post;
            this.action = action;
            this.iexProxy = iexProxy;
            this.startTime = System.currentTimeMillis();
        }
        //CSON: ParameterNumber

        public AsyncPostURIRequestProducerSupplier post() {
            return post;
        }

        public PostProcessAction action() {
            return action;
        }

        private Object exceptionToJson(final Exception e) {
            final StringBuilderWriter sbw = new StringBuilderWriter();
            e.printStackTrace(sbw);
            return Collections.singletonMap("error", sbw.toString());
        }

        @Override
        public void failed(final Exception e) {
            final long time = System.currentTimeMillis() - startTime;
            iexProxy.postActionsTime(time);
            if (e instanceof ServerException) {
                final ServerException se = (ServerException) e;
                final int httpCode = se.statusCode();
                final boolean nonFatal =
                    HttpStatusPredicates.NON_FATAL.test(httpCode);
                if (nonFatal && !action.name().equals("refund_fbl")) {
                    callback.completed(
                        new PostActionWithResult(
                            action,
                            exceptionToJson(e)));
                } else {
                    super.failed(e);
                }
            } else {
                super.failed(e);
            }
        }

        @Override
        public void completed(final Object result) {
            final long time = System.currentTimeMillis() - startTime;
            iexProxy.postActionsTime(time);
            if (action.isVoid()) {
                callback.completed(PostActionWithResult.EMPTY_RESULT);
            } else {
                callback.completed(new PostActionWithResult(action, result));
            }
        }
    }

    private static class PostActionsJsonMeringAdapter
        extends AbstractFilterFutureCallback<
            List<PostActionWithResult>,
            Solution>
    {
        private final Solution solution;

        PostActionsJsonMeringAdapter(
            final Solution solution,
            final FutureCallback<Solution> callback)
        {
            super(callback);
            this.solution = solution;
        }

        @Override
        public void completed(final List<PostActionWithResult> results) {
            for (final PostActionWithResult awr : results) {
                solution.context().abstractContext().session().logger().
                    info("PostActionsJsonMeringAdapter: got "
                            + awr.action() + " result");
                if (awr != PostActionWithResult.EMPTY_RESULT) {
                    solution.addSolution(awr);
                }
            }
            callback.completed(solution);
        }
    }

    private static class CokemulatorToSolutionAdapter
        extends AbstractFilterFutureCallback<Object, Solution>
    {
        private IndexationContext<Solution> context;

        CokemulatorToSolutionAdapter(
            final AbstractCallback<Solution> callback)
        {
            super(callback);
            context = callback.context();
        }

        private Solution createSolution(final Object result) {
            final Map<String, Object> solutions;
            if (!(result instanceof Map)) {
                context.abstractContext().session.logger().
                    warning(
                        "Can't rename entities: IEX solution is not a map: "
                            + result);
                solutions = Collections.emptyMap();
            } else {
                final Map<?, ?> root = (Map<?, ?>) result;
                solutions = new HashMap<>();
                for (Map.Entry<?, ?> entry : root.entrySet()) {
                    final String entity;
                    try {
                        entity = ValueUtils.asString(entry.getKey());
                    } catch (JsonUnexpectedTokenException e) {
                        context.abstractContext().session.logger().warning(
                            "can't rename unhandler IEX solution json object "
                            + "key: " + entry.getKey() + ", expected: string");
                        continue;
                    }
                    solutions.put(entity, (Object) entry.getValue());
                }
            }
            return new Solution(context, solutions);
        }

        @Override
        public void completed(final Object results) {
            callback.completed(createSolution(results));
        }
    }
}

