package ru.yandex.iex.proxy;

import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncPostURIRequestProducerSupplier;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;

public class BounceHandler extends AbstractEntityHandler<BounceContext> {
    private static final Long ONE = 1L;
    private final URI freemailUrl;
    private final TimeFrameQueue<Long> totalBounces;
    private final TimeFrameQueue<Long> invalidBounces;
    private final TimeFrameQueue<Long> spamBounces;
    private final TimeFrameQueue<Long> regularBounces;
    private final TimeFrameQueue<Long> smtpBounces;
    private final TimeFrameQueue<Long> forwardBounces;
    private final TimeFrameQueue<Long> unknownBounces;

    BounceHandler(final IexProxy iexProxy) {
        super(iexProxy, "bounce");
        freemailUrl = iexProxy.config().freemailConfig().uri();
        totalBounces = new TimeFrameQueue<>(iexProxy.config().metricsTimeFrame());
        invalidBounces = new TimeFrameQueue<>(iexProxy.config().metricsTimeFrame());
        spamBounces = new TimeFrameQueue<>(iexProxy.config().metricsTimeFrame());
        regularBounces = new TimeFrameQueue<>(iexProxy.config().metricsTimeFrame());
        smtpBounces = new TimeFrameQueue<>(iexProxy.config().metricsTimeFrame());
        forwardBounces = new TimeFrameQueue<>(iexProxy.config().metricsTimeFrame());
        unknownBounces = new TimeFrameQueue<>(iexProxy.config().metricsTimeFrame());
        iexProxy.registerStater(
            new PassiveStaterAdapter<>(
                totalBounces,
                new NamedStatsAggregatorFactory<>("bounces-total-requests_ammm", CountAggregatorFactory.INSTANCE)));
        iexProxy.registerStater(
            new PassiveStaterAdapter<>(
                invalidBounces,
                new NamedStatsAggregatorFactory<>("bounces-invalid-requests_ammm", CountAggregatorFactory.INSTANCE)));
        iexProxy.registerStater(
            new PassiveStaterAdapter<>(
                spamBounces,
                new NamedStatsAggregatorFactory<>("bounces-spam-requests_ammm", CountAggregatorFactory.INSTANCE)));
        iexProxy.registerStater(
            new PassiveStaterAdapter<>(
                forwardBounces,
                new NamedStatsAggregatorFactory<>("bounces-forward-requests_ammm", CountAggregatorFactory.INSTANCE)));
        iexProxy.registerStater(
            new PassiveStaterAdapter<>(
                regularBounces,
                new NamedStatsAggregatorFactory<>("bounces-regular-requests_ammm", CountAggregatorFactory.INSTANCE)));
        iexProxy.registerStater(
            new PassiveStaterAdapter<>(
                smtpBounces,
                new NamedStatsAggregatorFactory<>("bounces-smtp-requests_ammm", CountAggregatorFactory.INSTANCE)));
        iexProxy.registerStater(
            new PassiveStaterAdapter<>(
                unknownBounces,
                new NamedStatsAggregatorFactory<>("bounces-unknown-requests_ammm", CountAggregatorFactory.INSTANCE)));
    }

    @Override
    protected BounceContext createContext(final IexProxy iexProxy, final ProxySession session, final Map<?, ?> json)
        throws HttpException, JsonUnexpectedTokenException
    {
        return new BounceContext(iexProxy, session, json);
    }

    public void totalBounces() {
        totalBounces.accept(ONE);
    }

    public void invalidBounces() {
        invalidBounces.accept(ONE);
    }

    public void spamBounces() {
        spamBounces.accept(ONE);
    }

    public void forwardBounces() {
        forwardBounces.accept(ONE);
    }

    public void regularBounces() {
        regularBounces.accept(ONE);
    }

    public void smtpBounces() {
        smtpBounces.accept(ONE);
    }

    public void unknownBounces() {
        unknownBounces.accept(ONE);
    }

    @Override
    @SuppressWarnings("FutureReturnValueIgnored")
    protected void handle(final BounceContext context)
        throws JsonUnexpectedTokenException, BadRequestException
    {
        totalBounces();
        if (context.getMessageId() instanceof List) {
            boolean spam = false;
            boolean unknown = false;
            for (BounceContext.BounceTraits type: context.bounceTraits()) {
                switch (type) {
                    case SPAM:
                        spamBounces();
                        spam = true;
                        break;
                    case REGULAR:
                        regularBounces();
                        break;
                    case SMTP:
                        smtpBounces();
                        break;
                    case FORWARD:
                        forwardBounces();
                        break;
                    case UNKNOWN:
                        unknownBounces();
                        unknown = true;
                        break;
                    case INVALID:
                        invalidBounces();
                }
            }
            List<?> multiRequest = (List<?>) context.getMessageId();
            final MultiFutureCallback<String> multiCallback =
                new MultiFutureCallback<>(new FinishBounceMulticallback(context));
            final List<BounceResultCallback> subCallbacks = new ArrayList<>();
            for (final Object request: multiRequest) {
                subCallbacks.add(new BounceResultCallback(multiCallback.newCallback(), request));
            }
            AsyncClient msalClient = context.msalClient();
            HttpHost msalHost = context.msalHost();
            msalClient = msalClient.adjust(context.session().context());
            for (BounceResultCallback bounceCallback: subCallbacks) {
                QueryConstructor query = new QueryConstructor("/get-mid-by-message-id?");
                query.append("uid", context.uid());
                query.append("message-id", bounceCallback.getMessageId().toString());
                String requestString = query.toString();
                context.session().logger().info("BounceHandler: msal host = " + msalHost);
                context.session().logger().info("BounceHandler: request to msal " + requestString);
                msalClient.execute(
                    msalHost,
                    new BasicAsyncRequestProducerGenerator(requestString),
                    AsyncStringConsumerFactory.OK,
                    context.session().listener().createContextGeneratorFor(msalClient),
                    bounceCallback);
            }
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = JsonType.NORMAL.create(sbw)) {
                writer.startArray();
                if (spam || unknown) {
                    writer.startObject();   // bounce_totals & bounce_history
                    {
                        writer.key("type");
                        writer.value("Update");

                        writer.key("scheme");
                        writer.startArray();
                        {
                            writer.value("bounce_totals");
                            writer.value("bounce_history");
                        }
                        writer.endArray();

                        writer.key("fields");
                        writer.startArray();
                        {
                            writer.startObject();
                            {
                                writer.key("uuid");
                                writer.value(context.uid());
                                if (spam) {
                                    writer.key("spam");
                                    writer.value(1);
                                    writer.key("last_type");
                                    writer.value(2);
                                }
                                if (unknown) {
                                    writer.key("unknown");
                                    writer.value(1);
                                    if (!spam) {
                                        writer.key("last_type");
                                        writer.value(1);
                                    }
                                }
                            }
                            writer.endObject();
                        }
                        writer.endArray();
                    }
                    writer.endObject();
                }
                String senderDomain = context.senderDomain();
                if (!senderDomain.isEmpty()) {
                    writer.startObject();   // domain
                    {
                        writer.key("type");
                        writer.value("Update");

                        writer.key("scheme");
                        writer.startArray();
                        {
                            writer.value("domain");
                        }
                        writer.endArray();

                        writer.key("fields");
                        writer.startArray();
                        {
                            writer.startObject();
                            {
                                writer.key("domain");
                                writer.value(senderDomain);
                                writer.key("count");
                                writer.value(1);
                            }
                            writer.endObject();
                        }
                        writer.endArray();
                    }
                    writer.endObject();
                }
                writer.endArray();
            } catch (IOException e) {
                /*impossible*/
            }
            AsyncClient freemailClient = context.iexProxy().freemailClient().adjust(context.session().context());
            AsyncPostURIRequestProducerSupplier post =
                new AsyncPostURIRequestProducerSupplier(freemailUrl, sbw.toString(), ContentType.APPLICATION_JSON);
            freemailClient.execute(
                post,
                EmptyAsyncConsumerFactory.OK,
                context.session().listener().createContextGeneratorFor(freemailClient),
                new FreemailCallback(context, multiCallback.newCallback()));
            multiCallback.done();
        } else {
            invalidBounces();
            context.response();
        }
    }

    private static class FinishBounceMulticallback extends AbstractProxySessionCallback<List<String>>
    {
        private final BounceContext context;

        FinishBounceMulticallback(final BounceContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final List<String> results) {
            for (String mid: results) {
                if (mid != null && !mid.isEmpty()) {
                    context.setBounceMid(mid);
                    break;
                }
            }
            context.response();
        }
    }

    private static class BounceResultCallback implements FutureCallback<String>
    {
        private final FutureCallback<String> callback;
        private final Object messageId;

        BounceResultCallback(final FutureCallback<String> callback, final Object messageId) {
            this.callback = callback;
            this.messageId = messageId;
        }

        @Override
        public void cancelled() {
            callback.cancelled();
        }

        @Override
        public void failed(final Exception e) {
            RequestErrorType type = RequestErrorType.ERROR_CLASSIFIER.apply(e);
            if (type == RequestErrorType.IO || type == RequestErrorType.HTTP) {
                callback.failed(e);
            } else {
                callback.completed(null);
            }
        }

        @Override
        public void completed(final String result) {
            callback.completed(result);
        }

        public Object getMessageId() {
            return messageId;
        }
    }

    private static class FreemailCallback
        extends AbstractEntityCallback<BounceContext, Void>
    {
        private final FutureCallback<String> callback;

        FreemailCallback(final BounceContext context, final FutureCallback<String> callback) {
            super(context);
            this.callback = callback;
        }

        @Override
        public void completed(final Void response) {
            context.session().logger().info(
                "BounceHandler: Freemail counters successfully stored");
            callback.completed(null);
        }

        @Override
        public void failed(final Exception e) {
            context.session().logger().log(
                Level.WARNING,
                "BounceHandler: Freemail counters store failed",
                e);
            callback.completed(null);
        }
    }
}
