package ru.yandex.iex.proxy;

import java.io.IOException;
import java.io.StringWriter;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.Base64;
import java.util.function.Predicate;
import java.util.logging.Logger;
import java.util.regex.Pattern;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.SecretKeySpec;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.collection.LongPair;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.BasicProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.config.ConfigException;

/**
 * Uses params: mid, uid, folder_type, received_date
 */
public class BugBountyHandler implements HttpAsyncRequestHandler<JsonObject> {
    private final IexProxy iexProxy;
    private final ThreadLocal<Cipher> cipher;
    private final Logger sobbYtLogger;

    private final Predicate<String> possibleToken =
        Pattern.compile("[\\p{Alnum}\\-_]{21}[AQgw]").asMatchPredicate();
    private final Base64.Decoder decoder = Base64.getUrlDecoder();
    private static final byte[] salt = {80, -35, 4, -48};


    public BugBountyHandler(
        IexProxy iexProxy,
        Logger sobbYtLogger)
        throws IOException
    {
        this.iexProxy = iexProxy;
        this.sobbYtLogger = sobbYtLogger;

        ThreadLocal<Cipher> cipherFactory;
        try {
            String secret = iexProxy.config().extraSettingsConfig()
                .getString("bugbounty-secret", "").trim();
            Key aesKey = new SecretKeySpec(decoder.decode(secret), "AES");
            cipherFactory = ThreadLocal.withInitial(
                () -> {
                    try {
                        Cipher cur = Cipher.getInstance("AES/ECB/NoPadding");
                        cur.init(Cipher.DECRYPT_MODE, aesKey);
                        return cur;
                    } catch (
                        NoSuchAlgorithmException
                        | NoSuchPaddingException
                        | InvalidKeyException e) {
                        iexProxy.logger().warning("SOBB: key error: " + e);
                        return null;
                    }
                });
        } catch (
            IllegalArgumentException
            | NullPointerException
            | ConfigException e)
        {
            iexProxy.logger().warning(
                "BugBountyHandler not initialized: missing correct key in"
                + " extrasettings.bugbounty-secret");
            cipherFactory = new ThreadLocal<>();
        }
        this.cipher = cipherFactory;
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
        final HttpRequest request,
        final HttpContext context)
    throws HttpException
    {
        if (!(request instanceof HttpEntityEnclosingRequest)) {
            throw new BadRequestException("Payload expected");
        }
        return new JsonAsyncTypesafeDomConsumer(
            ((HttpEntityEnclosingRequest) request).getEntity(),
            StringCollectorsFactory.INSTANCE,
            BasicContainerFactory.INSTANCE);
    }

    @Override
    public void handle(
        JsonObject payload,
        HttpAsyncExchange exchange,
        HttpContext context)
        throws HttpException, IOException
    {
        ProxySession session =
            new BasicProxySession(iexProxy, exchange, context);
        UpdateStatsCallback finalCallback = new UpdateStatsCallback(
            session,
            new BasicProxySessionCallback(session));

        // Try to find token in headers
        try {
            String eml = payload.get("eml").get("message_body").asString();
            LongPair<String> hunterUid = findTokenInText(eml);
            if (hunterUid != null) {
                finalCallback.completed(hunterUid);
                return;
            }
        } catch (JsonBadCastException e) {
            iexProxy.logger().warning("no eml.message_body in sobb request");
        }
        // There is no token in headers

        new GetTextCallback(session, finalCallback).execute();
    }

    /**
     * Accepts hunterUid.
     * If it is not null, sends update to lucene
     */
    private class UpdateStatsCallback
        extends AbstractFilterFutureCallback<LongPair<String>, Object>
    {
        private final ProxySession session;

        public UpdateStatsCallback(
            ProxySession session,
            FutureCallback<Object> callback)
        {
            super(callback);
            this.session = session;
        }

        @Override
        public void completed(LongPair<String> hunterUid) {
            String targetUid = session.params().getOrNull("uid");
            boolean inbox = "inbox".equals(
                session.params().getString("folder_type", null));
            String receivedTime = session.params().getOrNull("received_date");
            if (hunterUid != null && targetUid != null && receivedTime != null)
            {
                logStats(
                    hunterUid.second(),
                    hunterUid.first(),
                    Long.parseLong(targetUid),
                    session.params().getOrNull("mid"),
                    inbox,
                    Long.parseLong(receivedTime));
                updateStats(hunterUid.first(), targetUid, inbox);
            } else {
                callback.completed(null);
            }

        }

        @SuppressWarnings("FutureReturnValueIgnored")
        void updateStats(Long hunterUid, String targetUid, boolean inbox) {
            String body =
                "{\"prefix\":" + hunterUid
                    + ",\"AddIfNotExists\":true,\"docs\":[{"
                    + "\"url\": \"sobb_record_"
                    + hunterUid + "_" + targetUid + "\"";
            if (inbox) {
                body += ",\"sobb_inbox\":{\"function\":\"inc\"}";
            }
            body += "}]}";

            HttpHost host =
                iexProxy.config().producerAsyncClientConfig().host();
            String service;
            if (BlackboxUserinfo.corp(hunterUid)) {
                service = iexProxy.corpMailSearchQueueName();
            } else {
                service = iexProxy.mailSearchQueueName();
            }
            BasicAsyncRequestProducerGenerator generator =
                new BasicAsyncRequestProducerGenerator(
                    "/update?prefix=" + hunterUid + "&service=" + service,
                    body);
            generator.addHeader(YandexHeaders.SERVICE, service);

            AsyncClient client =
                iexProxy.producerAsyncClient().adjust(session.context());
            client.execute(
                host,
                generator,
                EmptyAsyncConsumerFactory.OK,
                session.listener().createContextGeneratorFor(client),
                callback);
        }
    }

    /**
     * Search for tokens in text, validate them. Returns Hunter's Uid
     */
    private LongPair<String> findTokenInText(String text) {
        for (int i = 0; i + 22 < text.length(); i += 1) {
            if (text.charAt(i) != 'Q') {
                continue;
            }
            String candidate = text.substring(i + 1, i + 23);
            if (!possibleToken.test(candidate)) {
                continue;
            }
            byte[] decoded = null;
            try {
                decoded = cipher.get().doFinal(decoder.decode(candidate));
            } catch (IllegalBlockSizeException | BadPaddingException e) {
                return null;// unreachable, because candidate.len = 22
            }

            if (!Arrays.equals(salt, 0, 4, decoded, 12, 16)) {
                continue;
            }
            return new LongPair<>(
                bytesToLong(decoded), text.substring(i, i + 23));
        }
        return null;
    }

    private static long bytesToLong(final byte[] bytes) {
        long result = 0;
        for (int i = 0; i < Long.BYTES; i++) {
            result <<= Long.BYTES;
            result |= (bytes[i] & 0xFF);
        }
        return result;
    }

    void logStats(
        String guid,
        long hunterUid,
        long targetUid,
        String mid,
        boolean inbox,
        long receivedTime)
    {
        StringWriter logEntry = new StringWriter();
        JsonWriter json = JsonType.NORMAL.create(logEntry);
        try {
            json.startObject();
            json.key("guid");
            json.value(guid);
            json.key("hunter_uid");
            json.value(hunterUid);
            json.key("target_uid");
            json.value(targetUid);
            json.key("mid");
            json.value(mid);
            json.key("inboxed");
            json.value(inbox);
            json.key("unixtime");
            json.value(receivedTime);
            json.endObject();
        } catch (IOException unreachable) {
        }
        sobbYtLogger.severe(logEntry.toString());
    }

    class GetTextCallback
        extends AbstractFilterFutureCallback<String, LongPair<String>>
    {
        private final ProxySession session;

        protected GetTextCallback(
            ProxySession proxySession,
            FutureCallback<LongPair<String>> callback)
        {
            super(callback);
            this.session = proxySession;
        }

        @SuppressWarnings("FutureReturnValueIgnored")
        public void execute() throws BadRequestException {
            AsyncClient client =
                iexProxy.gettextClient().adjust(session.context());
            HttpHost host =
                iexProxy.config().gettextConfig().host();

            String uri = "/get-text?mid="
                + session.params().getString("mid")
                + "&uid=" + session.params().getString("uid");
            client.execute(
                host,
                new BasicAsyncRequestProducerGenerator(uri),
                AsyncStringConsumerFactory.ANY_GOOD,
                session.listener().createContextGeneratorFor(client),
                this);
        }

        @Override
        public void completed(String text) {
            callback.completed(findTokenInText(text));
        }


    }
}
