package ru.yandex.iex.proxy;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import ru.yandex.iex.proxy.eventtickethandler.EventParser;
import ru.yandex.iex.proxy.eventtickethandler.EventTicketHandler;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;

public class CinemaInfoParser extends EventParser {
    private static final String TIME = "time";
    private static final String KEY = "key";
    private static final String VALUE = "value";
    private static final String EVENT_TICKET = "eventTicket";
    private static final String AUXILIARY_FIELDS = "auxiliaryFields";
    private static final String RELEVANT_DATE = "relevantDate";
    private static final String HEADER_FIELDS = "headerFields";
    private static final String EVENT = "event";
    private static final String SECONDARY_FIELDS = "secondaryFields";
    private static final String DATE = "date";
    private static final String GEOCOORDS = "geocoords";
    private static final String RESERVATION_FOR = "reservationFor";
    private static final String START_DATE1 = "startDate";
    private static final String COM = ",";
    private static final String HYP = "-";
    private static final String ADDRESS = "address";

    private final Map<String, Object> internalFlatFormat = new HashMap<>();

    @Override
    public void parse(
        final Object in,
        final Map<String, Object> out)
    {
        internalFlatFormat.clear();
        if (in instanceof Map) {
            parseOnlyDate(in);
            parseCinemaName(in);
            parseFilmName(in);
            parseGeocoords(in);
            parseOnlyTime(in);
            parseSeats(in);
            parseFullDateAndTimestamp();
            parseTicketNumber(in, out);
            parseCityName(in);
            makeStructure(in, out);
        }
    }

    private void makeStructure(
        final Object inputJson,
        final Map<String, Object> structuredOutJson)
    {
        String[] fieldForCopy = {
            EventTicketHandler.TITLE,
            EventTicketHandler.START_DATE,
            EventTicketHandler.START_DATE_TS,
            EventTicketHandler.WIDGET_SUBTYPE,
            EventTicketHandler.SPECIAL_PARTS,
            EventTicketHandler.DOMAIN,
            EventTicketHandler.SEATS};
        for (final String x : fieldForCopy) {
            XJsonUtils.copyEntry(x, internalFlatFormat, structuredOutJson);
        }
        Map<String, Object> location = new HashMap<>();
        XJsonUtils.copyEntry(
            EventTicketHandler.NAME,
            internalFlatFormat,
            location);
        XJsonUtils.copyEntry(ADDRESS, internalFlatFormat, location);
        XJsonUtils.copyEntry(
            EventTicketHandler.CITY,
            internalFlatFormat,
            location);
        XJsonUtils.copyEntry(GEOCOORDS, internalFlatFormat, location);
        structuredOutJson.put(EventTicketHandler.LOCATION, location);
        Map<String, Object> rawData = new HashMap<>();
        XJsonUtils.copyEntry(
            EventTicketHandler.DATA,
            internalFlatFormat,
            rawData);
        structuredOutJson.put(EventTicketHandler.RAW_DATA, rawData);
        int numSeats = -1;
        if (internalFlatFormat.containsKey(EventTicketHandler.SEATS)) {
            List<Map<String, Object>> res =
                parseSeatsStructure(
                    (String) internalFlatFormat.get(EventTicketHandler.SEATS));
            numSeats = res.size();
            structuredOutJson.put(EventTicketHandler.SEATS, res);
        } else {
            try {
                Object nums =
                    XJsonUtils.getNodeByPathOrNull(inputJson, "numSeats");
                numSeats = ValueUtils.asInt(nums);
            } catch (JsonUnexpectedTokenException e) {
            }
        }
        if (numSeats != -1) {
            if (numSeats == 0) {
                numSeats = 1;
            }
            structuredOutJson.put("num_seats", numSeats);
        }
    }

    private void parseTicketNumber(
        final Object inputJson,
        final Map<String, Object> out)
    {
        Object backFields;
        try {
            backFields =
                XJsonUtils.getNodeByPathOrNull(
                    inputJson,
                    EVENT_TICKET,
                    "backFields");
            if (backFields != null) {
                XStrAlgo.Pair result =
                    getValueFromList(backFields, "ticket-num");
                String tickn = result.getValue();
                if (!tickn.isEmpty()) {
                    out.put("ticket_number", tickn);
                }
            }
        } catch (JsonUnexpectedTokenException e) {
        }
    }

    private void parseFullDateAndTimestamp() {
        if (internalFlatFormat.containsKey(DATE)
            && internalFlatFormat.containsKey(TIME))
        {
            final String fullDate =
                XTimeUtils.formatDate((String) internalFlatFormat.get(DATE))
                + String.valueOf(' ')
                + XTimeUtils.formatTime(
                    (String) internalFlatFormat.get(TIME));
            internalFlatFormat.put(EventTicketHandler.START_DATE, fullDate);
            internalFlatFormat.put(
                EventTicketHandler.START_DATE_TS,
                XTimeUtils.getTimestampISO8601(fullDate));
        }
    }

    private void parseOnlyDate(final Object inputJson) {
        try {
            Object rdate =
                XJsonUtils.getSubMapOfKey(inputJson, RELEVANT_DATE);
            if (rdate == null) {
                rdate = XJsonUtils.getNodeByPathOrNull(
                    inputJson,
                    RESERVATION_FOR,
                    START_DATE1);
            }
            if (rdate == null) {
                rdate = XJsonUtils.getNodeByPathOrNull(
                    inputJson,
                    EVENT_TICKET,
                    RELEVANT_DATE);
            }
            String res = "";
            if (rdate != null) {
                String relevantDate = asString(rdate);
                res = relevantDate.substring(0, relevantDate.indexOf('T'));
            }
            if (res.isEmpty()) {
                Object headerFields =
                    XJsonUtils.getNodeByPathOrNull(
                        inputJson,
                        EVENT_TICKET,
                        HEADER_FIELDS);
                if (headerFields != null) {
                    XStrAlgo.Pair dtime =
                        getValueFromList(headerFields, TIME);
                    String shortDate = dtime.getLabel();
                    if (!shortDate.isEmpty()) {
                        res = XTimeUtils.parseDate(shortDate);
                    }
                }
            }
            if (!res.isEmpty()) {
                internalFlatFormat.put(DATE, XRegexpUtils.toIEXDate(res));
            }
        } catch (JsonUnexpectedTokenException e) {
        }
    }

    private void parseCinemaName(final Object inputJson) {
        try {
            Object secondaryFields =
                XJsonUtils.getNodeByPathOrNull(
                    inputJson,
                    EVENT_TICKET,
                    SECONDARY_FIELDS);
            String cinemaName = "";
            if (secondaryFields != null) {
                XStrAlgo.Pair filmAndCinemaNames =
                    getValueFromList(secondaryFields, EVENT);
                cinemaName = filmAndCinemaNames.getLabel();
                if (cinemaName.isEmpty()) {
                    filmAndCinemaNames =
                        getValueFromList(secondaryFields, "event-venue");
                    cinemaName = filmAndCinemaNames.getLabel();
                    if (!cinemaName.isEmpty()) {
                        String passStrings =
                            asString(
                                XJsonUtils.getNodeByPathOrNull(
                                    inputJson,
                                    "pass.strings",
                                    "ru.lproj"));
                        if (!passStrings.isEmpty()) {
                            cinemaName =
                                XRegexpUtils.getCinemaVenueName(
                                    passStrings);
                        }
                    }
                }
            } else {
                Object name = XJsonUtils.getNodeByPathOrNull(
                    inputJson,
                    RESERVATION_FOR,
                    EventTicketHandler.LOCATION,
                    EventTicketHandler.NAME);
                if (name instanceof String) {
                    cinemaName = asString(name);
                }
            }
            if (!cinemaName.isEmpty()) {
                internalFlatFormat.put(EventTicketHandler.NAME, cinemaName);
            }
        } catch (JsonUnexpectedTokenException e) {
        }
    }

    private void parseCityName(final Object inputJson) {
        try {
            Object name = XJsonUtils.getNodeByPathOrNull(
                inputJson,
                RESERVATION_FOR,
                EventTicketHandler.LOCATION,
                ADDRESS,
                "addressLocality");
            if (name instanceof String) {
                internalFlatFormat.put(EventTicketHandler.CITY, name);
            }
        } catch (JsonUnexpectedTokenException e) {
        }
    }

    private void parseFilmName(final Object inputJson) {
        /*
        "description": "Безумный Макс: Дорога ярости",
        "eventTicket": {
          "secondaryFields": {
            "key": "event",
            "value": "Грань будущего (12+)"
          },
        */
        try {
            String filmName = "";
            Object secondaryFields =
                XJsonUtils.getNodeByPathOrNull(
                    inputJson,
                    EVENT_TICKET,
                    SECONDARY_FIELDS);
            if (secondaryFields != null) {
                XStrAlgo.Pair filmAndCinemaNames =
                    getValueFromList(secondaryFields, EVENT);
                filmName = filmAndCinemaNames.getValue();
            }
            if (!filmName.isEmpty()) {
                internalFlatFormat.put(EventTicketHandler.TITLE, filmName);
            } else {
                Object description =
                    XJsonUtils.getNodeByPathOrNull(
                        inputJson,
                        EventTicketHandler.DESCRIPTION);
                if (description == null) {
                    description = XJsonUtils.getNodeByPathOrNull(
                        inputJson,
                        RESERVATION_FOR,
                        EventTicketHandler.NAME);
                }
                if (description != null) {
                    internalFlatFormat.put(
                        EventTicketHandler.TITLE,
                        description);
                }
            }
        } catch (JsonUnexpectedTokenException e) {
        }
    }

    private void parseGeocoords(final Object inputJson) {
        try {
            Object locations =
                XJsonUtils.getNodeByPathOrNull(inputJson, "locations");
            if (locations != null) {
                internalFlatFormat.put(GEOCOORDS, locations);
            }
        } catch (JsonUnexpectedTokenException e) {
        }
    }

    private void parseOnlyTime(final Object inputJson) {
        try {
            Object headerFields =
                XJsonUtils.getNodeByPathOrNull(
                    inputJson,
                    EVENT_TICKET,
                    HEADER_FIELDS);
            XStrAlgo.Pair dtime = getValueFromList(headerFields, TIME);
            String res = dtime.getValue();
            if (res.isEmpty()) {
                dtime = getValueFromList(headerFields, "start-date");
                res = dtime.getValue();
            }
            if (res.isEmpty()) {
                Object time = XJsonUtils.getNodeByPathOrNull(
                    inputJson,
                    RESERVATION_FOR,
                    START_DATE1);
                if (time instanceof String) {
                    res = XTimeUtils.getTimeFromTimeAndGMT(asString(time));
                }
            }
            internalFlatFormat.put(TIME, res);
        } catch (JsonUnexpectedTokenException e) {
        }
    }

    @SuppressWarnings("StringSplitter")
    private void parseSeats(final Object inputJson) {
        /*
        "eventTicket": {
           "auxiliaryFields": [
            {
                "key": "tickets",
                "label": "Места",
                "value": "ряд 6, места 11, 12 (2 билета)"
            },
            or
            "auxiliaryFields": [
            {
               "value": "ряд 4, места 9, 10, 11, 12 (4 билета)",
               "label": "Зал 15",
               "key": "tickets"
            },
           or
            "auxiliaryFields": [
            {
                 "key": "tickets",
                 "value": "2 билета (ряд 14, места 22, 23)",
                 "label": "Места"
            },
            or
             "auxiliaryFields": [
            {
                "__type": "textField",
                "key": "hall",
                "label": "hall",
                "value": "Зал 5"
            },
            {
                "__type": "textField",
                "key": "row",
                "label": "row",
                "value": "5"
            },
            {
                "__type": "textField",
                "key": "seats",
                "label": "seats",
                "value": "7, 8"
            }
        ],
         */
        try {
            Object auxiliaryFields =
                XJsonUtils.getNodeByPathOrNull(
                    inputJson,
                    EVENT_TICKET,
                    AUXILIARY_FIELDS);
            if (auxiliaryFields != null) {
                XStrAlgo.Pair seats =
                    getValueFromList(auxiliaryFields, "tickets");
                if (!seats.getLabel().isEmpty()
                    && !seats.getValue().isEmpty())
                {
                    internalFlatFormat.put(
                        EventTicketHandler.SEATS,
                        seats.getLabel() + ": " + seats.getValue());
                    return;
                }
                XStrAlgo.Pair hall =
                    getValueFromList(auxiliaryFields, EventTicketHandler.HALL);
                XStrAlgo.Pair row =
                    getValueFromList(auxiliaryFields, EventTicketHandler.ROW);
                seats =
                    getValueFromList(auxiliaryFields, EventTicketHandler.SEATS);
                String seatStr = ", мест";
                if (seats.getValue().contains(COM)
                    || seats.getValue().contains(HYP))
                {
                    seatStr += "а ";
                } else {
                    seatStr += "о ";
                }
                String rawSeats = seats.getValue();
                String rawRows = row.getValue();
                String rawHalls = hall.getValue();
                if (rawRows.contains(HYP) && rawSeats.contains(HYP)) {
                    // 1-2-3-4     -> 1,2,3,4,4
                    // 6-4-5-5, 6  -> 6,4,5,5,6
                    String[] rwr = rawRows.split(HYP);
                    String[] rws = rawSeats.split(HYP);
                    if (rwr.length == rws.length) {
                        StringBuilder nrs = new StringBuilder();
                        StringBuilder nss = new StringBuilder();
                        for (int i = 0; i < rwr.length; i++) {
                            String[] rrow = rwr[i].split(COM);
                            String[] rseat = rws[i].split(COM);
                            for (int j = 0; j < rseat.length; j++) {
                                nrs.append(COM);
                                nrs.append(rrow[j % rrow.length]);
                                nss.append(COM);
                                nss.append(rseat[j]);
                            }
                        }
                        rawRows = nrs.toString();
                        rawSeats = nss.toString();
                    }
                }
                rawSeats = rawSeats.replace(HYP, COM);
                rawRows = rawRows.replace(HYP, COM);
                rawHalls = rawHalls.replace(HYP, COM);
                internalFlatFormat.put(
                    EventTicketHandler.SEATS,
                    rawHalls
                    + ": ряд " + rawRows
                    + seatStr + rawSeats);
            }
        } catch (JsonUnexpectedTokenException e) {
        }
    }

    private XStrAlgo.Pair getValueFromList(final Object l, final String key)
        throws JsonUnexpectedTokenException
    {
        return getValueFromList(l, key, "label");
    }

    private XStrAlgo.Pair getValueFromList(
        final Object l,
        final String key,
        final String second)
        throws JsonUnexpectedTokenException
    {
        XStrAlgo.Pair res = new XStrAlgo.Pair();
        if (l instanceof List) {
            for (final Object x : ValueUtils.asList(l)) {
                Map<?, ?> tmpt = ValueUtils.asMap(x);
                if (tmpt.containsKey(KEY) && tmpt.containsKey(VALUE)) {
                    if (asString(tmpt.get(KEY)).equals(key)) {
                        res.setFirst(asString(tmpt.get(VALUE)));
                        if (tmpt.containsKey(second)) {
                            res.setSecond(asString(tmpt.get(second)));
                        }
                        break;
                    }
                }
            }
        } else if (l instanceof Map) {
            Map<?, ?> tmpt = ValueUtils.asMap(l);
            if (tmpt.containsKey(KEY) && tmpt.containsKey(VALUE)) {
                if (asString(tmpt.get(KEY)).equals(key)) {
                    res.setFirst(asString(tmpt.get(VALUE)));
                    if (tmpt.containsKey(second)) {
                        res.setSecond(asString(tmpt.get(second)));
                    }
                }
            }
        }
        return res;
    }

    private String asString(final Object o)
        throws JsonUnexpectedTokenException
    {
        String res = "";
        if (o instanceof String) {
            res = ValueUtils.asString(o);
        } else if (o instanceof JsonString) {
            res = ((JsonString) o).asString();
        }
        return res;
    }
}
