package ru.yandex.iex.proxy;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;

import org.apache.http.Header;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.dbfields.PgFields;
import ru.yandex.http.config.HttpHostConfig;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.HttpExceptionConverter;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.NotFoundException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.iex.proxy.cacheupdate.FactsCacheUpdater;
import ru.yandex.io.FlushOnCloseWriter;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.search.document.mail.FirstlineMailMetaInfo;

public class FactsCachingCallback
    implements FutureCallback<List<Solution>>
{
    private static final char UNDERSCORE = '_';
    private static final long MILLIS = 1000L;
    private static final String NO_FACTS = "no_facts";
    private static final String LOG_PREFIX = "FactsCachingCallback: ";
    private static final String DOCS = "docs";
    private static final String PREFIX = "prefix";
    private static final String PREFIXN = "&prefix=";
    private static final String FACTSN = "&facts";
    private static final String FACT_IS_COKE_SOLUTION = "fact_is_coke_solution";
    private static final String FACT_DATA = "fact_data";
    private static final String FACT_NAME = "fact_name";
    private static final String FACT_MID = "fact_mid";
    private static final String FACT_EVENT_ID = "fact_event_id";
    private static final String FACT_EVENT_RECURRENCE_ID =
        "fact_event_recurrence_id";
    private static final String URL = "url";
    private static final String COKE_SUFF = "_coke";
    private static final String FACTS_PREFIX = "facts_";

    private final ChangeContext changeContext;
    private final AbstractContext context;
    private final List<?> changed;
    private final int batchSize;
    private final FutureCallback<CacheSentSolutions> callback;
    private List<FirstlineMailMetaInfo> metasWithEmptyFacts = null;
    private String xIndexOperationQueueName = null;
    private final List<String> updateCache;
    private final boolean onlyModifyRequest;
    private final boolean onlyAddRequest;
    private final boolean fromNotify;
    private final boolean cacheResult;

    public FactsCachingCallback(final OffsetCallback callback)
        throws JsonUnexpectedTokenException
    {
        this.context = callback.context();
        this.changeContext = callback.context();
        this.changed = ValueUtils.asList(context.json().get("changed"));
        batchSize = callback.batchSize();
        this.callback = callback;
        if (changeContext.isTteot()) {
            xIndexOperationQueueName = context.iexProxy().
                xIndexOperationQueueNameBacklog();
        }
        if (changeContext.zooQueueIsIexUpdate()) {
            xIndexOperationQueueName = context.iexProxy().
                xIndexOperationQueueNameUpdate();
        }
        updateCache = changeContext.updateCache();
        onlyModifyRequest = false;
        onlyAddRequest = (updateCache == null) || updateCache.isEmpty();
        fromNotify = true;
        this.cacheResult = true;
    }

    public FactsCachingCallback(
        final FactsContext context,
        final FutureCallback<CacheSentSolutions> callback)
    {
        this.context = context;
        this.changeContext = null;
        this.changed = null;
        batchSize = -1;
        this.callback = callback;
        xIndexOperationQueueName = context.iexProxy().
            xIndexOperationQueueNameFacts();
        onlyModifyRequest = true;
        onlyAddRequest = false;
        updateCache = null;
        fromNotify = false;
        this.cacheResult = context.updateCache();
    }

    public ChangeContext changeContext() {
        return this.changeContext;
    }

    public AbstractContext context() {
        return this.context;
    }

    public List<?> changed() {
        return this.changed;
    }

    public int batchSize() {
        return batchSize;
    }

    public void addMetasWithEmptyFacts(
        final List<FirstlineMailMetaInfo> metas)
    {
        this.metasWithEmptyFacts = metas;
    }

    @Override
    public void cancelled() {
        this.context.session()
            .logger()
            .warning("Request cancelled: "
                + this.context.session().listener().details());
        context.session().response(YandexHttpStatus.SC_CLIENT_CLOSED_REQUEST);
    }

    @Override
    public void failed(final Exception e) {
        this.context.session()
            .logger()
            .log(Level.WARNING, "Failed to process: "
                + this.context.humanReadableJson()
                + '\n' + this.context.session().listener().details()
                + " because of exception", e);
        this.context.session()
            .handleException(HttpExceptionConverter.toHttpException(e));
    }

    private void addKeyValue(
        final JsonWriter writer,
        final String key,
        final Object value) throws IOException
    {
        writer.key(key);
        writer.value(value);
    }

    // CSOFF: ParameterNumber
    private boolean addObjectsFromMap(
        final JsonWriter writer,
        final Map<String, Object> map,
        final IndexationContext<Solution> context,
        final boolean cokemulatorSolution,
        final FactsCacheUpdater factsCacheUpdater)
        throws IOException, JsonUnexpectedTokenException
    {
        Long uid = ValueUtils.asLongOrNull(this.context.uid());
        String mid = context.mid();
        int added = 0;
        for (Map.Entry<String, Object> entry : map.entrySet()) {
            String factName = entry.getKey();
            final String visibleFactName;
            if (cokemulatorSolution) {
                visibleFactName = UNDERSCORE + factName;
            } else {
                visibleFactName = factName;
            }
            if (!context.abstractContext().iexProxy().cachingFact(
                visibleFactName))
            {
                continue;
            } else {
                added++;
            }
            writer.startObject();
            Object factData = entry.getValue();
            String url = FACTS_PREFIX + uid + UNDERSCORE + mid
                + UNDERSCORE + factName;
            if (cokemulatorSolution) {
                url = url + COKE_SUFF;
            }
            addKeyValue(writer, URL, url);
            addKeyValue(writer, FACT_MID, mid);
            addKeyValue(writer, "fact_uid", uid);
            addKeyValue(writer, "fact_stid", context.stid());
            addKeyValue(writer, "fact_received_date", context.receivedDate());
            addKeyValue(
                writer,
                "fact_last_extracted_date",
                System.currentTimeMillis() / MILLIS);
            addKeyValue(
                writer,
                "fact_message_type",
                context.meta().get("message_type"));
            addKeyValue(writer, FACT_NAME, factName);
            Object externalId = null;
            Object recurrenceId = null;
            try {
                Map<?, ?> factDataMap = ValueUtils.asMapOrNull(factData);
                if (factDataMap != null) {
                    externalId = factDataMap.get("externalEventId");
                    recurrenceId = factDataMap.get("recurrenceEventId");
                }
            } catch (JsonUnexpectedTokenException ignored) {
            }
            if (externalId != null) {
                addKeyValue(writer, FACT_EVENT_ID, externalId);
                if (recurrenceId != null) {
                    addKeyValue(writer, FACT_EVENT_RECURRENCE_ID, recurrenceId);
                }
            }
            if (factData == null) {
                addKeyValue(writer, FACT_DATA, null);
            } else {
                writer.key(FACT_DATA);
                writer.startString();
                try (JsonWriter writer2 =
                         new JsonWriter(new FlushOnCloseWriter(writer)))
                {
                    writer2.value(factData);
                }
                writer.endString();
            }

            if (fromNotify && !cokemulatorSolution) {
                factsCacheUpdater.addCacheData(context, factData, externalId);
            }

            addKeyValue(writer, "fact_from", context.email());
            addKeyValue(writer, "fact_domain", context.domain());
            addKeyValue(writer, FACT_IS_COKE_SOLUTION, cokemulatorSolution);
            writer.endObject();
        }
        return added > 0;
    }

    @Override
    public void completed(final List<Solution> result) {
        if (!cacheResult) {
            context.session().logger().info(LOG_PREFIX
                + "in completed, will not cache result");
            callback.completed(new CacheSentSolutions(result));
        } else {
            context.session().logger().info(LOG_PREFIX
                + "in completed, will cache, result");
            HttpHostConfig producerAsyncClientConfig = context.
                iexProxy().config().producerAsyncClientConfig();
            String queueName = context.iexProxy().
                config().factsIndexingQueueName();
            FutureCallback<List<List<Object>>> callAdapter =
                new EmptyResponseCallbackAdapter(
                    callback,
                    result,
                    context.iexProxy());
            MultiFutureCallback<List<Object>> multiCallback =
                new MultiFutureCallback<>(callAdapter);
            if (producerAsyncClientConfig != null
                && queueName != null && result != null)
            {
                if (onlyModifyRequest) {
                    context.session().logger().info(LOG_PREFIX
                        + "preparing to send only modify request");
                    // will send 'modify'
                    sendRequestToLucene(
                        true,
                        result,
                        multiCallback.newCallback());
                } else if (onlyAddRequest) {
                    context.session().logger().info(LOG_PREFIX
                        + "preparing to send only add request");
                    // will send 'add'
                    sendRequestToLucene(
                        false,
                        result,
                        multiCallback.newCallback());
                } else {
                    // will try to send both requests, maybe will not send
                    // 'add'in case of all facts to be send via 'modify' in
                    // update_cache param
                    context.session().logger().info(LOG_PREFIX
                        + "will try to send both 'add' and 'modify' requests");
                    sendRequestToLucene(
                        false,
                        result,
                        multiCallback.newCallback());
                    sendRequestToLucene(
                        true,
                        result,
                        multiCallback.newCallback());
                }
            }
            multiCallback.done();
        }
    }

    // CSOFF: MethodLength
    @SuppressWarnings("FutureReturnValueIgnored")
    private void sendRequestToLucene(
        final boolean rewrite,
        final List<Solution> result,
        final FutureCallback<List<Object>> call)
    {
        try {
            MultiFutureCallback<Object> multi
                = new MultiFutureCallback<>(call);
            HttpRequest request = context.session().request();
            HttpHostConfig producerAsyncClientConfig = context.
                iexProxy().config().producerAsyncClientConfig();
            String queueName = context.iexProxy().
                config().factsIndexingQueueName();
            HttpHost host = producerAsyncClientConfig.host();
            Long operationId = ValueUtils.asLongOrNull(
                context.json().get(PgFields.OPERATION_ID));
            String uri;
            if (rewrite) {
                uri = "/modify?service=";
            } else {
                uri = "/add?service=";
            }
            uri += queueName + FACTSN + PREFIXN + context.uid();
            if (operationId != null) {
                uri = uri + "&operationId=" + operationId;
            }
            StringBuilderWriter sbw =
                new StringBuilderWriter(new StringBuilder(""));
            boolean nonEmptyRequest = false;
            try (JsonWriter writer = new JsonWriter(sbw)) {
                writer.startObject();
                writer.key(PREFIX);
                writer.value(context.uid());
                writer.key(DOCS);
                writer.startArray();
                if (result.size() > 0) {
                    String mid = result.get(0).context().mid();
                    context.session.logger().info(LOG_PREFIX
                        + "preparing data for caching facts for uid "
                        + context.uid() + " for mid " + mid);
                }
                FactsCacheUpdater factsCacheUpdater =
                    new FactsCacheUpdater(context, xIndexOperationQueueName);
                for (Solution solution : result) {
                    if (updateCache == null || updateCache.isEmpty()) {
                        nonEmptyRequest |= addObjectsFromMap(
                            writer,
                            solution.cokemulatorSolutions(),
                            solution.context(),
                            true,
                            factsCacheUpdater);
                        nonEmptyRequest |= addObjectsFromMap(
                            writer,
                            solution.postActionSolutions(),
                            solution.context(),
                            false,
                            factsCacheUpdater);
                    } else {
                        Map<String, Object> cokeSolutionsToProcess =
                            new HashMap<>();
                        for (Map.Entry<String, Object> entry
                            : solution.cokemulatorSolutions().entrySet())
                        {
                            if ((rewrite
                                && updateCache.contains(entry.getKey()))
                                || (!rewrite
                                && !updateCache.contains(entry.getKey())))
                            {
                                cokeSolutionsToProcess.put(
                                    entry.getKey(),
                                    entry.getValue());
                            }
                        }
                        nonEmptyRequest |= addObjectsFromMap(
                            writer,
                            cokeSolutionsToProcess,
                            solution.context(),
                            true,
                            factsCacheUpdater);
                        Map<String, Object> postActionsToProcess =
                            new HashMap<>();
                        for (Map.Entry<String, Object> entry
                            : solution.postActionSolutions().entrySet())
                        {
                            if ((rewrite
                                && updateCache.contains(entry.getKey()))
                                || (!rewrite
                                && !updateCache.contains(entry.getKey())))
                            {
                                postActionsToProcess.put(
                                    entry.getKey(),
                                    entry.getValue());
                            }
                        }
                        nonEmptyRequest |= addObjectsFromMap(
                            writer,
                            postActionsToProcess,
                            solution.context(),
                            false,
                            factsCacheUpdater);
                    }
                }
                if ((updateCache == null || updateCache.isEmpty())
                    || (rewrite && updateCache.contains(NO_FACTS))
                    || (!rewrite && !updateCache.contains(NO_FACTS)))
                {
                    if (metasWithEmptyFacts != null) {
                        for (FirstlineMailMetaInfo meta : metasWithEmptyFacts) {
                            Map<String, Object> noFactsToNull = new HashMap<>();
                            noFactsToNull.put(NO_FACTS, null);
                            nonEmptyRequest = true;
                            addObjectsFromMap(
                                writer,
                                noFactsToNull,
                                new IndexationContext<>(
                                    context,
                                    meta,
                                    null),
                                false,
                                factsCacheUpdater);
                        }
                    }
                }
                writer.endArray();
                writer.endObject();
                factsCacheUpdater.update(multi);
            } catch (IOException | NotFoundException e) {
                throw new RuntimeException(e);
            }
            if (nonEmptyRequest) {
                if (rewrite) {
                    context.session.logger().info(
                        "Lucene request type: MODIFY");
                } else {
                    context.session.logger().info("Lucene request type: ADD");
                }
                BasicAsyncRequestProducerGenerator generator =
                    new BasicAsyncRequestProducerGenerator(
                        uri, sbw.toString());
                generator.addHeader(YandexHeaders.SERVICE, queueName);
                generator.addHeader(
                    YandexHeaders.X_INDEX_OPERATION_TIMESTAMP,
                    Long.toString(context.operationDateMillis()));
                if (xIndexOperationQueueName != null) {
                    generator.addHeader(
                        YandexHeaders.X_INDEX_OPERATION_QUEUE,
                        xIndexOperationQueueName);
                }
                Header zooShardId = request.getFirstHeader(
                    YandexHeaders.ZOO_SHARD_ID);
                if (zooShardId != null) {
                    generator.addHeader(zooShardId);
                }
                AsyncClient producerAsyncClient =
                    context.iexProxy().producerAsyncClient().adjust(
                        context.session().context());
                producerAsyncClient.execute(
                    host,
                    generator,
                    AsyncStringConsumerFactory.OK,
                    context.session().listener()
                        .createContextGeneratorFor(producerAsyncClient),
                    multi.newCallback());
            }
            multi.done();
        } catch (JsonUnexpectedTokenException e) {
            context.session().logger().log(
                Level.WARNING,
                "Failed to send request to lucene",
                e);
        }
    }

    private static class EmptyResponseCallbackAdapter
        extends AbstractFilterFutureCallback<List<List<Object>>,
        CacheSentSolutions>
    {
        private final List<Solution> result;
        private final long startTime;
        private final IexProxy iexProxy;

        EmptyResponseCallbackAdapter(
            final FutureCallback<CacheSentSolutions> callback,
            final List<Solution> result,
            final IexProxy iexProxy)
        {
            super(callback);
            this.result = result;
            this.startTime = System.currentTimeMillis();
            this.iexProxy = iexProxy;
        }

        @Override
        public void failed(final Exception e) {
            final long time = System.currentTimeMillis() - startTime;
            iexProxy.cacheStoreTime(time);
            super.failed(e);
        }

        @Override
        public void completed(final List<List<Object>> queueIds) {
            long maxQueueId = Long.MIN_VALUE;
            try {
                for (List<Object> list: queueIds) {
                    for (Object position: list) {
                        if (position != null && !position.toString().isEmpty()) {
                            long queueId = Long.parseLong(position.toString());
                            if (queueId > maxQueueId) {
                                maxQueueId = queueId;
                            }
                        }
                    }
                }
            } catch (NumberFormatException nfe) {
                iexProxy.logger().log(
                    Level.WARNING,
                    "Producer returned bad queueId",
                    nfe);
            }

            final long time = System.currentTimeMillis() - startTime;
            iexProxy.cacheStoreTime(time);
            callback.completed(new CacheSentSolutions(this.result, maxQueueId));
        }
    }
}
