package ru.yandex.iex.proxy;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

import io.netty.util.HashedWheelTimer;
import io.netty.util.Timeout;
import io.netty.util.TimerTask;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;

import ru.yandex.client.producer.ProducerClient;
import ru.yandex.client.producer.QueueHostInfo;
import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.iex.proxy.xiva.XivaFactsUpdatingCallback;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.PrimitiveHandler;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.parser.searchmap.SearchMapHost;
import ru.yandex.parser.searchmap.SearchMapShard;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.document.mail.FirstlineMailMetaInfo;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.proxy.SearchResultConsumerFactory;
import ru.yandex.search.result.BasicSearchResult;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;
import ru.yandex.util.string.StringUtils;

public class FactsHandler
    extends AbstractGetMidsHandler<FactsContext, Solution>
{
    //private static final String COKEDUMP = "cokedump";
    private static final String EXTRACT = "extract";
    private static final String RETRY = "retry";
    private static final String TRUE = "true";
    //private static final String UID = "uid";
    private static final String MID = "mid";
    private static final String NO_FACTS = "no_facts";
    private static final String FACTS_HANDLER = "FactsHandler: ";
    private static final String OR_DELIMITER = " OR ";
    private static final String URL = "url";
    private static final String EMPTY_FIELD = "{}";
    private static final String DP = "dp";
    private static final char COMMA = ',';
    private static final int PERC = 100;
    private static final int MAX_DEADLINE_DRIFT = 5;
    private static final HashedWheelTimer DELAYED_EXECUTOR =
        newHashedWheelTimer();
//    private static final ScheduledThreadPoolExecutor DELAYED_EXECUTOR =
//        new ScheduledThreadPoolExecutor(
//            Runtime.getRuntime().availableProcessors(),
//            new ThreadPoolExecutor.CallerRunsPolicy());

    public FactsHandler(final IexProxy iexProxy) {
        super(iexProxy);
    }

    private static HashedWheelTimer newHashedWheelTimer() {
        HashedWheelTimer timer =
            new HashedWheelTimer(1 + 2 + 2, TimeUnit.MILLISECONDS);
        timer.start();
        return timer;
    }

    @Override
    protected FactsContext createContext(
        final ProxySession session,
        final IexProxy iexProxy)
        throws HttpException
    {
        return new FactsContext(session, iexProxy);
    }

    @Override
    protected AbstractFilterSearchCallback<Solution>
        createFilterSearchCallback(
            final FactsContext context,
            final Set<String> mids)
    {
        return new FilterSearchCallback(
            context,
            new FactsCachingCallback(
                context,
                new XivaFactsUpdatingCallback(
                    context,
                    new FactsCallback(context, null))),
            mids);
    }

    @Override
    @SuppressWarnings("FutureReturnValueIgnored")
    protected void completed(final FactsContext factsContext)
        throws HttpException
    {
        ProducerClient producerClient = factsContext.iexProxy().
            producerClient();
        AsyncClient searchClient =
            factsContext.iexProxy().searchClient().adjust(
                    factsContext.session().context());
        boolean ignoreCacheParams = false;
        try {
            ignoreCacheParams = factsContext.session().
                params().getBoolean("ignore_cache", false);
        } catch (BadRequestException e) {
            // just log and skip, ignoreCacheParams is false
            log(factsContext, "Exception in getting ignore_cache param:" + e);
        }
        if (ignoreCacheParams) {
            log(factsContext, "ignoring cache");
            processFSRequest(factsContext, new HashSet<>(factsContext.mids()));
        } else {
            if (producerClient != null) {
                long start = System.currentTimeMillis();
                Set<String> uniqueMids;
                uniqueMids = new HashSet<>(factsContext.mids());
                log(factsContext, "request for lucene for uid "
                        + factsContext.prefix()
                        + ", mids " + uniqueMids);
                String serviceName = factsContext.iexProxy().config().
                        factsIndexingQueueName();
                Prefix prefix = new LongPrefix(factsContext.prefix());
                User user = new User(serviceName, prefix);

                // TODO check times for statistics
                SearchMapShard searchMapShard = factsContext.iexProxy().
                    searchMap().apply(user);

                ParseFactsFromLuceneCallback luceneCallback =
                    new ParseFactsFromLuceneCallback(factsContext);

                SearchResultsSynchronizer synchronizer =
                    new SearchResultsSynchronizer(
                            luceneCallback,
                            factsContext,
                            searchMapShard.size());
                final long almostAllFactsTimeout =
                    factsContext.iexProxy().almostAllFactsTimeout();
                DELAYED_EXECUTOR.newTimeout(
//                producerClient.scheduleRetry(
                        new RequestDeadlineTimerTask(
                            synchronizer,
                            System.currentTimeMillis() + almostAllFactsTimeout),
                            almostAllFactsTimeout,
                            TimeUnit.MILLISECONDS);
                log(factsContext, "retry scheduled in "
                    + (System.currentTimeMillis() - start)
                    + " ms");

                final BasicAsyncRequestProducerGenerator request =
                    new BasicAsyncRequestProducerGenerator(
                            getRequestString(
                                factsContext.uid(),
                                factsContext.iexProxy().config().
                                    factsIndexingQueueName(),
                                uniqueMids,
                                factsContext.factNames()));
                for (SearchMapHost host : searchMapShard) {
                    final HostSearchMapResultCallback callback =
                        new HostSearchMapResultCallback(
                                factsContext,
                                synchronizer,
                                host.searchHost(),
                                start);
                    searchClient.execute(
                        host.searchHost(),
                        request,
                        SearchResultConsumerFactory.OK,
                        factsContext.session().listener().
                            createContextGeneratorFor(searchClient),
                        new ErrorSuppressingFutureCallback<>(
                            callback,
                            new BasicSearchResult()));
                }
                log(factsContext, "backend requests sent in "
                    + (System.currentTimeMillis() - start)
                    + " ms ");
                producerClient.executeWithInfo(
                    user,
                    factsContext.session().listener().
                        createContextGeneratorFor(producerClient),
                    new ErrorSuppressingFutureCallback<>(
                        new HostsCallback(
                            factsContext,
                            synchronizer,
                            start),
                        Collections.<QueueHostInfo>emptyList()));
                log(factsContext, "producer request sent in "
                    + (System.currentTimeMillis() - start)
                    + " ms  ");
            } else {
                log(factsContext, "producerClient is null");
                super.completed(factsContext);
            }
        }
    }

    private void processFSRequest(
        final FactsContext context,
        final Set<String> mids)
        throws BadRequestException
    {
        createFilterSearchCallback(context, mids).execute();
    }

    private static class FilterSearchCallback
        extends AbstractFilterSearchCallback<Solution>
    {
        private final FactsCachingCallback factsCachingCallback;

        FilterSearchCallback(
            final AbstractContext context,
            final FactsCachingCallback callback,
            final Set<String> mids)
        {
            super(context, callback, mids);
            factsCachingCallback = callback;
        }

        @Override
        public SolutionPassThroughtCallback subMessageCallback(
            final IndexationContext<Solution> indexationContext)
        {
            return new SolutionPassThroughtCallback(indexationContext);
        }

        @Override
        public void executeSubCallback(
            final AbstractCallback<Solution> callback)
        {
            Blackbox2CokemulatorCallback
                .cokemulatorExecuter(callback);
        }

        @SuppressWarnings("unused")
        public FactsCachingCallback factsCachingCallback() {
            return factsCachingCallback;
        }

        @Override
        public void addMetasWithEmptyFacts(
                final List<FirstlineMailMetaInfo> metas)
        {
            factsCachingCallback.addMetasWithEmptyFacts(metas);
        }
    }

    private static class SolutionPassThroughtCallback
        extends AbstractCallback<Solution>
    {
        SolutionPassThroughtCallback(
            final IndexationContext<Solution> context)
        {
            super(context);
        }

        @Override
        public void completed(final Solution solution) {
            context.callback().completed(solution);
        }
    }

    //CSOFF: ParameterNumber
    private static class HostSearchMapResultCallback
         extends AbstractProxySessionCallback<SearchResult>
    {
        private final SearchResultsSynchronizer sync;
        private final HttpHost host;
        private final FactsContext context;
        private final long startTime;

        HostSearchMapResultCallback(
                final FactsContext context,
                final SearchResultsSynchronizer sync,
                final HttpHost host,
                final long startTime)
        {
            super(context.session());
            this.context = context;
            this.sync = sync;
            this.host = host;
            this.startTime = startTime;
        }

        @Override
        public void completed(final SearchResult result) {
            context.iexProxy().perDcLuceneTime(
                System.currentTimeMillis() - startTime,
                host.getHostName());
            log(context, "Got search result from: " + host + ", "
                + "queueId: " + result.zooQueueId()
                + ", time: " + (System.currentTimeMillis() - startTime));
            sync.gotSearchResult(host, result);
        }

        @Override
        public void failed(final Exception e) {
            log(context, "Search failed for host: " + host, e);
            sync.searchFailed(host);
        }
    }
    //CSON: ParameterNumber

    private class ParseFactsFromLuceneCallback
         extends AbstractProxySessionCallback<SearchResult>
    {
        private FactsContext context;
        private long startLucene;

        ParseFactsFromLuceneCallback(
                final FactsContext factsContext)
        {
            super(factsContext.session());
            context = factsContext;
            startLucene = System.currentTimeMillis();
        }

        //CSOFF: MethodLength - TODO remove when logs number will reduce
        @Override
        public void completed(final SearchResult result) {
            long timeTaken = System.currentTimeMillis() - startLucene;
            log(context, "got result from lucene");
            context.iexProxy().luceneCompleted(timeTaken);
            context.iexProxy().addLucenePureTime(timeTaken);
            Set<String> actualMidsFromCache = new HashSet<String>();
            Set<String> midsFromCacheToResponse = new HashSet<String>();
            List<CutSolution> cachedFacts = new ArrayList<CutSolution>();
            Map<String, HashMap<String, Object>> midsToCokeSolutions =
                new HashMap<String, HashMap<String, Object>>();
            Map<String, HashMap<String, Object>> midsToPostActionSolutions =
                new HashMap<String, HashMap<String, Object>>();
            boolean cacheUpdateNeeded = false;
            boolean extractionNeeded = false;
            boolean retry = true;
            try {
                extractionNeeded = context.session().
                    params().getBoolean(EXTRACT, false);
            } catch (BadRequestException e) {
                // just log and skip, extractionNeeded is false
                log(context, "Exception in getting extract param:" + e);
            }
            try {
                retry = context.session().params().getBoolean(RETRY, true);
            } catch (BadRequestException e) {
                log(context, "Exception in getting retry param:" + e);
            }
            if (extractionNeeded) {
                log(context, "extraction is needed");
            } else {
                log(context, "extraction is not needed");
            }
            Set<String> midsForRequestToFS = new HashSet<String>();
            final BasicGenericConsumer<JsonObject, JsonException> consumer =
                new BasicGenericConsumer<>();
            final JsonParser jsonParser = new JsonParser(
                new StackContentHandler(
                    new TypesafeValueContentHandler(
                        consumer)));

            for (SearchDocument document: result.hitsArray()) {
                String url = document.attrs().get(URL);
                if (url.startsWith("facts_null")) {
                    // do not consider those wrong records
                    continue;
                }
                String mid = document.attrs().get("fact_mid");
                String factString = document.attrs().get("fact_data");
                Long factReceivedDate = Long.parseLong(document.attrs().
                    get("fact_received_date"));
                Long factLastExtractedDate;
                try {
                    factLastExtractedDate = Long.parseLong(document.attrs().
                        get("fact_last_extracted_date"));
                } catch (NumberFormatException e) {
                    // for backward compatibility, in case of null value
                    factLastExtractedDate = factReceivedDate;
                }
                String factName = document.attrs().get("fact_name");
                Long factActualTimestamp = context.iexProxy().
                    getFactTimeStamp(factName);
                boolean isCokeSolution = false;
                if (extractionNeeded) {
                    log(context, "factName: " + factName
                        + ", fact mid:" + mid
                        + ", last extracted:" + factLastExtractedDate
                        + ", actual:" + factActualTimestamp);
                }
                try {
                    isCokeSolution = ValueUtils.asBoolean(
                        document.attrs().get("fact_is_coke_solution"));
                } catch (JsonUnexpectedTokenException e) {
                    // ignore, isCokeSolution remains false;
                }
                if (factLastExtractedDate < factActualTimestamp) {
                    cacheUpdateNeeded = true;
                    midsForRequestToFS.add(mid);
                    if (extractionNeeded) {
                        log(context, "will update because of " + factName);
                    }
                }
                JsonObject obj = null;
                if (factString != null) {
                    try {
                        jsonParser.parse(factString);
                        obj = consumer.get();
                    } catch (JsonException e) { // skip, obj is null
                    }
                }
                HashMap<String, Object> cokemulatorSolutions =
                    midsToCokeSolutions.computeIfAbsent(mid, k -> new HashMap<>());
                HashMap<String, Object> postActionSolutions =
                    midsToPostActionSolutions.computeIfAbsent(mid, k -> new HashMap<>());
                midsFromCacheToResponse.add(mid);
                if (!factName.equals(NO_FACTS)) {
                    if (isCokeSolution) {
                        cokemulatorSolutions.put(factName, obj);
                    } else {
                        postActionSolutions.put(factName, obj);
                    }
                }
            }
            if (cacheUpdateNeeded) {
                log(context, "not all cache is actual, "
                        + "silent update is needed");
            }
            for (String mid : midsFromCacheToResponse) {
                Map<String, Object> postActionSolutions =
                    midsToPostActionSolutions.get(mid);
                Map<String, Object> cokemulatorSolutions =
                    midsToCokeSolutions.get(mid);
                CutSolution cutSolution = new CutSolution(
                    cokemulatorSolutions,
                    postActionSolutions,
                    mid
                );
                cachedFacts.add(cutSolution);
            }
            context.addCachedFacts(cachedFacts);
            log(context, "got facts for uid " + context.prefix()
                    + " for mids " + midsFromCacheToResponse);
            for (final String mid : context.mids()) {
                if (!midsFromCacheToResponse.contains(mid)) {
                    midsForRequestToFS.add(mid);
                }
            }
            for (final String mid : context.mids()) {
                if (!midsForRequestToFS.contains(mid)) {
                    actualMidsFromCache.add(mid);
                }
            }
            if (!extractionNeeded) {
                int size = context.mids().size();
                log(context, "cached facts ratio is "
                    + PERC * midsFromCacheToResponse.size() / size + '%');
                log(context, "actual cached facts ratio is "
                    + PERC * actualMidsFromCache.size() / size + '%');
                context.iexProxy().
                    addFactsCacheHitRatio(midsFromCacheToResponse.size(), size);
                context.iexProxy().addActualFactsCacheHitRatio(
                        actualMidsFromCache.size(),
                        size);
            }
            FactsCallback callback = new FactsCallback(context, null);
            if (extractionNeeded) {
                log(context, "will extract");
                if (!midsForRequestToFS.isEmpty()) {
                    try {
                        if (cacheUpdateNeeded) {
                            log(context, "update cache is needed");
                        } else {
                            log(context, "cache update is not needed");
                        }
                        processFSRequest(context, midsForRequestToFS);
                    } catch (BadRequestException e) {
                        callback.failed(e);
                    }
                } else {
                    log(context, "nothing to re-extract");
                    callback.completed(Collections.emptyList());
                }
            } else {
                if (!midsForRequestToFS.isEmpty() && retry) {
                    if (context.updateCache() || cacheUpdateNeeded) {
                        sendNewRequestWithExtractParam(
                            context,
                            midsForRequestToFS);
                    }
                }
                callback.completed(Collections.<Solution>emptyList());
            }
        }
        //CSON: MethodLength
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    private void sendNewRequestWithExtractParam(
        final FactsContext factsContext,
        final Set<String> mids)
    {
        ProxySession session = factsContext.session();
        HttpHost host = new HttpHost(
            factsContext.iexProxy().factsExtractURI().getHost(),
            factsContext.iexProxy().factsExtractURI().getPort());
        AsyncClient client = factsContext.
            iexProxy().factsExtractClient().adjust(
                session.context());
        try {
            CgiParams cgiParams = session.params();
            QueryConstructor cgiQueryConstructor =
                new QueryConstructor(factsContext.iexProxy().
                        factsExtractURI().getPath() + '?');
            for (Map.Entry<String, List<String>> entry: cgiParams.entrySet()) {
                if (!entry.getKey().equals(MID)) {
                    for (String value: entry.getValue()) {
                        cgiQueryConstructor.append(entry.getKey(), value);
                    }
                }
            }
            for (String mid: mids) {
                cgiQueryConstructor.append(MID, mid);
            }
            cgiQueryConstructor.append(EXTRACT, TRUE);
            String uri = cgiQueryConstructor.toString();
            BasicAsyncRequestProducerGenerator generator =
                new BasicAsyncRequestProducerGenerator(uri);
            String ticket = session.headers().getOrNull(YandexHeaders.TICKET);
            if (ticket != null) {
                generator.addHeader(YandexHeaders.TICKET, ticket);
            }
            String requestId =
                session.headers().getOrNull(YandexHeaders.X_REQUEST_ID);
            if (requestId != null) {
                generator.addHeader(YandexHeaders.X_REQUEST_ID, requestId);
            }
            String expBoxes =
                session.headers().getOrNull(YandexHeaders.X_ENABLED_BOXES);
            if (expBoxes != null) {
                generator.addHeader(YandexHeaders.X_ENABLED_BOXES, expBoxes);
            }
            log(factsContext, "silent request " + host + uri);
            client.execute(
                    host,
                    generator,
                    EmptyAsyncConsumerFactory.INSTANCE,
                    session.listener().
                            createContextGeneratorFor(client),
                    EmptyFutureCallback.INSTANCE
            );
        } catch (BadRequestException e) {
            e.printStackTrace();
        }
    }

    private class HostsCallback
        extends AbstractProxySessionCallback<List<QueueHostInfo>>
    {
        private final FactsContext factsContext;
        private final SearchResultsSynchronizer sync;
        private final long startTime;

        HostsCallback(
            final FactsContext factsContext,
            final SearchResultsSynchronizer sync,
            final long startTime)
        {
            super(factsContext.session());
            this.factsContext = factsContext;
            this.sync = sync;
            this.startTime = startTime;
        }

        @Override
        public synchronized void completed(final List<QueueHostInfo> hosts) {
            long timeTaken = System.currentTimeMillis() - startTime;
            factsContext.iexProxy()
                .producerClientCompleted(timeTaken);
            factsContext.iexProxy()
                .addProducerClientPureTime(timeTaken);
            factsContext.iexProxy().
                addToProducerClientTotalCounter(FactsHandler.this);
            if (hosts instanceof PrimitiveHandler) {
                log(factsContext, "got hosts from producerClient");
            } else {
                log(factsContext, "got hosts from searchmap");
                factsContext.iexProxy().
                    addToProducerClientSearchMapCounter(FactsHandler.this);
            }
            sync.gotProducerClientResult(hosts);
        }

        @Override
        public void failed(final Exception e) {
        }
    }

    // CSOFF: ParameterNumber
    private String getRequestString(
            final String uid,
            final String queueName,
            final Set<String> uniqueMids,
            final Set<String> factNames)
    {
        String requestString = "";
        try {
            QueryConstructor query = new QueryConstructor(
                "/search-iex-proxy?IO_PRIO=100");
            query.append("prefix", uid);
            query.append("service", queueName);
            query.append("json-type", "dollar");
            query.append(
                "get",
                "url,fact_name,fact_data,fact_mid,fact_is_coke_solution,"
                + "fact_received_date,fact_last_extracted_date");
            query.append(
                "text",
                StringUtils.join(uniqueMids, OR_DELIMITER, "fact_mid:(", ")"));
            query.append("sort", URL);
            query.append("collector", "sorted");
            query.append("asc", TRUE);
            query.append(DP, "const(" + EMPTY_FIELD + " empty_field)");
            addFactNameRestrictionsToRequest(
                query,
                factNames,
                iexProxy.config().factNamesToEraseFactData());
            requestString = query.toString();
        } catch (BadRequestException e) {
            e.printStackTrace();
        }
        return requestString;
    }
    // CSON: ParameterNumber

    private void addFactNameRestrictionsToRequest(
        final QueryConstructor query,
        final Set<String> names,
        final Set<String> namesToErase)
        throws BadRequestException
    {
        final String prefix = "equals(fact_name,";
        String equalsDp = "";
        String ifDp = "";
        if (!names.isEmpty()) {
            equalsDp =
                StringUtils.join(names, COMMA, prefix, " not_erase_data)");
            ifDp = "if(not_erase_data,fact_data,empty_field fact_data)";
        } else if (!namesToErase.isEmpty()) {
            equalsDp =
                StringUtils.join(namesToErase, COMMA, prefix, " erase_data)");
            ifDp = "if(erase_data,empty_field,fact_data fact_data)";
        }
        if (!ifDp.isEmpty()) {
            query.append(DP, equalsDp);
            query.append(DP, ifDp);
        }
    }

    private static void log(final AbstractContext context, final String text) {
        context.session.logger().info(FACTS_HANDLER + text);
    }

    private static void log(
        final AbstractContext context,
        final String text,
        final Exception e)
    {
        context.session.logger().log(Level.INFO, FACTS_HANDLER + text, e);
    }

    private static class SearchResultsSynchronizer {
        private final Map<HttpHost, SearchResult> hostToResult;
        private boolean done = false;
        private final FactsContext context;
        private final ParseFactsFromLuceneCallback luceneCallback;
        private final int backendCount;
        private long actualQueueId = SearchResult.DEFAULT_QUEUE_ID;
        private long bestQueueId = SearchResult.DEFAULT_QUEUE_ID;
        private HttpHost bestHost = null;

        SearchResultsSynchronizer(
            final ParseFactsFromLuceneCallback callback,
            final FactsContext context,
            final int backendCount)
        {
            hostToResult = new HashMap<>();
            this.luceneCallback = callback;
            this.context = context;
            this.backendCount = backendCount;
        }

        public FactsContext context() {
            return context;
        }

        public synchronized void gotSearchResult(
            final HttpHost host,
            final SearchResult result)
        {
            hostToResult.put(host, result);
            if (bestQueueId < result.zooQueueId() || bestHost == null) {
                bestQueueId = result.zooQueueId();
                bestHost = host;
            }
            gotResult();
        }

        public synchronized void searchFailed(final HttpHost host) {
            hostToResult.put(host, new BasicSearchResult());
            if (bestHost == null) {
                bestHost = host;
            }
            gotResult();
        }

        public void gotProducerClientResult(
            final List<QueueHostInfo> hosts)
        {
            for (QueueHostInfo info : hosts) {
                if (actualQueueId < info.queueId()) {
                    actualQueueId = info.queueId();
                }
            }
            log(context, "Producers actualQueueId: " + actualQueueId);
            gotResult();
        }

        public synchronized boolean done() {
            return this.done;
        }

        public synchronized void gotResult() {
            if (done) {
                return;
            }
            if (actualQueueId != SearchResult.DEFAULT_QUEUE_ID
                    && actualQueueId <= bestQueueId
                    && bestHost != null)
            {
                log(context, "host " + bestHost.getHostName()
                    + " is actual: expectedId: "
                    + actualQueueId + ", hostId: "
                    + bestQueueId + ", done");
                done = true;
                context.iexProxy().perDcLuceneAnswerSelected(
                    bestHost.getHostName());
                luceneCallback.completed(hostToResult.get(bestHost));
            } else if (hostToResult.size() == backendCount) {
                log(
                    context,
                    "got results from all " + backendCount + " hosts.");
                done = true;
                useBestResult();
            }
        }

        private void useBestResult() {
            log(
                context,
                "Using result from " + bestHost.getHostName()
                    + ", queueId: " + bestQueueId);
            context.iexProxy().perDcLuceneAnswerSelected(
                bestHost.getHostName());
            luceneCallback.completed(hostToResult.get(bestHost));
        }

        public synchronized void deadline() {
            if (!done) {
                if (hostToResult.size() > 0) {
                    log(context, "timeout happened, got results from "
                            + hostToResult.size() + " hosts");
                    done = true;
                    useBestResult();
                } else {
                    log(
                        context,
                        "timeout happened, no results, will wait for any");
                }
            }
        }
    }

    private static class RequestDeadlineTimerTask implements TimerTask {
        private final SearchResultsSynchronizer sync;
        private final long deadline;

        RequestDeadlineTimerTask(
            final SearchResultsSynchronizer sync,
            final long deadline)
        {
            this.sync = sync;
            this.deadline = deadline;
        }

        @Override
        public void run(final Timeout timeout) {
            final long time = System.currentTimeMillis();
            if (time >= deadline - MAX_DEADLINE_DRIFT) {
                sync.deadline();
            } else {
                log(
                    sync.context(),
                    "premature deadline timer wakeup, rescheduling");
                IexProxy iexProxy = sync.context().iexProxy();
                iexProxy.factsSporadicTimerWakeup();
//                ProducerClient producerClient = iexProxy.producerClient();
                DELAYED_EXECUTOR.newTimeout(
                    this,
                    deadline - time,
                    TimeUnit.MILLISECONDS);
//                producerClient.scheduleRetry(this, deadline - time);
            }
        }
    }
}
