package ru.yandex.iex.proxy;

import java.io.IOException;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.EmptyAsyncConsumer;
import ru.yandex.iex.proxy.xutils.spamreport.SendMailContext;
import ru.yandex.iex.proxy.xutils.spamreport.SpamReportSender;
import ru.yandex.parser.uri.CgiParams;


/**
 * Creates report for so-compains.
 * Requires cgi params as Post Process Action
 *
 * Additional params:
 * forward_type - stored as x-forward-type header, also reported as a stat
 *      adds to "others", unless mentioned in extrasettings.extra_stats
 * subject_prefix
 * prepend_folder_name - adds [%folder_name%] before subject.
 * do_not_send - do not create report (only increase stat)
 */
public class ForwardHandler implements HttpAsyncRequestHandler<Object> {
    private final static String SPAM = "_spam";
    private final static String HAM = "_ham";

    private final IexProxy iexProxy;

    public ForwardHandler(IexProxy iexProxy) {
        this.iexProxy = iexProxy;
    }

    @Override
    public HttpAsyncRequestConsumer<Object> processRequest(
        HttpRequest request,
        HttpContext context) {
        // Without valid object produces "Internal error processing request"
        return new EmptyAsyncConsumer<>(new Object());
    }

    @Override
    public void handle(
        Object v,
        HttpAsyncExchange exchange,
        HttpContext context)
        throws HttpException, IOException
    {
        ProxySession session =
            new BasicProxySession(iexProxy, exchange, context);

        String reportStat = session.params().getOrNull("forward_type");
        if (reportStat != null) {
            String folderType = session.params().getString("folder_type");
            if ("spam".equals(folderType)) {
                iexProxy.reportForward(reportStat + SPAM);
            } else {
                iexProxy.reportForward(reportStat + HAM);
            }
        }

        if (session.params().getBoolean("do_not_send", false)) {
            session.response(HttpStatus.SC_OK);
        } else {
            SpamReportSender.send(
                new ReportContext(iexProxy, session),
                new Callback(session),
                false,
                Collections.emptyList());
        }

    }

    private static class ReportContext implements SendMailContext {
        private final CgiParams params;
        private final IexProxy iexProxy;
        private final ProxySession proxySession;

        private ReportContext(IexProxy iexProxy, ProxySession proxySession) {
            this.params = proxySession.params();
            this.iexProxy = iexProxy;
            this.proxySession = proxySession;
        }

        @Override
        public long date() {
            return Long.parseLong(params.getOrNull("received_date"));
        }

        @Override
        public String from() {
            return params.getOrNull("email");
        }

        @Override
        public String to() {
            return params.getOrNull("user_email");
        }

        @Override
        public String subject() {
            StringBuilder sb = new StringBuilder();
            sb.append(params.getString("subject_prefix", ""));
            try {
                if (params.getBoolean("prepend_folder_name", false)) {
                    String folderName = params.getOrNull("folder_name");
                    if (folderName != null) {
                        sb.append('[');
                        sb.append(folderName);
                        sb.append(']');
                    }
                }
            } catch (BadRequestException ignored) {
                // means flag present, but value is not parsable
            }
            sb.append(params.getOrNull("subject"));
            return sb.toString();
        }

        @Override
        public String mid() {
            return params.getOrNull("mid");
        }

        @Override
        public String stid() {
            return params.getOrNull("stid");
        }

        @Override
        public long prefix() {
            return Long.parseLong(params.getOrNull("uid"));
        }

        @Override
        public String msgId() {
            return "uid:" + prefix() + "/stid:" + stid();
        }

        @Override
        public IexProxy iexProxy() {
            return iexProxy;
        }

        @Override
        public ProxySession session() {
            return proxySession;
        }

        @SuppressWarnings("unchecked")
        @Override
        public List<Integer> types() {
            String types = params.getOrNull("types");
            if (types == null) {
                return Collections.EMPTY_LIST;
            }
            return Arrays.stream(types.split(","))
                .map(Integer::parseInt)
                .collect(Collectors.toList());
        }

        @Override
        public List<Map.Entry<String, String>> additionalHeaders() {
            List<Map.Entry<String, String>> res = new LinkedList<>();
            String forwardType = params.getOrNull("forward_type");
            if (forwardType == null) {
                forwardType = "";
            }
            res.add(new AbstractMap.SimpleEntry<>(
                "x-forward-type", forwardType));

            String folderName = params.getOrNull("folder_name");
            if (folderName != null) {
                res.add(new AbstractMap.SimpleEntry<>(
                    "x-folder-name", folderName));
            }
            return res;
        }

        @Override
        public Map<?, ?> json() {
            return params;
        }
    }

    private static class Callback
        extends AbstractProxySessionCallback<Void>
    {
        Callback(final ProxySession session) {
            super(session);
        }

        @Override
        public void completed(final Void result) {
            session.logger().info("Mail forwarded");
            session.response(HttpStatus.SC_OK, "{}");
        }
    }

    public static List<String> forwardStats(
        final String[] forwardTypes) {
        List<String> stats = new ArrayList<>(forwardTypes.length * 2);
        for (final String type : forwardTypes) {
            stats.add(type + SPAM);
            stats.add(type + HAM);
        }
        return stats;
    }
}
