package ru.yandex.iex.proxy;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.iex.proxy.xutils.mailsender.limiters.IexForwardLimiter;
import ru.yandex.iex.proxy.xutils.mailsender.limiters.IexForwardLimiterFactory;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.CollectionParser;

public class IexForwarderConfigBuilder implements IexForwarderConfig {
    private List<IexForwardLimiterFactory> limiters;
    private String email;
    private boolean verboseLog;

    public IexForwarderConfigBuilder(final IexForwarderConfig config) {
        this.email(config.email());
        this.limiters(config.limiters());
        this.verboseLog(config.verboseLog());
    }

    public IexForwarderConfigBuilder(
        final IniConfig config,
        final IexForwarderConfig defaults)
        throws ConfigException
    {
        this(config, IniConfig.empty(), defaults);
    }

    public IexForwarderConfigBuilder(
        final IniConfig forwarderConfig,
        final IniConfig globalLimitersConfig,
        final IexForwarderConfig defaults)
        throws ConfigException
    {
        this.limiters = new ArrayList<>();

        Set<String> names =
            forwarderConfig.get(
                "limiters",
                new CollectionParser<>(String::trim, LinkedHashSet::new));

        if (names == null) {
            names = defaults.limiters().stream().map(
                (l) -> l.getClass().getSimpleName()).collect(
                Collectors.toSet());
        }

        for (String name: names) {
            this.limiters.add(
                loadLimiter(name, forwarderConfig, globalLimitersConfig));
        }

        this.email = forwarderConfig.getString("email", defaults.email());
        this.verboseLog =
            forwarderConfig.getBoolean("verbose-log", defaults.verboseLog());
    }

    private IexForwardLimiterFactory loadLimiter(
        final String name,
        final IniConfig forwarderConfig,
        final IniConfig globalLimitersConfig)
        throws ConfigException
    {
        final String className =
            IexForwardLimiter.class.getPackage().getName()
                + '.' + StringUtils.capitalize(name) + "LimiterFactory";
        try {
            Class<? extends IexForwardLimiterFactory> limiterClass =
                Class.forName(
                    className,
                    true,
                    this.getClass().getClassLoader())
                    .asSubclass(IexForwardLimiterFactory.class);
            IniConfig limiterConfig;
            try {
                Object property =
                    limiterClass.getDeclaredField("PROPERTY").get(null);
                limiterConfig =
                    forwarderConfig.sectionOrDefault(
                        property.toString(),
                        IniConfig.empty());
                IniConfig global =
                    globalLimitersConfig.sectionOrDefault(
                        property.toString(),
                        IniConfig.empty());

                for (String key: global.keys()) {
                    if (!limiterConfig.keys().contains(key)) {
                        limiterConfig.put(key, global.getString(key));
                    }
                }
            } catch (NoSuchFieldException nme) {
                limiterConfig = IniConfig.empty();
            }

            return limiterClass
                .getDeclaredConstructor(IniConfig.class)
                .newInstance(limiterConfig);
        } catch (
            ClassNotFoundException | IllegalAccessException
                | InvocationTargetException | NoSuchMethodException
                | InstantiationException e)
        {
            e.printStackTrace();
            throw new ConfigException(
                "Unable to construct ForwardLimiter " + name,
                e);
        }
    }

    @Override
    public List<IexForwardLimiterFactory> limiters() {
        return limiters;
    }

    public IexForwarderConfigBuilder limiters(
        final List<IexForwardLimiterFactory> limiters)
    {
        this.limiters = new ArrayList<>(limiters);
        return this;
    }

    public IexForwarderConfigBuilder email(final String email) {
        this.email = email;
        return this;
    }

    @Override
    public String email() {
        return email;
    }

    public ImmutableIexForwarderConfig build() throws ConfigException {
        return new ImmutableIexForwarderConfig(this);
    }

    @Override
    public boolean verboseLog() {
        return verboseLog;
    }

    public IexForwarderConfigBuilder verboseLog(
        final boolean verboseLog)
    {
        this.verboseLog = verboseLog;
        return this;
    }
}
