package ru.yandex.iex.proxy;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public class IexForwardersConfigBuilder implements IexForwardersConfig {
    private Map<String, List<IexForwarderConfigBuilder>> forwarders;

    public IexForwardersConfigBuilder(final IexForwardersConfig config) {
        this.forwarders(config.forwarders());
    }

    public IexForwardersConfigBuilder(
        final IniConfig config)
        throws ConfigException
    {
        this(config, IexForwardersConfigDefaults.INSTANCE);
    }

    public IexForwardersConfigBuilder(
        final IniConfig config,
        final IexForwardersConfig defaults)
        throws ConfigException
    {
        IniConfig limitersGlobalConfig =
            config.sections().remove("limiters_globals");
        if (limitersGlobalConfig == null) {
            limitersGlobalConfig = IniConfig.empty();
        }

        forwarders = new LinkedHashMap<>();
        if (config.sections().isEmpty()) {
            forwarders(defaults.forwarders());
        }

        for (Map.Entry<String, IniConfig> section
            : config.sections().entrySet())
        {
            if (section.getValue().keys().isEmpty()) {
                // so multiple forwarders for postprocessor
                List<IexForwarderConfigBuilder> subforwarders =
                    new ArrayList<>();

                for (Map.Entry<String, IniConfig> subsection
                    : section.getValue().sections().entrySet())
                {
                    subforwarders.add(
                        new IexForwarderConfigBuilder(
                            subsection.getValue(),
                            limitersGlobalConfig,
                            IexForwarderConfigDefaults.INSTANCE));
                }
                forwarders.put(section.getKey(), subforwarders);
            } else {
                forwarders.put(
                    section.getKey(),
                    Collections.singletonList(
                        new IexForwarderConfigBuilder(
                            section.getValue(),
                            limitersGlobalConfig,
                            IexForwarderConfigDefaults.INSTANCE)));
            }
        }
    }

    @Override
    public Map<String, List<IexForwarderConfigBuilder>> forwarders() {
        return forwarders;
    }

    public IexForwardersConfigBuilder forwarders(
        final Map<String, ? extends List<? extends IexForwarderConfig>>
            forwarders)
    {
        this.forwarders = new LinkedHashMap<>();
        for (Map.Entry<String, ? extends List<? extends IexForwarderConfig>>
            entry
            : forwarders.entrySet())
        {
            this.forwarders.put(
                entry.getKey(),
                entry.getValue().stream()
                    .map(IexForwarderConfigBuilder::new)
                    .collect(Collectors.toList()));
        }

        return this;
    }

    public ImmutableIexForwardersConfig build() throws ConfigException {
        return new ImmutableIexForwardersConfig(this);
    }
}
