package ru.yandex.iex.proxy;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.http.util.NotFoundException;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.search.document.mail.AttachInfo;
import ru.yandex.search.document.mail.FirstlineMailMetaInfo;
import ru.yandex.search.document.mail.MailMetaInfo;

public class IndexationContext<T> {
    public static final String IEX_PROXY1_YANDEX_RU = "iex.proxy1@yandex.ru";
    private static final int MILLIS = 1000;

    private final AbstractContext abstractContext;
    private final FirstlineMailMetaInfo meta;
    private FutureCallback<T> callback = null;
    private final String mid;
    private final String uid;
    private final Long uidLong;
    private final String stid;
    private final String tid;
    private Map<String, EntityOptions> entities;
    private Set<String> headersWanted;
    private Map<String, String> headers;
    private final Set<PostProcessAction> postActions;
    private final String email;
    private final String userEmail;
    private final String domain;
    private final String subject;
    private final long receivedDate;
    private final List<String> icsHids = new ArrayList<>();
    private final List<String> pdfHids = new ArrayList<>();
    private final List<String> pkpassHids = new ArrayList<>();
    private final List<String> dmarcHids = new ArrayList<>();
    private final List<String> allHids = new ArrayList<>();
    private final List<Integer> types;

    public IndexationContext(
        final AbstractContext abstractContext,
        final FirstlineMailMetaInfo meta,
        final FutureCallback<T> callback)
        throws JsonUnexpectedTokenException, NotFoundException
    {
        this.abstractContext = abstractContext;
        this.meta = meta;
        this.callback = callback;
        mid = meta.get("mid");
        uid = meta.get("uid");
        try {
            uidLong = Long.parseLong(uid);
        } catch (RuntimeException nfe) {
            throw new NotFoundException(nfe);
        }
        stid = meta.get("stid");
        Object tidObject = meta.get(MailMetaInfo.THREAD_ID_FIELD);
        if (tidObject != null) {
            tid = ValueUtils.asString(tidObject);
        } else {
            tid = null;
        }
        subject = ValueUtils.asString(meta.get(MailMetaInfo.HDR + MailMetaInfo.SUBJECT));
        final Logger logger = abstractContext.session().logger();
        this.receivedDate = parseReceivedDate(meta.get(MailMetaInfo.RECEIVED_DATE), logger);
        types = new ArrayList<>(meta.messageTypes());
        Collections.sort(types);
        email = getEmail(meta);
        userEmail = getUserEmail(meta);
        domain = extractDomain(email);
        headers = Collections.<String, String>emptyMap();
        final Set<String> typesHeaders = abstractContext.iexProxy().headersFromTypes(types, logger);
        final Set<String> domainHeaders = abstractContext.iexProxy().headersFromDomain(domain, logger);
        headersWanted = mergeSets(typesHeaders, domainHeaders);

        final Map<String, EntityOptions> e = abstractContext.iexProxy().entitiesFromEmail(this.email);
        final Map<String, EntityOptions> re = abstractContext.iexProxy().entitiesFromRcptEmail(this.userEmail);
        final Map<String, EntityOptions> ed = abstractContext.iexProxy().entitiesFromDomain(domain);
        final Map<String, EntityOptions> eru = abstractContext.iexProxy().entitiesFromRcptUid(uidLong);

        if (!e.isEmpty() || !ed.isEmpty() || !re.isEmpty() || !eru.isEmpty()) {
            final Map<String, EntityOptions> mergeOfEntites = new HashMap<>();
            mergeEntities(mergeOfEntites, abstractContext.iexProxy().entitiesFromTypes(types, logger));
            mergeEntities(mergeOfEntites, e);
            mergeEntities(mergeOfEntites, ed);
            mergeEntities(mergeOfEntites, re);
            mergeEntities(mergeOfEntites, eru);
            entities = Collections.unmodifiableMap(mergeOfEntites);
        } else {
            entities = abstractContext.iexProxy().entitiesFromTypes(types, logger);
        }
        boolean hasDmarc = entities.containsKey(IexProxy.DMARC_ENTITY);

        for (final AttachInfo attachment: meta.attachments()) {
            String file = attachment.filename();
            String hid = attachment.hid();
            if ("text/calendar".equals(attachment.contentType())) {
                icsHids.add(hid);
            } else if (file != null) {
                if (file.endsWith(".ics")) {
                    icsHids.add(hid);
                } else if (file.endsWith(".pdf")) {
                    pdfHids.add(hid);
                } else if (file.endsWith(".pkpass")) {
                    pkpassHids.add(hid);
                } else if (hasDmarc
                    && (file.endsWith(".xml.gz")
                        || file.endsWith(".xml")
                        || file.endsWith(".zip")))
                {
                    dmarcHids.add(hid);
                }
            }
            allHids.add(hid);
        }

        final Set<PostProcessAction> a = abstractContext.iexProxy().postProcessActionsFromEmail(this.email);
        final Set<PostProcessAction> ra = abstractContext.iexProxy().postProcessActionsFromRcptEmail(this.userEmail);
        final Set<PostProcessAction> epd = abstractContext.iexProxy().postProcessActionsFromDomain(domain);
        final Set<PostProcessAction> au = abstractContext.iexProxy().postProcessActionsFromRcptUid(uidLong);

        if (!a.isEmpty() || !epd.isEmpty() || !ra.isEmpty() || !au.isEmpty()) {
            final Set<PostProcessAction> mergeOfAction = new HashSet<>();
            mergeOfAction.addAll(abstractContext.iexProxy().postProcessActionsFromTypes(types, logger));
            mergeOfAction.addAll(a);
            mergeOfAction.addAll(ra);
            mergeOfAction.addAll(epd);
            mergeOfAction.addAll(au);
            postActions = Collections.unmodifiableSet(mergeOfAction);
        } else {
            postActions = abstractContext.iexProxy().postProcessActionsFromTypes(types, logger);
        }
    }

    public static <E> void mergeEntities(final Map<E, EntityOptions> target, final Map<E, EntityOptions> src)
    {
        if (!src.isEmpty()) {
            for (Map.Entry<E, EntityOptions> entry: src.entrySet()) {
                E name = entry.getKey();
                EntityOptions options = entry.getValue();
                EntityOptions oldOptions = target.get(name);
                if (oldOptions == null) {
                    target.put(name, options);
                } else {
                    target.put(name, EntityOptions.merge(oldOptions, options));
                }
            }
        }
    }

    public Set<String> mergeSets(final Set<String> a, final Set<String> b) {
        Set<String> result;
        if (!a.isEmpty() && !b.isEmpty()) {
            result = new HashSet<>();
            result.addAll(a);
            result.addAll(b);
        } else if (!a.isEmpty()) {
            result = a;
        } else {
            result = b;
        }
        return result;
    }

    public FutureCallback<T> callback() {
        return this.callback;
    }

    public void callback(final FutureCallback<T> callback) {
        this.callback = callback;
    }

    public long receivedDate() {
        return receivedDate;
    }

    public AbstractContext abstractContext() {
        return this.abstractContext;
    }

    public MailMetaInfo meta() {
        return this.meta;
    }

    public String mid() {
        return this.mid;
    }

    public String uid() {
        return uid;
    }

    public Long uidLong() {
        return uidLong;
    }

    public String stid() {
        return this.stid;
    }

    public String tid() {
        return this.tid;
    }

    public List<String> iscHids() {
        return icsHids;
    }

    public List<String> pkpassHids() {
        return pkpassHids;
    }

    public List<String> pdfHids() {
        return pdfHids;
    }

    public List<String> dmarcHids() {
        return dmarcHids;
    }

    public List<String> allHids() {
        return allHids;
    }

    public String entitiesString() {
        if (!entities.isEmpty()) {
            return joinEntities(entities.keySet());
        }
        return null;
    }

    public String firstline() {
        return meta.firstline();
    }

    public Map<String, String> headers() {
        return headers;
    }

    public void headers(final Map<String, String> headers) {
        this.headers = headers;
    }

    public Set<String> headersWanted() {
        return headersWanted;
    }

    public Map<String, EntityOptions> entities() {
        return entities;
    }

    public List<Integer> types() {
        return types;
    }

    public Set<PostProcessAction> postActions() {
        return postActions;
    }

    public String domain() {
        return domain;
    }

    public String email() {
        return email;
    }

    public String userEmail() {
        return userEmail;
    }

    public String subject() {
        return subject;
    }

    public String folderType() {
        String type = meta.get(MailIndexFields.FOLDER_TYPE);
        if (type == null) {
            type = "user";
        }
        return type;
    }

    private static long parseReceivedDate(
        final String rd,
        final Logger logger)
    {
        final long ts;
        final long currentTs = System.currentTimeMillis() / MILLIS;
        if (rd == null) {
            ts = currentTs;
        } else {
            long parsedTs;
            try {
                parsedTs = Long.parseLong(rd);
                if (parsedTs > currentTs) {
                    logger.warning("Message's received_date is in future, "
                        + "replacing with current time: " + rd);
                    parsedTs = currentTs;
                }
            } catch (Exception e) {
                parsedTs = currentTs;
                logger.log(
                    Level.SEVERE,
                    "Can't parse received_date, using current time",
                    e);
            }
            ts = parsedTs;
        }
        return ts;
    }

    public static String getEmail(final MailMetaInfo meta) {
        String from =
            meta.get(MailMetaInfo.HDR + MailMetaInfo.FROM + MailMetaInfo.EMAIL);
        if (from == null) {
            from = meta.get(MailMetaInfo.HDR + MailMetaInfo.FROM);
        }
        if (from == null) {
            // HACK: NPE fix, TODO: delete this :)
            return IEX_PROXY1_YANDEX_RU;
        }
        return from.trim();
    }

    public static String getUserEmail(final MailMetaInfo meta) {
        String from =
            meta.get(MailMetaInfo.HDR + MailMetaInfo.TO + MailMetaInfo.EMAIL);
        if (from == null) {
            from = meta.get(MailMetaInfo.HDR + MailMetaInfo.TO);
        }
        if (from == null) {
            // HACK: NPE fix, TODO: delete this :)
            return IEX_PROXY1_YANDEX_RU;
        }
        return from.trim();
    }

    public static String extractDomain(final String email) {
        final int at = email.indexOf('@');
        if (at == -1 || at == email.length() - 1) {
            return "";
        }
        int end = email.length();
        for (int i = at + 1; i < email.length(); i++) {
            char c = email.charAt(i);
            if (c == '>' || c == ' ') {
                end = i;
                break;
            }
        }
        return email.substring(at + 1, end);
    }

    private static String joinEntities(final Set<String> entities) {
        StringBuilder sb = new StringBuilder();
        for (String entity: entities) {
            switch (entity) {
                case IexProxy.VOID_ENTITY:
                case IexProxy.DMARC_ENTITY:
                    break;
                default:
                    if (sb.length() > 0) {
                        sb.append(',');
                    }
                    sb.append(entity);
                    break;
            }
        }
        return new String(sb);
    }
}
