package ru.yandex.iex.proxy;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Scanner;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.message.BasicHeader;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.dbfields.FilterSearchFields;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.AsyncStringConsumer;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumer;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.HttpAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncPostURIRequestProducerSupplier;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.email.types.MessageType;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.BooleanParser;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.search.request.util.SearchRequestText;

// https://wiki.yandex-team.ru/ps/documentation/iex-proxy/handles/#post/mark
public class MarkHandler implements HttpAsyncRequestHandler<String> {
    private final IexProxy iexProxy;

    public MarkHandler(IexProxy iexProxy) {
        this.iexProxy = iexProxy;
    }

    @Override
    public HttpAsyncRequestConsumer<String> processRequest(
        HttpRequest httpRequest,
        HttpContext httpContext) {
        return new AsyncStringConsumer();
    }


    /**
     Callbacks pass mids, some of them might be null
     Callback execution order:
     handle - parses request. Calls requestMid for queueId entries
        requestMid - sends request to search_backend
        MidParsingCallback - extracts mids from json
     \\\ GROUP BY uid: next operate on all mids for uid ///
     PhishingMarkingCallback - acquires lid. If marking not required, skips
        LabelSetter - labels message with lid
     SpamMarkingCallback - executes next one
     FilterSearchCallback - filters out seen and already in right folder
     SpamHamMovingCallback - sends Spam/Unspam order, if required
     \\\ Aggregates all SpamHamMovingCallbacks ///
     StatusReportingCallback - finishes session
     */
    @Override
    @SuppressWarnings("StringSplitter")
    public void handle(
        String body,
        HttpAsyncExchange exchange,
        HttpContext context)
        throws HttpException, IOException
    {
        ProxySession session = new BasicProxySession(iexProxy, exchange, context);

        if (!session.params().containsKey("spam") &&
            !session.params().containsKey("phishing")) {
            throw new BadRequestException(
                "Nothing to do. Add spam and/or phishing");
        }

        Context markContext = new Context(session, iexProxy);

        MultiFutureCallback<List<String>> aggregationCallback
            = new MultiFutureCallback<>(new StatusReportingCallback(session, markContext));
        Map<Long, MultiFutureCallback<String>> uidToCallbacks = new HashMap<>();

        Function<Long, MultiFutureCallback<String>> uidToCallbacksGenerator =
            uid -> new MultiFutureCallback<>(
                new PhishingMarkingCallback(
                    uid,
                    markContext,
                    new SpamMarkingCallback(
                        uid,
                        markContext,
                        aggregationCallback.newCallback())));

        if ("GET".equals(
                exchange.getRequest().getRequestLine().getMethod())) {
            // from query

            Long uid = session.params().getLong("uid");

            String mid = session.params().getOrNull("mid");
            String queueId = session.params().getOrNull("queueId");
            String msgId = session.params().getOrNull("msgId");
            int count = 0;
            if (mid != null) {
                ++count;
            }
            if (queueId != null) {
                ++count;
            }
            if (msgId != null) {
                ++count;
            }
            if (count != 1) {
                throw new BadRequestException(
                    "specify exactly one of: mid, queueId, msgId");
            }
            if (mid != null) {
                markContext.midResolved();
                uidToCallbacks.computeIfAbsent(
                    uid,
                    uidToCallbacksGenerator)
                    .newCallback()
                    .completed(mid);
            } else {
                requestMid(
                    uid,
                    queueId,
                    msgId,
                    markContext,
                    uidToCallbacks.computeIfAbsent(
                        uid,
                        uidToCallbacksGenerator).newCallback());
            }
        } else {
            // from body
            for (String rawRow : body.split("\n")) {
                ParsedRow row = ParsedRow.of(rawRow);
                if (row != null) {
                    FutureCallback<String> thisEntryCallback =
                            uidToCallbacks.computeIfAbsent(row.uid,
                                    uidToCallbacksGenerator).newCallback();
                    if (row.mid != null) {
                        markContext.midResolved();
                        thisEntryCallback.completed(row.mid);
                    } else {
                        requestMid(
                            row.uid,
                            row.queueId,
                            row.msgId,
                            markContext,
                            thisEntryCallback);
                    }
                }
            }
        }
        for (Map.Entry<Long, MultiFutureCallback<String>> entry
            : uidToCallbacks.entrySet()) {
            entry.getValue().done();
        }
        aggregationCallback.done();
    }

    private void requestMid(
        Long uid,
        String queueId,
        String msgId,
        Context markContext,
        FutureCallback<String> callback)
        throws BadRequestException
    {
        iexProxy.logger().fine(
            "Resolving queueId = " + queueId
            + " and msgId = " + msgId + " to mid");
        QueryConstructor query = new QueryConstructor("/search?", false);
        query.append("prefix", uid);
        query.append("service", iexProxy.config().factsIndexingQueueName());
        query.append("json-type", "dollar");
        query.append("get", "mid");
        String text;
        if (queueId == null) {
            text = "msg_id:\"" + SearchRequestText.quoteEscape(msgId) + '"';
        } else {
            text =
                "all_smtp_ids:\"" + SearchRequestText.quoteEscape(queueId)
                + '"';
        }
        query.append("text", text);

        ProxySession session = markContext.session();

        // for standalone can be replaced by
        // curl http://new-msearch-proxy.mail.yandex.net:8051/sequential/{query}
        AsyncClient client =
                iexProxy.searchClient().adjust(session.context());
        UniversalSearchProxyRequestContext requestContext =
            new PlainUniversalSearchProxyRequestContext(
                new User(
                    iexProxy.config().factsIndexingQueueName(),
                    new LongPrefix(uid)),
                null,
                true,
                client,
                session.logger());
        iexProxy.sequentialRequest(
            session,
            requestContext,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            iexProxy.almostAllFactsTimeout(),
            true,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener()
                .createContextGeneratorFor(client),
            new MidParsingCallback(markContext, callback));
    }

    private static class MidParsingCallback
        extends AbstractFilterFutureCallback<JsonObject, String> {
        private final Context markContext;

        public MidParsingCallback(
            Context markContext,
            FutureCallback<String> callback)
        {
            super(callback);
            this.markContext = markContext;
        }

        @Override
        public void completed(JsonObject response) {
            try {
                JsonList hits = response.get("hitsArray").asList();
                String mid;
                if (hits.isEmpty()) {
                    mid = null;
                } else {
                    mid = hits.get(0).get("mid").asStringOrNull();
                }
                if (mid != null) {
                    markContext.midResolved();
                }
                callback.completed(mid);
            } catch (JsonBadCastException e) {
                callback.completed(null);
            }
        }
    }

    private static class PhishingMarkingCallback
        extends AbstractFilterFutureCallback<List<String>, List<String>> {

        final Long uid;
        Context markContext;

        public PhishingMarkingCallback(
            Long uid,
            Context markContext,
            FutureCallback<List<String>> callback)
        {
            super(callback);
            this.uid = uid;
            this.markContext = markContext;
        }

        @Override
        public void completed(List<String> mids) {
            String phishingParam =
                markContext.session().params().getOrNull("phishing");
            if (phishingParam != null)  {
                QueryConstructor qc = new QueryConstructor("/labels/create?", false);
                try {
                    qc.append("uid", uid);
                    qc.append("name", MessageType.SOFP.typeNumber());
                    qc.append("type", "so");
                    qc.append("strict", 0);
                } catch (BadRequestException unreachable) {}
                makeRawMopsRequest(
                    uid,
                    qc.toString(),
                    markContext,
                    JsonAsyncTypesafeDomConsumerFactory.INSTANCE,
                    new LabelSetterCallback(
                        uid,
                        mids,
                        BooleanParser.INSTANCE.apply(phishingParam),
                        markContext,
                        callback));
            } else {
                callback.completed(mids);
            }
        }

        private static class LabelSetterCallback
            extends AbstractFilterFutureCallback<JsonObject, List<String>>
        {
            private final List<String> mids;
            private final Long uid;
            private final boolean makeSet;
            private final Context markContext;

            public LabelSetterCallback(
                    Long uid,
                    List<String> mids,
                    boolean makeSet,
                    Context markContext,
                    FutureCallback<? super List<String>> callback)
            {
                super(callback);
                this.mids = mids;
                this.uid = uid;
                this.markContext = markContext;
                this.makeSet = makeSet;
            }

            @Override
            public void completed(JsonObject lidResponse) {
                try {
                    String lid = lidResponse.get("lid").asString();
                    String h = (makeSet ? "label" : "unlabel") + "?lids=" + lid;
                    makeMopsRequest(markContext, uid, mids, h, callback);
                } catch (JsonBadCastException e) {
                    failed(e);
                }
            }
        }
    }

    private static class SpamMarkingCallback
            extends AbstractFilterFutureCallback<List<String>, List<String>> {
        private final Long uid;
        private final Context markContext;

        public SpamMarkingCallback(
                Long uid,
                Context markContext,
                FutureCallback<? super List<String>> callback) {
            super(callback);
            this.uid = uid;
            this.markContext = markContext;
        }

        @Override
        public void completed(List<String> mids) {
            String spamParam = markContext.session().params().getOrNull("spam");
            if (spamParam != null) {
                Boolean spamAction =
                    BooleanParser.INSTANCE.apply(spamParam);
                new FilterSearchCallback(
                    spamAction,
                    mids,
                    uid,
                    markContext,
                    new SpamHamMovingCallback(
                        spamAction,
                        callback))
                    .execute();
            } else {
                // Like if every message is filtered out
                callback.completed(Collections.emptyList());
            }
        }

        private class SpamHamMovingCallback
                extends AbstractFilterFutureCallback<List<String>, List<String>> {
            private final Boolean spamAction;

            public SpamHamMovingCallback(
                    Boolean spamAction,
                    FutureCallback<? super List<String>> callback) {
                super(callback);
                this.spamAction = spamAction;
            }

            @Override
            public void completed(List<String> mids) {
                if (spamAction) {
                    makeMopsRequest(
                        markContext,
                        uid,
                        mids,
                        "spam?",
                        callback);
                } else {
                    makeMopsRequest(
                        markContext,
                        uid,
                        mids,
                        "unspam?",
                        callback);
                }
            }
        }
    }

    // TODO: why errors in filtersearch are resulted in HTTP.OK?
    private static class FilterSearchCallback
            extends AbstractFilterFutureCallback<JsonObject, List<String>> {
        private final Boolean spamUnspamAction;
        private final List<String> mids;
        private final Long uid;
        private final Context markContext;

        @SuppressWarnings("FutureReturnValueIgnored")
        public void execute() {
            if (spamUnspamAction == null) {
                callback.completed(mids);
            } else {
                final AsyncClient httpClient;
                String tvmTicket;
                QueryConstructor qc;
                IexProxy iexProxy = markContext.iexProxy();
                if (BlackboxUserinfo.corp(uid)) {
                    httpClient = iexProxy.corpFilterSearchClient();
                    tvmTicket = iexProxy.corpFilterSearchTvm2Ticket();
                    qc = new QueryConstructor(iexProxy.corpFilterSearchUri());
                } else {
                    httpClient = iexProxy.filterSearchClient();
                    tvmTicket = iexProxy.filterSearchTvm2Ticket();
                    qc = new QueryConstructor(iexProxy.filterSearchUri());
                }
                try {
                    boolean haveMids = false;
                    qc.append("uid", uid);
                    for (String mid : mids) {
                        if (mid != null) {
                            haveMids = true;
                            qc.append("mids", mid);
                        }
                    }
                    if (!haveMids) {
                        callback.completed(mids);
                        return;
                    }
                    httpClient.execute(
                        new HeaderAsyncRequestProducerSupplier(
                            new AsyncGetURIRequestProducerSupplier(qc.toString()),
                            new BasicHeader(
                                YandexHeaders.X_YA_SERVICE_TICKET,
                                tvmTicket)),
                            JsonAsyncTypesafeDomConsumerFactory.OK,
                            markContext.session().listener()
                                .createContextGeneratorFor(httpClient),
                            this);
                } catch (URISyntaxException | BadRequestException e) {
                    callback.failed(e);
                }
            }
        }

        public FilterSearchCallback(
            Boolean spamUnspamAction,
            List<String> mids,
            Long uid,
            Context markContext,
            FutureCallback<List<String>> callback)
        {
            super(callback);
            this.spamUnspamAction = spamUnspamAction;
            this.mids = mids;
            this.uid = uid;
            this.markContext = markContext;
        }

        @Override
        public void completed(JsonObject fsResult) {
            try {
                List<String> midsToMove = new ArrayList<>();
                for (JsonObject mailInfo : fsResult.get("envelopes").asList()) {
                    JsonObject folder = mailInfo.get(FilterSearchFields.FOLDER);
                    String folderType = folder
                        .get(FilterSearchFields.TYPE)
                        .get(FilterSearchFields.TITLE).asString();
                    if (folderType.equals(FilterSearchFields.TYPE_SYSTEM)) {
                        folderType = folder
                            .get(FilterSearchFields.SYMBOLIC_NAME)
                            .get(FilterSearchFields.TITLE).asString();
                    }
                    boolean isInSpam = "spam".equals(folderType);
                    if (spamUnspamAction == isInSpam) {
                        markContext.iexProxy().logger().info(
                            "Skipping uid=" + uid
                                + " mid=" + mailInfo.get("mid").asString()
                                + " because already in correct folder");
                        continue;
                    }

                    JsonList lids = mailInfo.asMap()
                        .getListOrNull(FilterSearchFields.LABELS);
                    boolean seen = false;
                    if (lids != null &&
                        !markContext.session().params()
                            .getBoolean("ignore_seen", false)) {
                        JsonMap labelsInfo = mailInfo
                                .get(FilterSearchFields.LABELS_INFO).asMap();
                        for (JsonObject lid : lids) {
                            JsonMap labelInfo =
                                labelsInfo.getMap(lid.asString());
                            if (FilterSearchFields.TYPE_SYSTEM.equals(
                                    labelInfo
                                        .getMap(FilterSearchFields.TYPE)
                                        .getString(FilterSearchFields.TITLE)) &&
                                FilterSearchFields.SEEN_LABEL_TITLE.equals(
                                    labelInfo
                                        .getMap(FilterSearchFields.SYMBOLIC_NAME)
                                        .getString(FilterSearchFields.TITLE))) {
                                seen = true;
                                break;
                            }
                        }
                    }
                    if (!seen) {
                        midsToMove.add(mailInfo.get("mid").asString());
                    } else {
                        markContext.iexProxy().logger().info(
                            "Skipping uid=" + uid
                                + " mid=" + mailInfo.get("mid").asString()
                                + " because of seen_label");
                    }
                }
                callback.completed(midsToMove);
            } catch (JsonException | BadRequestException e) {
                markContext.session().logger().log(
                    Level.SEVERE,
                    "MakrHandler.FilterSearchCallback.completed failed",
                    e);
                failed(e);
            }
        }

        @Override
        public void failed(final Exception e) {
            markContext.session().logger().log(Level.SEVERE, "MakrHandler.FilterSearchCallback failed", e);
            callback.failed(e);
        }
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    private static <T> void makeRawMopsRequest(
        Long uid,
        String handle,
        Context markContext,
        HttpAsyncResponseConsumerFactory<T> consumerFactory,
        FutureCallback<T> callback)
    {
        final AsyncClient httpClient;
        final String tvmTicket;
        final HttpHost host;
        IexProxy iexProxy = markContext.iexProxy();
        if (BlackboxUserinfo.corp(uid)) {
            httpClient = iexProxy.corpMopsClient();
            host = iexProxy.config().corpMopsConfig().host();
            tvmTicket = iexProxy.corpMopsTvm2Ticket();
        } else {
            httpClient = iexProxy.mopsClient();
            host = iexProxy.config().mopsConfig().host();
            tvmTicket = iexProxy.mopsTvm2Ticket();
        }

        try {
            httpClient.execute(
                new HeaderAsyncRequestProducerSupplier(
                    new AsyncPostURIRequestProducerSupplier(
                        host + handle,
                        "",
                        ContentType.TEXT_PLAIN),
                    new BasicHeader(
                        YandexHeaders.X_YA_SERVICE_TICKET,
                        tvmTicket)),
                new StatusCheckAsyncResponseConsumerFactory<>(
                    HttpStatusPredicates.OK,
                    consumerFactory),
                markContext.session().listener()
                    .createContextGeneratorFor(httpClient),
                callback);
        } catch (URISyntaxException e) {
            callback.failed(e);
        }
    }

    private static void makeMopsRequest(
        Context markContext,
        long uid,
        List<String> mids,
        String handle,
        FutureCallback<? super List<String>> callback) {
        try {
            QueryConstructor qc =
                new QueryConstructor('/' + handle);
            qc.append("uid", uid);
            StringBuilder midsJoined = new StringBuilder();
            for (String mid : mids) {
                if (mid != null) {
                    if (midsJoined.length() > 0) {
                        midsJoined.append(',');
                    }
                    midsJoined.append(mid);
                }
            }
            if (midsJoined.length() == 0) {
                callback.completed(mids);
                return;
            }
            qc.append("mids", midsJoined.toString());
            makeRawMopsRequest(
                uid,
                qc.toString(),
                markContext,
                (producer, response) -> new EmptyAsyncConsumer<>(mids),
                callback);
        } catch (BadRequestException e) {
            callback.failed(e);
        }
    }

    private static class StatusReportingCallback
            extends AbstractProxySessionCallback<List<List<String>>> {
        private final Context markContext;

        public StatusReportingCallback(
            ProxySession session,
            Context markContext)
        {
            super(session);
            this.markContext = markContext;
        }

        @Override
        public void completed(List<List<String>> mids) {
            long fullyProcessed = mids.stream()
                .flatMap(List::stream)
                .filter(Objects::nonNull)
                .count();
            session.response(
                HttpStatus.SC_OK,
                String.valueOf(fullyProcessed)
                    + ' '
                    + markContext.resolved());
        }
    }


    /**
     * Stores common data and stats
     */
    static class Context {
        private final ProxySession session;
        private final IexProxy iexProxy;

        private final AtomicInteger resolved = new AtomicInteger(0);

        Context(ProxySession session, IexProxy iexProxy) {
            this.session = session;
            this.iexProxy = iexProxy;
        }

        void midResolved() {
            resolved.incrementAndGet();
        }

        public ProxySession session() {
            return session;
        }

        public IexProxy iexProxy() {
            return iexProxy;
        }

        public int resolved() {
            return resolved.get();
        }
    }

    private static class ParsedRow {
        String mid;
        Long uid;
        String queueId;
        String msgId;

        public static ParsedRow of(String row) throws BadRequestException {
            Scanner scan = new Scanner(row);
            if (!scan.hasNext()) {
                return null;
            }
            ParsedRow res = new ParsedRow();
            String id = scan.next();
            if (scan.hasNext()) {
                res.uid = Long.parseLong(id);
                id = scan.next();
            } else {
                throw new BadRequestException(
                    "Failed to parse row \"" + row + '"');
            }
            try {
                Long.parseLong(id);
                res.mid = id;
            } catch (NumberFormatException e) {
                int len = id.length();
                if (len > 2
                    && id.charAt(0) == '<'
                    && id.charAt(len - 1) == '>')
                {
                    res.msgId = id;
                } else {
                    res.queueId = id;
                }
            }
            return res;
        }
    }
}
