package ru.yandex.iex.proxy;

import java.nio.charset.CharacterCodingException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;
import java.util.logging.Level;

import org.apache.http.entity.ContentType;
import org.apache.http.message.BasicHeader;
import org.apache.http.nio.protocol.HttpAsyncRequestProducer;

import ru.yandex.function.StringBuilderProcessorAdapter;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncPostURIRequestProducerSupplier;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.parser.uri.PctEncoder;
import ru.yandex.parser.uri.PctEncodingRule;
import ru.yandex.search.document.mail.MailMetaInfo;

public class NotifyCallback extends AbstractCallback<Solution> {
    public static final String UNKNOWN_ERROR = "Unknown error";
    public static final String MESSAGE_TYPES = "message-types";
    public static final String HEADERS = "headers";
    public static final String RECEIVED_DATE = "received-date";
    public static final String LAG = "lag";
    public static final String FACTS = "facts";
    public static final String UID = "uid";
    public static final String COMSEP = ",";
    public static final String ESHOP = "eshop";
    public static final String ESHOP_BK = "eshop_bk";
    public static final String BK_CATEGORY_ID = "bk_category_id";
    public static final String TYPE = "type";
    public static final String DATA = "data";
    public static final String ENTITY = "entity";
    public static final String ORDER_ITEMS = "order_items";

    private static final String COKEMULATOR_ENTITIES_PREFIX = "_";
    private static final long SHARDS = 65534L;

    private static final ThreadLocal<StringBuilder> LOCAL_SB =
        new ThreadLocal<StringBuilder>() {
            @Override
            public StringBuilder initialValue() {
                return new StringBuilder();
            }
        };
    private static final String FROM = "from";
    private static final String DOMAIN = "domain";
    private static final String MID = "mid";
    private static final String MIDN = "&mid=";
    private static final String STID = "stid";
    private static final String TID = "tid";
    private static final String STIDN = "&stid=";
    private static final String CONTENT_LINE = "contentline";
    private static final double MILLIS = 1000.0D;
    private static final int CONTENT_LINE_MAX_LENGTH = 360;

    private long uid;

    public NotifyCallback(final IndexationContext<Solution> context) {
        super(context);
    }

    public void setUid(final long u) {
        uid = u;
    }

    public long getUid() {
        return uid;
    }

/*
DEBUG: 2016-02-29 18:26:30.029 +0300
    conv.cpp:100 timestamp=1456759590
    taxi=done;query=e=taxi&origin=proxy&time=1456759590
    &stid=89376.12555535.2382752547162006561062530993658&mid=2300000009735939396
    &uid=5819843
    &from=receipts.moscow@uber.com&domain=uber.com
    &subject=%D0%92%D0%B0%D1%88%D0%B0+%D1%83%D1%82%D1%80%D0%B5%D0%BD
    %D0%BD%D1%8F%D1%8F+%D0%BF%D0%BE%D0%B5%D0%B7%D0%B4%D0%BA%D0%B0+
    %D1%81+%D0%A3%D0%B1%D0%B5%D1%80%2C+%D0%BF%D0%BE%D0%BD%D0%B5%D0
    %B4%D0%B5%D0%BB%D1%8C%D0%BD%D0%B8%D0%BA;
    ENTITIES=[{"dist_cost":"86.1","duration_cost":"281.87",
    "car_delivery_cost":"50","cost":"417"},{"time_dep":"0.0.0 11:40:0",
    "cost":"417","city_dep":"Москва","time_arr":"0.0.0 12:16:0",
    "city_arr_geoid":"213","car":"uberX","dep":"Litovskiy b-r,
     1с2, Moskva, Russia, 117593","city_dep_geoid":"213",
    "arr":"Bolshaya Cheremushkinskaya ul., 11к2, Moskva, Russia, 117447",
    "duration":"00:35:14","city_arr":"Москва"}]
*/

    //CSOFF: ParameterNumber
    private void logIexDebugEntry(
        final IndexationContext<Solution> context,
        final Map<String, Object> result,
        final String namePrefix,
        final String entitiesPrefix)
        throws JsonUnexpectedTokenException
    {
        final int ts = (int) (System.currentTimeMillis() / MILLIS);
        final IexProxy iexProxy = context.abstractContext().iexProxy();
        final IexProxyLogger iexLogger = iexProxy.iexProxyLogger();
        if (result.isEmpty()) {
            context().abstractContext().session.logger().warning(
                "IEX solution is empty, nothing to log: " + result);
            return;
        }
        for (Map.Entry<String, Object> entry : result.entrySet()) {
            String entity = entry.getKey();
            if (namePrefix != null && !namePrefix.isEmpty()) {
                entity = namePrefix + entity;
            }
            final Object solution = entry.getValue();
            if (iexProxy.isIgnoreEmptySolution() && solution == null) {
                continue;
            }
            if (!iexProxy.journalingFact(entity)
                || !iexLogger.isLoggable(Level.INFO))
            {
                continue;
            }
            final StringBuilder sb = LOCAL_SB.get();
            sb.setLength(0);
            sb.append(context.abstractContext().sessionId());
            sb.append(".cpp:666 ");
            sb.append("timestamp=");
            sb.append(ts);
            sb.append(' ');
            sb.append(entity);
            sb.append("=done;");
            sb.append("query=e=");
            sb.append(entity);
            sb.append("&origin=proxy");
            sb.append("&time=");
            sb.append(ts);
            sb.append(STIDN);
            sb.append(context.stid());
            sb.append(MIDN);
            sb.append(context.mid());
            sb.append("&uid=");
            sb.append(context.abstractContext().uid());
            sb.append("&from=");
            sb.append(context.email());
            sb.append("&domain=");
            sb.append(context.domain());
            sb.append("&subject=");
            final String subject =
                context.meta().get(MailMetaInfo.HDR + MailMetaInfo.SUBJECT);
            try {
                PctEncoder encoder = new PctEncoder(PctEncodingRule.QUERY);
                encoder.process(subject.toCharArray());
                encoder.processWith(
                        new StringBuilderProcessorAdapter(sb));
            } catch (CharacterCodingException e) {
                context.abstractContext().session().logger().log(
                    Level.SEVERE,
                    "Subject url encoding error",
                    e);
                sb.append(context.meta().get(subject));
            }
            sb.append(';');
            sb.append("ENTITIES=[");
            JsonType.NORMAL.toStringBuilder(
                sb,
                produceJson(solution, entity, context, entitiesPrefix));
            sb.append(']');
            iexLogger.info(sb.toString());
        }
    }
    //CSON: ParameterNumber

    //CSOFF: ParameterNumber
    private Object produceJson(
        final Object solution,
        final String entity,
        final IndexationContext<Solution> context,
        final String entitiesPrefix) // not "facts" for some cases
        throws JsonUnexpectedTokenException
    {
        final Map<String, Object> jsonMap = new HashMap<>();
        jsonMap.put(MESSAGE_TYPES, messageTypes(context));
        jsonMap.put(RECEIVED_DATE, Long.toString(context.receivedDate()));
        jsonMap.put(LAG, getLag());
        jsonMap.put(FROM, context.email());
        jsonMap.put(DOMAIN, context.domain());
        jsonMap.put(MID, context.mid());
        jsonMap.put(UID, context.abstractContext().uid());
        jsonMap.put(STID, context.stid());
        if (context.tid() != null) {
            jsonMap.put(TID, context.tid());
        }
        jsonMap.put(HEADERS, context.headers());
        String reindex = "reindex";
        if (context.abstractContext() instanceof ChangeContext) {
            if (((ChangeContext) context.abstractContext()).
                    addReindexToAxis())
            {
                jsonMap.put(reindex, true);
            }
        }
        if (context.abstractContext() instanceof HandlersContext) {
            if (((HandlersContext) context.abstractContext()).
                    addReindexToAxis())
            {
                jsonMap.put(reindex, true);
            }
        }
        final String subject =
            context.meta().get(MailMetaInfo.HDR + MailMetaInfo.SUBJECT);
        final String displayName = context.meta().get(MailMetaInfo.HDR
                + MailMetaInfo.FROM + MailMetaInfo.DISPLAY_NAME);
        jsonMap.put("subject", subject.trim());
        jsonMap.put(ENTITY, entity.trim());
        if (displayName != null) { //NPE fix
            jsonMap.put("from-display-name", displayName.trim());
        }
        jsonMap.put("X-Proxy-Session-ID", context.abstractContext().sessionId());
        //TODO: make config section for the this purpose
        if (solution instanceof List) {
            jsonMap.put(entitiesPrefix, solution);
        } else {
            List<Object> redundantList = new ArrayList<>();
            if (solution != null) {
                redundantList.add(solution);
            }
            jsonMap.put(entitiesPrefix, redundantList);
        }
        return jsonMap;
    }
    //CSON: ParameterNumber

    private String getLag() {
        return String.valueOf(System.currentTimeMillis()
            - (long) (context.abstractContext().operationDate() * MILLIS));
    }

    private String messageTypes(final IndexationContext<Solution> context) {
        final StringBuilder sb = new StringBuilder();
        String sep = "";
        for (final Integer type : context.meta().messageTypes()) {
            sb.append(sep);
            sep = COMSEP;
            sb.append(type);
        }
        return new String(sb);
    }

    //CSOFF: ParameterNumber
    private StringBuilder axisSolutions(
        final StringBuilder sb,
        final Map<String, Object> solutions,
        final Map<String, Object> factHeader,
        final String namePrefix)
        throws JsonUnexpectedTokenException
    {
        final IexProxy iexProxy = context.abstractContext().iexProxy();
        StringBuilder sbEntites = new StringBuilder();
        String sep = "";
        for (Map.Entry<String, Object> entry : solutions.entrySet()) {
            String entity = entry.getKey();
            if (namePrefix != null && !namePrefix.isEmpty()) {
                entity = namePrefix + entity;
            }
            context().abstractContext().session.logger().info(
                "NotifyCallback: processing " + entity + " entity");
            sbEntites.append(sep);
            sep = COMSEP;
            sbEntites.append(entity);
            final Object solution = entry.getValue();
            if (iexProxy.isIgnoreEmptySolution() && solution == null) {
                context().abstractContext().session.logger().info(
                    "NotifyCallback: solution is null or ignored as empty");
                continue;
            }
            if (!iexProxy.axisEnabled(entity)) {
                context().abstractContext().session.logger().info(
                    "NotifyCallback: axis is not enabled for " + entity);
                continue;
            }
            factHeader.put(TYPE, entity);
            factHeader.put(
                DATA,
                produceJson(solution, entity, context, FACTS));
            if (sb.length() > 0) {
                sb.append('\n');
            }
            JsonType.NORMAL.toStringBuilder(sb, factHeader);
        }
        return sbEntites;
    }
    //CSON: ParameterNumber

    private boolean axisSolutions(final Solution result)
        throws JsonUnexpectedTokenException
    {
        final IexProxy iexProxy = context.abstractContext().iexProxy();
        final Map<String, Object> factHeader = new LinkedHashMap<>();
        factHeader.put(UID, context.abstractContext().uid());
        factHeader.put("extract_time", System.currentTimeMillis());
        factHeader.put("source", "mail");
        StringBuilder sb = new StringBuilder();
        StringBuilder entities = new StringBuilder("empty");
        if (result.cokemulatorSolutions() != null) {
            entities = axisSolutions(
                sb,
                result.cokemulatorSolutions(),
                factHeader,
                COKEMULATOR_ENTITIES_PREFIX);
        }
        if (!result.postActionSolutions().isEmpty()) {
            entities = axisSolutions(
                sb,
                result.postActionSolutions(),
                factHeader,
                "");
        }
        // add eshop_bk in case of bk_category_id is present in facts, IEX-1932
        if (!result.postActionSolutions().isEmpty()) {
            entities = addAxisEshopBkSolution(
                sb,
                result.postActionSolutions(),
                factHeader,
                "",
                entities);
        }
        if (sb.length() > 0) {
            final AsyncClient axisClient =
                iexProxy.axisClient().adjust(
                    context.abstractContext().session().context());
            Supplier<HttpAsyncRequestProducer> post =
                new AsyncPostURIRequestProducerSupplier(
                    iexProxy.axisURI(),
                    sb.toString(),
                    ContentType.APPLICATION_JSON);
            post = new HeaderAsyncRequestProducerSupplier(
                post,
                new BasicHeader(
                    YandexHeaders.SERVICE,
                    iexProxy.axisQueueName()),
                new BasicHeader(
                    YandexHeaders.ZOO_SHARD_ID,
                    Long.toString(Long.parseLong(context.abstractContext().
                            uid()) % SHARDS)));
            StringBuilder sbLog = new StringBuilder();
            sbLog.append("Preparing to send facts");
            sbLog.append(" to axis for uid: ");
            sbLog.append(context.abstractContext().uid());
            sbLog.append(", mid: ");
            sbLog.append(context.mid());
            sbLog.append(", axisURI = ");
            sbLog.append(iexProxy.axisURI());
            sbLog.append(", proxy = ");
            sbLog.append(iexProxy.config().axisConfig().proxy());
            sbLog.append(", types = ");
            sbLog.append(messageTypes(context));
            sbLog.append(", entities = ");
            sbLog.append(entities.toString());
            context().abstractContext().session.logger().info(
                new String(sbLog));
            axisClient.execute(
                post,
                AsyncStringConsumerFactory.NON_FATAL,
                context.abstractContext().session().listener()
                    .createContextGeneratorFor(axisClient),
                new AxisCallback(context, result));
            return true;
        } else {
            return false;
        }
    }

    //CSOFF: ParameterNumber
    private StringBuilder addAxisEshopBkSolution(
        final StringBuilder sb,
        final Map<String, Object> solutions,
        final Map<String, Object> factHeader,
        final String namePrefix,
        final StringBuilder entities)
        throws JsonUnexpectedTokenException
    {
        final IexProxy iexProxy = context.abstractContext().iexProxy();
        String entity = ESHOP_BK;
        if (iexProxy.axisEnabled(entity) && solutions.containsKey(ESHOP)) {
            Map<?, ?> eshopSolution =
                ValueUtils.asMapOrNull(solutions.get(ESHOP));
            if (eshopSolution.containsKey(ORDER_ITEMS)) {
                List<?> orderItems =
                    ValueUtils.asListOrNull(eshopSolution.get(ORDER_ITEMS));
                List<String> bkIdsList = new ArrayList<String>();
                for (Object orderItemEntry : orderItems) {
                    Map<?, ?> itemMap =
                        ValueUtils.asMapOrNull(orderItemEntry);
                    if (
                        itemMap != null
                            && itemMap.containsKey(BK_CATEGORY_ID))
                    {
                        String bkId = ValueUtils.
                            asString(itemMap.get(BK_CATEGORY_ID));
                        if (!bkId.isEmpty()) {
                            bkIdsList.add(bkId);
                        }
                    }
                }
                if (!bkIdsList.isEmpty()) {
                    String yuids = ValueUtils.asStringOrNull(
                            eshopSolution.get("yuids"));
                    if (yuids == null) {
                        return entities;
                    }
                    entities.append(COMSEP).append(entity);
                    Set<String> ids = new HashSet<>();
                    for (String idsEntry : bkIdsList) {
                        for (String id : idsEntry.split(COMSEP)) {
                            ids.add(id);
                        }
                    }
                    final StringBuilder idsSb = new StringBuilder();
                    String sep = "";
                    for (final String id : ids) {
                        idsSb.append(sep).append(id);
                        sep = COMSEP;
                    }
                    Map<String, Object> entityMap = new HashMap<>();
                    entityMap.put(ENTITY, entity);
                    entityMap.put(
                                  "passport_uid",
                                  context.abstractContext().uid());
                    entityMap.put("categories", new String(idsSb));
                    entityMap.put("yandex_yuids", yuids);
                    entityMap.put(
                                  "timestamp",
                                  Long.toString(context.receivedDate()));
                    factHeader.put(TYPE, entity);
                    factHeader.put(DATA, entityMap);
                    if (sb.length() > 0) {
                        sb.append('\n');
                    }
                    JsonType.NORMAL.toStringBuilder(sb, factHeader);
                    Map<String, Object> eshopBkMapToLog = new HashMap<>();
                    eshopBkMapToLog.put(
                            ESHOP_BK,
                            JsonType.NORMAL.toString(factHeader));
                    logIexDebugEntry(context, eshopBkMapToLog, "", "entities");
                }
            }
        }
        return entities;
    }
    //CSON: ParameterNumber

    public void logSolutions(final Solution result)
        throws JsonUnexpectedTokenException
    {
        logIexDebugEntry(
            context,
            result.cokemulatorSolutions(),
            COKEMULATOR_ENTITIES_PREFIX,
            FACTS);
        if (result.postActionSolutions().isEmpty()) {
            context().abstractContext().session.logger().warning(
                "PostAtcions IEX solution map is empty, nothing to log");
        } else {
            logIexDebugEntry(context, result.postActionSolutions(), "", FACTS);
        }
    }

    @Override
    public void completed(final Solution result) {
        // Input: response of cokemulator-iexlib
        // Output: request to Msal
        try {
            if (axisSolutions(result)) {
                //will logSolutions in axisCallback
                return;
            } else {
                logSolutions(result);
                context().callback().completed(result);
            }
        } catch (Exception e) {
            context().abstractContext()
                    .session().logger()
                    .log(Level.WARNING, UNKNOWN_ERROR, e);
            context().callback().failed(e);
        }
    }

    public void cancelled() {
        context().abstractContext().session().logger()
            .warning("request cancelled");
        context().callback().cancelled();
    }

    public void failed(final Exception e) {
        context().abstractContext().session().logger()
            .warning("request failed");
        context().callback().failed(e);
    }

    private class AxisCallback
        extends AbstractProxySessionCallback<String>
    {
        private final IndexationContext<Solution> context;
        private final Solution result;
        private final long startTime;

        AxisCallback(
            final IndexationContext<Solution> context,
            final Solution result)
        {
            super(context.abstractContext().session());
            this.context = context;
            this.result = result;
            this.startTime = System.currentTimeMillis();
        }

        @Override
        public void failed(final Exception e) {
            final long time = System.currentTimeMillis() - startTime;
            context().abstractContext().iexProxy().axisStoreTime(time);
            super.failed(e);
        }

        @Override
        public void completed(final String result) {
            final long time = System.currentTimeMillis() - startTime;
            context().abstractContext().iexProxy().axisStoreTime(time);
            context.abstractContext().session().logger().info(
                "Axis result for mid "
                + context.mid() + " : " + result);
            try {
                logSolutions(this.result);
                context().callback().completed(this.result);
            } catch (JsonUnexpectedTokenException e) {
                context().abstractContext()
                        .session().logger()
                        .log(Level.WARNING, UNKNOWN_ERROR, e);
                context().callback().failed(e);
            }
        }
    }
}
