package ru.yandex.iex.proxy;

import java.util.Iterator;
import java.util.List;
import java.util.logging.Level;

import org.apache.http.HttpRequest;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.function.StringBuilderProcessorAdapter;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.HttpExceptionConverter;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.parser.uri.QueryParameter;
import ru.yandex.parser.uri.QueryParser;

public class OffsetCallback implements FutureCallback<CacheSentSolutions> {
    //private static final long MILLIS = 1000L;

    private final ChangeContext context;
    private final List<?> changed;
    private final int nextOffset;
    private final int batchSize;

    @SuppressWarnings("unused")
    public OffsetCallback(final ChangeContext context)
        throws BadRequestException, JsonUnexpectedTokenException
    {
        this(context, false);
    }

    public OffsetCallback(final ChangeContext context, final boolean useBatches)
        throws BadRequestException, JsonUnexpectedTokenException
    {
        this.context = context;
        this.changed = ValueUtils.asList(context.json().get("changed"));
        if (useBatches) {
            int request = this.changed.size() - context.offset();
            if (request <= 0) {
                throw new BadRequestException("Malformed request: "
                        + context.humanReadableJson()
                        + " with offset " + context.offset()
                        + " because changed size is " + this.changed.size());
            }
            int maxBatch =
                    context.iexProxy().config()
                            .filterSearchConfig().batchSize();
            if (request <= maxBatch) {
                this.batchSize = request;
                this.nextOffset = 0;
            } else {
                this.batchSize = maxBatch;
                this.nextOffset = context.offset() + this.batchSize;
            }
        } else {
            this.nextOffset = 0;
            this.batchSize = 1;
        }

        if (this.nextOffset != 0 || this.batchSize > 1) {
            HttpRequest request1 = context.session().request();
            request1.removeHeaders("ZooQueue");
            request1.removeHeaders("ZooQueueId");
            request1.removeHeaders("ZooShardId");
        }
    }

    public ChangeContext context() {
        return this.context;
    }

    public List<?> changed() {
        return this.changed;
    }

    public int batchSize() {
        return this.batchSize;
    }

    @Override
    public void cancelled() {
        this.context.session()
                .logger()
                .warning("Request cancelled: "
                        + this.context.session().listener().details());
        context.session().response(YandexHttpStatus.SC_CLIENT_CLOSED_REQUEST);
    }

    @Override
    public void failed(final Exception e) {
        this.context.session()
                .logger()
                .log(Level.WARNING, "Failed to process: "
                        + this.context.humanReadableJson()
                        + '\n' + this.context.session().listener().details()
                        + " because of exception", e);
        this.context.session()
                .handleException(HttpExceptionConverter.toHttpException(e));
    }

    public void logLag() {
        long now = System.currentTimeMillis();
        long operationDate = context.operationDateMillis();
        long lag = now - operationDate;
        context.session().logger().info("PG processing lag: " + lag + " ms");
        if (context.isOnline() && context.hasTransferTimestamp()) {
            context.session().logger().info("Online lag: " + lag + " ms");
            context.iexProxy().processingLag(operationDate, now);
        }
    }

    @Override
    public void completed(final CacheSentSolutions result) {
        if (this.nextOffset == 0) {
            this.logLag();
            this.context.session().response(YandexHttpStatus.SC_OK);
        } else {
            this.logLag();
            StringBuilder uri = new StringBuilder();
            StringBuilderProcessorAdapter adapter =
                    new StringBuilderProcessorAdapter(uri);
            this.context.session()
                    .uri().path(null).processWith(adapter);
            uri.append('?');
            boolean offsetSet = false;
            boolean first = true;
            QueryParser parser =
                    this.context.session().uri().queryParser(null);
            Iterator<?> location = parser.iterator();
            while (location.hasNext()) {
                QueryParameter param = (QueryParameter) location.next();
                if (first) {
                    first = false;
                } else {
                    uri.append('&');
                }
                uri.append(param.name());
                if (param.name().equals("offset")) {
                    offsetSet = true;
                    uri.append('=');
                    uri.append(this.nextOffset);
                } else if (param.value() != QueryParser.NULL) {
                    uri.append('=');
                    param.value().processWith(adapter);
                }
            }
            if (!offsetSet) {
                if (!first) {
                    uri.append('&');
                }
                uri.append("offset=");
                uri.append(this.nextOffset);
            }
            if (context.session().uri().hasFragment()) {
                uri.append('#');
                context.session().uri().fragment(null).processWith(adapter);
            }
            String location1 = new String(uri);
            this.context.session()
                    .logger().info("Redirecting "
                    + context.session().request().getRequestLine().getUri()
                    + " to " + location1);
            context.session().getResponse().addHeader("Location", location1);
            context.session().response(YandexHttpStatus.SC_TEMPORARY_REDIRECT);
        }
    }
}

