package ru.yandex.iex.proxy;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncPostURIRequestProducerSupplier;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.mail.search.MailSearchParams;
import ru.yandex.parser.uri.QueryConstructor;

public class PeopleUrlsHandler implements HttpAsyncRequestHandler<JsonObject> {
    private static final String TYPES = "types";

    private final IexProxy proxy;

    public PeopleUrlsHandler(final IexProxy proxy) throws IOException {
        this.proxy = proxy;
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException
    {
        if (!(request instanceof HttpEntityEnclosingRequest)) {
            throw new BadRequestException("Payload expected");
        }
        return new JsonAsyncTypesafeDomConsumer(
            ((HttpEntityEnclosingRequest) request).getEntity(),
            StringCollectorsFactory.INSTANCE,
            BasicContainerFactory.INSTANCE);
    }

    @Override
    public void handle(
        final JsonObject payload,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        ProxySession session = new BasicProxySession(proxy, exchange, context);
        String stid = session.params().getString(MailSearchParams.STID);
        Long uid = session.params().getLong(MailSearchParams.UID);

        if (!BlackboxUserinfo.corp(uid)) {
            BasicAsyncRequestProducerGenerator requestGenerator =
                new BasicAsyncRequestProducerGenerator(
                    "/mail/handler?json-type=dollar&ultra-fast-mode&stid="
                    + stid);
            requestGenerator.addHeader(
                YandexHeaders.X_YA_SERVICE_TICKET,
                proxy.tikaiteTvm2Ticket());
            requestGenerator.addHeader(
                YandexHeaders.X_SRW_SERVICE_TICKET,
                proxy.unistorageTvm2Ticket());
            AsyncClient client = proxy.tikaiteClient().adjust(context);
            client.execute(
                proxy.tikaiteHost(),
                requestGenerator,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                session.listener().createContextGeneratorFor(client),
                new TikaiteCallback(session, uid, stid, context));
        } else {
            session.response(
                HttpStatus.SC_OK,
                JsonType.NORMAL.toString(Collections.emptyMap()));
        }
    }

    private final class TikaiteCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final Logger logger;
        private final Long uid;
        private final String mid;
        private final String stid;
        private final String types;
        private final HttpContext context;

        private TikaiteCallback(
            final ProxySession session,
            final Long uid,
            final String stid,
            final HttpContext context)
            throws BadRequestException
        {
            super(session);
            this.logger = session.logger();
            this.uid = uid;
            this.stid = stid;
            this.mid = session.params().getString(MailSearchParams.MID);
            this.types = session.params().getString(TYPES, "");
            this.context = context;
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                // https://st.yandex-team.ru/SMARTOBJECT-3
                if (proxy.config().smartObjectConfig() != null
                    && uid % 100 < proxy.config().smartObjectConfig().passPercent())
                {
                    JsonMap smoMap = new JsonMap(BasicContainerFactory.INSTANCE);
                    smoMap.put("uid", new JsonLong(uid));
                    smoMap.put("stid", new JsonString(stid));
                    smoMap.put("mid", new JsonString(mid));
                    smoMap.put("types", new JsonString(types));
                    smoMap.put("tikaite", result);
                    session.logger().info("Smartobject called " + JsonType.NORMAL.toString(smoMap));
                    AsyncClient client = proxy.smartobjectClient().adjust(context);
                    SmartObjectCallback smoCallback = new SmartObjectCallback(session.logger());

                    try {
                        client.execute(
                            new AsyncPostURIRequestProducerSupplier(
                                proxy.config().smartObjectConfig().uri(),
                                new StringEntity(
                                    JsonType.NORMAL.toString(smoMap),
                                    ContentType.APPLICATION_JSON
                                        .withCharset(StandardCharsets.UTF_8))
                            ),
                            JsonAsyncTypesafeDomConsumerFactory.OK,
                            session.listener().createContextGeneratorFor(client),
                            smoCallback);
                    } catch (IOException ioe) {
                        smoCallback.failed(ioe);
                    }
                }


                Set<String> capturedUrls = new HashSet<>();
                boolean isFirstHtmlAndNotAttach = true;
                for (JsonObject docObj: result.get("docs").asList()) {
                    JsonMap doc = docObj.asMap();
                    String urlsStr = doc.getOrNull("x_urls");
                    if (urlsStr == null || urlsStr.isEmpty()) {
                        continue;
                    }

                    String hid = doc.getString(MailIndexFields.HID);
                    String[] urls = urlsStr.trim().split("\n");
                    if (isFirstHtmlAndNotAttach
                           && (doc.getString("mimetype").equals("text/html")
                           || doc.getString("mimetype").equals("text/plain"))
                           && doc.getString("attachsize_b", null) == null
                    )
                    {
                        Collections.addAll(capturedUrls, urls);
                        isFirstHtmlAndNotAttach = false;
                    }
                }

                logger.info("captured urls count " + capturedUrls.size());
                for (String captUrl : capturedUrls) {
                    logger.info("captured url: " + captUrl);
                }
                Set<String> urlsToRca = getValidToRcaUrls(capturedUrls, logger);
                proxy.addUrlsToRcaAndTotalUrlsCount(
                        urlsToRca.size(),
                        capturedUrls.size());
                logger.info("UrlsToRca count " + urlsToRca.size());
                if (urlsToRca.size() > 0) {
                    HttpHost host = new HttpHost(
                        proxy.rcaURI().getHost(),
                        proxy.rcaURI().getPort());
                    final AsyncClient rcaClient =
                        proxy.rcaClient().adjust(context);
                    try {
                        final MultiFutureCallback<RequestWithJsonResult>
                            multiCallback = new MultiFutureCallback<>(
                                new MergeRcaResultsMultiCallback(session));
                        final List<RequestWithRcaResultCallback>
                            subCallbacks =
                                new ArrayList<>(urlsToRca.size());
                        for (final String url : urlsToRca) {
                            subCallbacks.add(
                                new RequestWithRcaResultCallback(
                                    url,
                                    multiCallback.newCallback(),
                                    proxy));
                        }
                        multiCallback.done();
                        for (RequestWithRcaResultCallback
                                subCallback : subCallbacks)
                        {
                            QueryConstructor queryConstructor =
                                new QueryConstructor(proxy.rcaURI().toString());
                            queryConstructor.append(
                                "url",
                                subCallback.getRequest());
                            String request = queryConstructor.toString();
                            BasicAsyncRequestProducerGenerator generator =
                                new BasicAsyncRequestProducerGenerator(request);

                            rcaClient.execute(
                                host,
                                generator,
                                JsonAsyncTypesafeDomConsumerFactory.OK,
                                session.listener()
                                    .createContextGeneratorFor(rcaClient),
                                subCallback);
                        }
                    } catch (Exception e) {
                        logger.log(
                                Level.WARNING,
                                "Rca sending request failed with ", e);
                        failed(e);
                    }
                } else {
                    session.response(
                            HttpStatus.SC_OK,
                            JsonType.NORMAL.toString(Collections.emptyMap()));
                }
            } catch (JsonException je) {
                failed(je);
            }
        }
    }

    private Set<String> getValidToRcaUrls(
        final Set<String> urls,
        final Logger logger)
    {
        // don't filter, go to RCA for all urls
        //  return urls;
        Set<String> validUrls = new HashSet<String>();
        for (String url : urls) {
            if (proxy.isValidWhitelistUrl(url)) {
                validUrls.add(url);
            }
        }
        return validUrls;
    }

    private static class MergeRcaResultsMultiCallback implements
          FutureCallback<List<RequestWithJsonResult>>
    {
        private static final String LOG_MULTI_PREFIX =
            "PeopleUrlHandler.MergeRcaResultsCallback: ";
        private final ProxySession session;
        private final List<JsonMap> response = new ArrayList<>();

        MergeRcaResultsMultiCallback(final ProxySession session) {
            this.session = session;
        }

        @Override
        public void completed(final
                List<RequestWithJsonResult> listResults)
        {
            try {
                for (RequestWithJsonResult requestWithResult : listResults) {
                    JsonMap result = requestWithResult.getResult().asMap();
                    result.put(
                        "request_url",
                        new JsonString(requestWithResult.getRequest()));
                    response.add(result);
                }
            } catch (JsonBadCastException e) {
                session.logger().info("JsonBadCastException "
                    + "in constructing urls response");
            }
            session.response(
                    HttpStatus.SC_OK,
                    JsonType.NORMAL.toString(response));
        }

        @Override
        public void cancelled() {
            session.logger().
                info(LOG_MULTI_PREFIX + "cancelled");
            session.response(
                    HttpStatus.SC_OK,
                    JsonType.NORMAL.toString(Collections.emptyMap()));
        }

        @Override
        public void failed(final Exception e) {
            session.logger().
                info(LOG_MULTI_PREFIX + "failed" + e.getMessage());
            session.response(
                    HttpStatus.SC_OK,
                    JsonType.NORMAL.toString(Collections.emptyMap()));
        }
    }

    private static class SmartObjectCallback implements FutureCallback<JsonObject> {
        private final Logger logger;

        public SmartObjectCallback(final Logger logger) {
            this.logger = logger;
        }

        @Override
        public void completed(final JsonObject o) {
            logger.warning("Smartobject completed  " +JsonType.NORMAL.toString(o));
        }

        @Override
        public void failed(final Exception e) {
            logger.warning("Smartobject failed " + e.getMessage());
        }

        @Override
        public void cancelled() {
            logger.warning("Smartobject cancelled");
        }
    }
}
