package ru.yandex.iex.proxy;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.message.BasicHeader;

import ru.yandex.client.wmi.Labels;
import ru.yandex.client.wmi.LabelsConsumerFactory;
import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.http.config.ImmutableURIConfig;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.HttpExceptionConverter;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncPostURIRequestProducerSupplier;
import ru.yandex.iex.proxy.move.SingleMessageReporter;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.document.mail.FirstlineMailMetaInfo;

public enum RealUpdateHandler implements ChangeHandler {
    INSTANCE;

    private static final String WMI_SUFFIX = "caller=msearch";
    private static final String MDB = "mdb";
    private static final String PG = "pg";
    private static final String UID = "uid";
    private static final String MID = "mid";
    private static final String STID = "stid";
    private static final String FOLDER = "folder";
    private static final String OPERATION_DATE = "operation_date";
    private static final String SEEN = "seen";
    private static final String ACTION = "action";
    private static final String LABEL = "label";
    private static final String CHANGED = "changed";
    private static final String LIDS_ADD = "lids_add";
    private static final String LIDS_DEL = "lids_del";
    private static final String SPEC_PREFIX = "_so_";

    @Override
    public void handle(final ChangeContext context) {
        IexProxy iexProxy = context.iexProxy();
        if (iexProxy.trustedSoCoworkers(Long.parseLong(context.uid()))) {
            context.session.logger().info("RealUpdateHandler: trusted user: uid=" + context.prefix());
            checkLabelsChange(context);
        } else {
            context.session.response(HttpStatus.SC_OK);
        }
    }

    private void checkLabelsChange(final ChangeContext context) {
        IexProxy iexProxy = context.iexProxy();
        ImmutableURIConfig labelsConfig;
        AsyncClient labelsClient;
        String tvmTicket;
        if (context.corp()) {
            labelsConfig = iexProxy.config().corpLabelsConfig();
            labelsClient = iexProxy.corpLabelsClient();
            tvmTicket = context.iexProxy().corpLabelsTvm2Ticket();
        } else {
            labelsConfig = iexProxy.config().labelsConfig();
            labelsClient = iexProxy.labelsClient();
            tvmTicket = context.iexProxy().labelsTvm2Ticket();
        }
        try {
            QueryConstructor query = new QueryConstructor(
                new StringBuilder(labelsConfig.uri().toASCIIString())
                    .append(labelsConfig.firstCgiSeparator())
                    .append(WMI_SUFFIX));
            query.append(MDB, PG);
            query.append(UID, context.prefix());
            String uri = query.toString();
            context.session().logger().info("Send labels request: " + uri);
            labelsClient.execute(
                new HeaderAsyncRequestProducerSupplier(
                    new AsyncGetURIRequestProducerSupplier(uri),
                    new BasicHeader(YandexHeaders.X_YA_SERVICE_TICKET, tvmTicket)),
                LabelsConsumerFactory.OK,
                context.session().listener().createContextGeneratorFor(labelsClient),
                new LabelsCallback(context));
        } catch (BadRequestException | URISyntaxException e) {
            context.session().logger().log(Level.SEVERE, "Labels request building error", e);
            context.session().handleException(HttpExceptionConverter.toHttpException(e));
        }
    }

    private static class LabelsCallback extends AbstractProxySessionCallback<Labels>
    {
        private final ChangeContext context;

        LabelsCallback(final ChangeContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final Labels labels) {
            Map<String, String> lids = new HashMap<>();
            for (Map.Entry<String, List<String>> label: labels.lids().entrySet()) {
                if (label.getKey().startsWith(SPEC_PREFIX)) {
                    for (String lid: label.getValue()) {
                        lids.put(lid, label.getKey());
                    }
                }
            }
            session.logger().info("LabelsCallback: Labels request completed, special lids: " + lids.toString());
            try {
                Map<String, List<Long>> addedLabelsMessages = getLabelsChangedMessages(lids, LIDS_ADD);
                Map<String, List<Long>> deletedLabelsMessages = getLabelsChangedMessages(lids, LIDS_DEL);
                if (addedLabelsMessages.isEmpty() && deletedLabelsMessages.isEmpty()) {
                    session.logger().info(
                        "LabelsCallback: Unable to find messages which have labeled or unlabeled with "
                        + "special labels");
                    session.response(HttpStatus.SC_OK);
                    return;
                } else {
                    session.logger().info(
                        "Labels: added: " + addedLabelsMessages.size() + ", deleted: " + deletedLabelsMessages.size());
                }
                try {
                    sendSelectedCoworkersMessagesRequest(addedLabelsMessages, deletedLabelsMessages);
                } catch (HttpException e) {
                    context.session().logger().log(
                        Level.SEVERE,"LabelsCallback: request to sendSelectedCoworkersMessagesRequest failed", e);
                    failed(e);
                }
            } catch (JsonUnexpectedTokenException e) {
                context.session().logger().log(Level.SEVERE, "Labels request's processing failed", e);
                failed(e);
            }
        }

        private void addMids(Set<String> mids, final Map<String, List<Long>> labelsMessages)
        {
            for (List<Long> labelMids: labelsMessages.values()) {
                for (final Long mid: labelMids) {
                    try {
                        mids.add(ValueUtils.asString(mid));
                    } catch (JsonUnexpectedTokenException e) {
                        context.session().logger().log(Level.SEVERE, "Unable to convert mid to String: " + mid, e);
                    }
                }
            }
        }

        private void sendSelectedCoworkersMessagesRequest(
            final Map<String, List<Long>> addedLabelsMessages,
            final Map<String, List<Long>> deletedLabelsMessages)
            throws HttpException
        {
            context.session().logger().log(Level.SEVERE,
                "sendSelectedCoworkersMessagesRequest: ADDED "
                + addedLabelsMessages.toString() + " DELETED " + deletedLabelsMessages.toString());
            Set<String> mids = new HashSet<>();
            addMids(mids, addedLabelsMessages);
            addMids(mids, deletedLabelsMessages);
            new GatherStidsFilterSearchCallback(
                context,
                new ComplaintsRequestCallback(context, addedLabelsMessages, deletedLabelsMessages),
                mids).execute();
        }

        private Map<String, List<Long>> getLabelsChangedMessages(
            final Map<String, String> lids, // lid -> labelName
            final String filterKey)
            throws JsonUnexpectedTokenException
        {
            List<String> labelNames = new ArrayList<>();
            Map<String, List<Long>> labelsMids = new HashMap<>();
            Map<?, ?> arguments = ValueUtils.asMapOrNull(context.json().get("arguments"));
            if (arguments != null) {
                List<?> changedLids = ValueUtils.asListOrNull(arguments.get(filterKey));
                if (changedLids != null) {
                    for (Object lid: changedLids) {
                        try {
                            String labelName = lids.get(ValueUtils.asString(lid));
                            if (labelName != null) {    // changed lid must be special!
                                labelNames.add(labelName);
                                labelsMids.put(labelName, new ArrayList<>());
                            }
                        } catch (JsonUnexpectedTokenException e) {
                            context.session().logger().log(Level.SEVERE, "Unable to convert lid to String: " + lid, e);
                        }
                    }
                }
            }
            if (labelNames.size() > 0) {
                Object changed = context.json().get(CHANGED);
                if (changed instanceof List) {
                    for (Object changedItem: (List<?>) changed) {
                        if (changedItem instanceof Map) {
                            Map<?, ?> changedMap = (Map<?, ?>) changedItem;
                            if (changedMap.get(MID) instanceof Long) {
                                Long mid = (Long)changedMap.get(MID);
                                for (String labelName: labelNames) {
                                    labelsMids.get(labelName).add(mid);
                                }
                            }
                        }
                    }
                }
            } else {
                context.session().logger().log(
                        Level.SEVERE, "getLabelsChangedMessages " + filterKey + ": there are no labels!");
            }
            return labelsMids;
        }
    }

    private static class GatherStidsFilterSearchCallback
        extends AbstractFilterSearchCallback<Map.Entry<Long, HashMap<String, Object>>>
    {
        GatherStidsFilterSearchCallback(
            final ChangeContext context,
            final FutureCallback<List<Map.Entry<Long, HashMap<String, Object>>>> callback,
            final Set<String> mids)
        {
            super(context, callback, mids);
        }

        public boolean skipEmptyEntities() {
            return false;
        }

        public boolean skipSpam() {
            return false;
        }

        public AbstractCallback<Map.Entry<Long, HashMap<String, Object>>> subMessageCallback(
            final IndexationContext<Map.Entry<Long, HashMap<String, Object>>> context)
        {
            return new SingleMessageReporter<>(context);
        }

        public void executeSubCallback(
            final AbstractCallback<Map.Entry<Long, HashMap<String, Object>>> callback)
        {
        }

        @Override
        public void completed(final List<FirstlineMailMetaInfo> metas) {
            context.iexProxy().filterSearchCompleted(System.currentTimeMillis() - start);
            if (metas.isEmpty()) {
                context.session().logger().info("All mids from " + context.iexProxy().filterSearchUri() + " have gone");
                callback.completed(Collections.emptyList());
            } else {
                Long operationDate = (long) Math.floor(new Date().getTime() / 1000.0);
                ArrayList<Map.Entry<Long, HashMap<String, Object>>> midsInfo = new ArrayList<>();
                Object operationDateStr = context.json().get(OPERATION_DATE);
                if (operationDateStr instanceof String) {
                    int i = ((String) operationDateStr).indexOf('.');
                    if (i > 0) {
                        operationDate = Long.parseLong(((String) operationDateStr).substring(0, i));
                    }
                }
                for (final FirstlineMailMetaInfo meta: metas) {
                    context.session().logger().info("STID for mid <" + meta.get(MID) + ">: " + meta.get(STID));
                    try {
                        Long mid = ValueUtils.asLong(meta.get(MID));
                        HashMap<String, Object> info = new HashMap<>();
                        info.put(MID, meta.get(MID));
                        info.put(STID, meta.get(STID));
                        info.put(SEEN, (meta.get(MailIndexFields.UNREAD) == "true") ? "" : "1");
                        info.put(FOLDER, meta.get(MailIndexFields.FOLDER_NAME));
                        info.put(OPERATION_DATE, operationDate);
                        midsInfo.add(Map.entry(mid, info));
                    } catch (JsonUnexpectedTokenException e) {
                        context.session().logger().log(
                            Level.SEVERE, "Unable to convert mid to Long: " + meta.get(MID), e);
                    }
                }
                this.callback.completed(midsInfo);
            }
        }
    }

    private static class ComplaintsRequestCallback
        implements FutureCallback<List<Map.Entry<Long, HashMap<String, Object>>>>
    {
        private final ChangeContext context;
        private final Map<String, Map<String, List<Long>>> labelsMessages;

        ComplaintsRequestCallback(
            final ChangeContext context,
            final Map<String, List<Long>> addedLabelsMessages,
            final Map<String, List<Long>> deletedLabelsMessages)
        {
            this.context = context;
            this.labelsMessages = new HashMap<>();
            this.labelsMessages.put(LIDS_ADD, addedLabelsMessages);
            this.labelsMessages.put(LIDS_DEL, deletedLabelsMessages);
        }

        private void sendRequest(
            final String action,
            final String uri,
            final Map<Long, HashMap<String, Object>> msgsInfo)
        {
            QueryConstructor query;
            try {
                // sequental separate query for separate labelName and action
                for (Map.Entry<String, List<Long>> labelInfo : this.labelsMessages.get(action).entrySet()) {
                    query = new QueryConstructor(uri);
                    query.append(ACTION, action);
                    query.append(LABEL, labelInfo.getKey());
                    try (AsyncClient client =
                        context.iexProxy().complaintsCoworkersSelectionClient().adjust(context.session().context()))
                    {
                        StringBuilderWriter sbw = new StringBuilderWriter();
                        try (JsonWriter writer = new JsonWriter(sbw)) {
                            writer.startArray();
                            for (Map.Entry<Long, HashMap<String, Object>> msgInfo: msgsInfo.entrySet()) {
                                HashMap<String, Object> info = msgInfo.getValue();
                                writer.startObject();
                                writer.key(MID);
                                writer.value(msgInfo.getKey());
                                writer.key(STID);
                                writer.value(info.get(STID));
                                writer.key(SEEN);
                                writer.value(info.get(SEEN));
                                writer.key(FOLDER);
                                writer.value(info.get(FOLDER));
                                writer.key(OPERATION_DATE);
                                writer.value(info.get(OPERATION_DATE));
                                writer.endObject();
                            }
                            writer.endArray();
                        }
                        final AsyncPostURIRequestProducerSupplier post =
                            new AsyncPostURIRequestProducerSupplier(
                                query.toString(),
                                sbw.toString(),
                                ContentType.APPLICATION_JSON);
                        client.execute(
                            post,
                            EmptyAsyncConsumerFactory.OK,
                            context.session().listener().createContextGeneratorFor(client),
                            new ComplaintsResponseCallback(context));
                        context.session().logger().log(Level.SEVERE,
                            "ComplaintsRequestCallback: Request to " + query + " has sent");
                    } catch (IOException e) {
                        context.session().logger().log(Level.SEVERE,
                            "ComplaintsRequestCallback: Labels request failed", e);
                        context.session().handleException(HttpExceptionConverter.toHttpException(e));
                    }
                }
            } catch (BadRequestException | URISyntaxException e) {
                context.session().logger().log(Level.SEVERE,
                    "ComplaintsRequestCallback: Labels request building error", e);
                context.session().handleException(HttpExceptionConverter.toHttpException(e));
            }
        }

        @Override
        public void completed(final List<Map.Entry<Long, HashMap<String, Object>>> midsInfo) {
            context.session().logger().info("ComplaintsRequestCallback: call to transfer messages");
            ImmutableURIConfig complaintsCoworkersSelectionConfig =
                this.context.iexProxy.config().complaintsCoworkersSelectionConfig();
            String cUri = complaintsCoworkersSelectionConfig.uri().toASCIIString()
                + complaintsCoworkersSelectionConfig.firstCgiSeparator() + UID + '=' + context.prefix();
            Map<Long, HashMap<String, Object>> msgsInfo = new HashMap<>();
            for (final Map.Entry<Long, HashMap<String, Object>> entry : midsInfo) {
                msgsInfo.put(entry.getKey(), entry.getValue());
            }
            sendRequest(LIDS_ADD, cUri, msgsInfo);
            sendRequest(LIDS_DEL, cUri, msgsInfo);
        }

        @Override
        public void cancelled() {
            context.session().logger().log(Level.SEVERE,
                "ComplaintsRequestCallback: Labels request to complaints handler cancelled");
            //this.context.callback().cancelled();
            context.session().response(YandexHttpStatus.SC_CLIENT_CLOSED_REQUEST);
        }

        @Override
        public void failed(final Exception e) {
            context.session().logger().log(Level.SEVERE,
                "ComplaintsRequestCallback: Labels request to complaints handler cancelled", e);
            //this.context.callback().failed(e);
            context.session().handleException(HttpExceptionConverter.toHttpException(e));
        }
    }

    private static class ComplaintsResponseCallback implements FutureCallback<Void>
    {
        private final ChangeContext context;

        ComplaintsResponseCallback(final ChangeContext context) {
            this.context = context;
        }

        @Override
        public void completed(final Void response) {
            context.session().logger().info(
                "ComplaintsResponseCallback: Labels info successfully stored to special user's folder");
            context.session().response(YandexHttpStatus.SC_OK);
        }

        @Override
        public void cancelled() {
            this.context.session().logger().warning(
                "ComplaintsResponseCallback: Request cancelled: " + this.context.session().listener().details());
            context.session().response(YandexHttpStatus.SC_CLIENT_CLOSED_REQUEST);
        }

        @Override
        public void failed(final Exception e) {
            this.context.session().logger().log(
                Level.WARNING,
                "ComplaintsResponseCallback: Failed to process: "
                    + context.humanReadableJson()
                    + '\n' + context.session().listener().details()
                    + " because of exception", e);
            context.session().handleException(HttpExceptionConverter.toHttpException(e));
        }
    }
}

