package ru.yandex.iex.proxy;

import java.io.ByteArrayOutputStream;
import java.io.IOException;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.parser.uri.QueryConstructor;

public class SOIndexHandler
    implements HttpAsyncRequestHandler<JsonObject>
{
    private final IexProxy iexProxy;
    private final String soService;
    private final HttpHost producerHost;
    private final ThreadLocal<ByteArrayOutputStream> baosTls =
        ThreadLocal.<ByteArrayOutputStream>withInitial(
            () -> new ByteArrayOutputStream());

    public SOIndexHandler(final IexProxy iexProxy) {
        this.iexProxy = iexProxy;
        this.soService = iexProxy.config().soService();
        if (soService != null) {
            this.producerHost =
                iexProxy.config().soProducerConfig().host();
        } else {
            this.producerHost = null;
        }
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException
    {
        if (!(request instanceof HttpEntityEnclosingRequest)) {
            throw new BadRequestException("Payload expected");
        }
        return new JsonAsyncTypesafeDomConsumer(
            ((HttpEntityEnclosingRequest) request).getEntity(),
            StringCollectorsFactory.INSTANCE,
            BasicContainerFactory.INSTANCE);
    }

    @Override
    public void handle(
        final JsonObject payload,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        if (soService == null) {
            throw new ServiceUnavailableException(
                "config.so-service is not set");
        }
        ProxySession session =
            new BasicProxySession(iexProxy, exchange, context);
        Context soContext = new Context();
        soContext.uid = session.params().getLong("uid");
        soContext.smtpId = session.params().getString("queue-id");
        final ByteArrayOutputStream baos = baosTls.get();
        baos.reset();
        try (
            Utf8JsonWriter writer = JsonType.DOLLAR.create(baos))
        {
            writePayload(writer, payload, soContext);
            writer.flush();
        } catch (JsonException|IOException e) {
            throw new BadRequestException(e);
        }
        byte[] array = baos.toByteArray();
        QueryConstructor query = new QueryConstructor("/update?solog");
        query.append("prefix", Long.toString(soContext.uid));
        query.append("service", soService);
        query.append("smtp-id", soContext.smtpId);
        final BasicAsyncRequestProducerGenerator post =
            new BasicAsyncRequestProducerGenerator(
                query.toString(),
                array,
                ContentType.DEFAULT_BINARY);

        post.addHeader(YandexHeaders.SERVICE, soService);
        post.addHeader(
            YandexHeaders.X_INDEX_OPERATION_TIMESTAMP,
            Long.toString(System.currentTimeMillis()));
        post.addHeader(
            YandexHeaders.X_INDEX_OPERATION_QUEUE,
            soService + "_index");
        AsyncClient producerAsyncClient =
            iexProxy.soProducerClient().adjust(context);
        producerAsyncClient.execute(
            producerHost,
            post,
            AsyncStringConsumerFactory.OK,
            session.listener()
                .createContextGeneratorFor(producerAsyncClient),
            new Callback(session, soContext));
    }

    private void writePayload(
        final Utf8JsonWriter writer,
        final JsonObject payload,
        final Context soContext)
        throws JsonException, IOException
    {
        JsonMap map = payload.asMap();
//        soContext.smtpId = map.getString("queue_id");
        String url = "solog_" + soContext.uid + "_" + soContext.smtpId;
        writer.startObject();
        writer.key("prefix");
        writer.value(soContext.uid);
        writer.key("AddIfNotExists");
        writer.value(true);
        writer.key("docs");
        writer.startArray();
        writer.startObject();
        writer.key("url");
        writer.value(url);
        writer.key("solog_timestamp");
        writer.value(System.currentTimeMillis());
        writer.key("solog_uid");
        writer.value(soContext.uid);
        writer.key("solog_smtp_id");
        writer.value(soContext.smtpId);
        writer.key("solog_data");
        writer.value(JsonType.NORMAL.toString(map));
        writer.endObject();
        writer.endArray();
        writer.endObject();
    }

    private static class Context {
        private String smtpId;
        private long uid;
    }

    private static class Callback extends AbstractProxySessionCallback<String> {
        private final Context soContext;

        Callback(final ProxySession session, final Context soContext) {
            super(session);
            this.soContext = soContext;
        }

        @Override
        public void completed(final String result) {
            session.logger().info("So log <" + soContext.smtpId + "> enqueued: "
                + result);
            session.response(HttpStatus.SC_OK);
        }
    }
}
