package ru.yandex.iex.proxy;

import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import com.twmacinta.util.MD5;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.util.string.HexStrings;

public class TikaiteHeadersConvertingCallback
    extends AbstractFilterFutureCallback<JsonObject, Map<String, String>>
{
    private static final String HEADERS = "headers";
    private static final String RPOP_INFO_HEADER_NAME = "x-yandex-rpop-info";
    private static final String MESSAGE_ID = "message-id";
    private static final String MESSAGE_ID_HASH = "message-id-hash";
    private static final String RPOP_INFO = "rpop-info";
    private static final String NONE = "none";
    private static final String FOREIGN = "foreign";
    private static final String YANDEX = "yandex";

    private final Set<String> headersWanted;
    private final boolean parseRpopInfo;
    private final boolean messageIdHash;

    public TikaiteHeadersConvertingCallback(
        final FutureCallback<? super Map<String, String>> callback,
        final Set<String> headersWanted)
    {
        super(callback);
        this.headersWanted = headersWanted;
        parseRpopInfo = headersWanted.contains(RPOP_INFO);
        messageIdHash = headersWanted.contains(MESSAGE_ID_HASH);
    }

    @Override
    public void completed(final JsonObject response) {
        try {
            Map<String, String> result =
                new HashMap<>(headersWanted.size() << 1);
            for (JsonObject headerObject: response.get("headers").asList()) {
                for (Map.Entry<String, JsonObject> entry
                    : headerObject.asMap().entrySet())
                {
                    processField(
                        result,
                        entry.getKey(),
                        entry.getValue().asString());
                }
            }
            if (parseRpopInfo) {
                result.putIfAbsent(RPOP_INFO, NONE);
            }
            callback.completed(result);
        } catch (JsonException e) {
            callback.failed(e);
        }
    }

    private void processField(
        final Map<String, String> result,
        final String name,
        final String value)
    {
        if (headersWanted.contains(name)) {
            result.put(name, value);
        }
        if (parseRpopInfo && name.equals(RPOP_INFO_HEADER_NAME)) {
            String info;
            if (value.contains(YANDEX)) {
                info = YANDEX;
            } else {
                info = FOREIGN;
            }
            result.put(RPOP_INFO, info);
        } else if (messageIdHash && name.equals(MESSAGE_ID)) {
            MD5 md5 = new MD5();
            md5.Update(value.getBytes(StandardCharsets.UTF_16));
            result.put(
                MESSAGE_ID_HASH,
                new String(HexStrings.UPPER.process(md5.Final())));
        }
    }
}

