package ru.yandex.iex.proxy;

import java.io.IOException;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.FormBodyPartBuilder;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.StringBody;

import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.blackbox.SingleUserinfoBlackboxCallback;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.iex.proxy.complaints.TraceFutureCallback;
import ru.yandex.iex.proxy.move.SingleMessageReporter;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.DollarJsonWriter;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.document.mail.FirstlineMailMetaInfo;
import ru.yandex.search.document.mail.MailMetaInfo;
import ru.yandex.search.request.util.SearchRequestText;

public enum UpdateStidHandler implements ChangeHandler {
    INSTANCE;

    private static final String MID = "mid";

    @Override
    public void handle(final ChangeContext context) {
        ModifyHandler.indexDocument(
            context,
            new SingleUserinfoBlackboxCallback(new BlackboxCallback(context)));
    }

    private static class BlackboxCallback
        extends AbstractProxySessionCallback<BlackboxUserinfo>
    {
        private final ChangeContext context;

        BlackboxCallback(final ChangeContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final BlackboxUserinfo userinfo) {
            try {
                Set<String> mids = new HashSet<>();
                for (Object change : ValueUtils.asList(context.json().get("changed"))) {
                    mids.add(ValueUtils.asString(ValueUtils.asMap(change).get(MID)));
                }
                new UpdateStidsFilterSearchCallback(
                    context,
                    mids,
                    TraceFutureCallback.wrap(
                        new FilterSearchCallback(context),
                        context,
                        "UpdateStidHandler.BlackboxCallback.completed"))
                    .execute();
            } catch (JsonUnexpectedTokenException e) {
                failed(new BadRequestException(e));
            }
        }
    }

    private static class UpdateStidsFilterSearchCallback
        extends AbstractFilterSearchCallback<FirstlineMailMetaInfo>
    {
        UpdateStidsFilterSearchCallback(
            final ChangeContext context,
            final Set<String> mids,
            final FutureCallback<List<FirstlineMailMetaInfo>> callback)
        {
            super(context, callback, mids);
        }

        public boolean skipEmptyEntities() {
            return false;
        }

        public boolean skipSpam() {
            return false;
        }

        public AbstractCallback<FirstlineMailMetaInfo> subMessageCallback(
            final IndexationContext<FirstlineMailMetaInfo> context)
        {
            return new SingleMessageReporter<>(context);
        }

        public void executeSubCallback(final AbstractCallback<FirstlineMailMetaInfo> callback)
        {
        }

        @Override
        public void completed(final List<FirstlineMailMetaInfo> metas) {
            callback.completed(metas);
        }
    }

    private static class FilterSearchCallback
        extends AbstractProxySessionCallback<List<FirstlineMailMetaInfo>>
    {
        private final ChangeContext context;

        FilterSearchCallback(final ChangeContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final List<FirstlineMailMetaInfo> metas) {
            IexProxy iexProxy = context.iexProxy();
            ImmutableIexProxyConfig config = iexProxy.config();
            String queueName = config.factsIndexingQueueName();
            String uri =
                "/update?stid-update&prefix=" + context.prefix()
                + "&service=" + queueName;
            String xIndexOperationQueueName =
                config.xIndexOperationQueueNameFacts();
            MultipartEntityBuilder builder = MultipartEntityBuilder.create();
            builder.setMimeSubtype("mixed");
            try {
                StringBuilderWriter sbw = new StringBuilderWriter();
                for (FirstlineMailMetaInfo meta : metas) {
                    sbw.clear();
                    String mid = meta.get(MailMetaInfo.MID);
                    try (JsonWriter writer = new DollarJsonWriter(sbw)) {
                        writer.reset();
                        writer.startObject();
                        writer.key("prefix");
                        writer.value(context.prefix());
                        writer.key("query");
                        writer.value(
                            "fact_mid:"
                            + SearchRequestText.fullEscape(mid, false));
                        writer.key("docs");
                        writer.startArray();
                        writer.startObject();
                        writer.key("fact_stid");
                        writer.value(meta.get(MailMetaInfo.STID));
                        writer.endObject();
                        writer.endArray();
                        writer.endObject();
                    }
                    QueryConstructor query = new QueryConstructor(uri);
                    query.append(MID, mid);
                    builder.addPart(
                        FormBodyPartBuilder.create()
                            .addField(YandexHeaders.URI, query.toString())
                            .setBody(
                                new StringBody(
                                    sbw.toString(),
                                    ContentType.APPLICATION_JSON))
                            .setName("update.json")
                            .build());
                }
                UpdateCallback updateCallback = new UpdateCallback(session);
                if (metas.isEmpty()) {
                    updateCallback.completed(null);
                } else {
                    BasicAsyncRequestProducerGenerator producerGenerator =
                        new BasicAsyncRequestProducerGenerator(
                            uri,
                            builder.build());
                    producerGenerator.addHeader(
                        YandexHeaders.SERVICE,
                        queueName);
                    producerGenerator.addHeader(
                        YandexHeaders.X_INDEX_OPERATION_TIMESTAMP,
                        Long.toString(context.operationDateMillis()));
                    if (xIndexOperationQueueName != null) {
                        producerGenerator.addHeader(
                            YandexHeaders.X_INDEX_OPERATION_QUEUE,
                            xIndexOperationQueueName);
                    }

                    HttpHost host = config.producerAsyncClientConfig().host();
                    AsyncClient client =
                        iexProxy.producerAsyncClient().adjust(
                            session.context());
                    client.execute(
                        host,
                        producerGenerator,
                        EmptyAsyncConsumerFactory.OK,
                        session.listener().createContextGeneratorFor(client),
                        updateCallback);
                }
            } catch (IOException e) {
                failed(new BadRequestException(e));
            } catch (BadRequestException e) {
                failed(e);
            }
        }
    }

    private static class UpdateCallback
        extends AbstractProxySessionCallback<Object>
    {
        UpdateCallback(final ProxySession session) {
            super(session);
        }

        @Override
        public void completed(final Object result) {
            session.response(HttpStatus.SC_OK);
        }
    }
}

