package ru.yandex.iex.proxy;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;

public final class XDataStructure {
    private XDataStructure() {
    }

    interface IStopWordsChecker {
        boolean onWord(final String word);
    }

    public static class StopWords {
        private YeildWordTokenizer wt;
        private final String text;
        private final List<IStopWordsChecker> checkers = new ArrayList<>();

        StopWords(final String text, final IStopWordsChecker... checkers) {
            this.text = text;
            wt = new YeildWordTokenizer(text);
            resetCheckers(checkers);
        }

        public void resetCheckers(final IStopWordsChecker... checkers) {
            this.checkers.clear();
            wt.reset();
            this.checkers.addAll(Arrays.asList(checkers));
        }

        public boolean startChecking() {
            while (true) {
                final String w = wt.next();
                if (w.isEmpty()) {
                    break;
                }
                for (final IStopWordsChecker x: checkers) {
                    if (x.onWord(w)) {
                        return true;
                    }
                }
            }
            return false;
        }

        public String getText() {
            return text;
        }
    }

    public static class SimpleStopWords implements IStopWordsChecker {
        private static final Set<String> WORDS; // = new ArrayList<>();

        static {
            WORDS = new HashSet<String>(
                Arrays.asList(
                    "акция",
                    "приезд",
                    "выходные",
                    "срок",
                    "заказ",
                    "телефон",
                    "уважаемый",
                    "работа",
                    "счет",
                    "рабочий"));
        }

        SimpleStopWords() {
        }

        @Override
        public boolean onWord(final String word) {
            return WORDS.contains(word.toLowerCase(Locale.getDefault()));
        }
    }

    public static class SimplePastVerbStopWords implements IStopWordsChecker {
        private static final List<String> SUFFIXES; // = new ArrayList<>();

        static {
            SUFFIXES = Arrays.asList(
                "ел", "ал", "ила", "ил", "ены", "ен", "ым", "нно");
        }

        SimplePastVerbStopWords() {
        }

        @Override
        public boolean onWord(final String word) {
            final String lowWord = word.toLowerCase(Locale.getDefault());
            for (final String suf: SUFFIXES) {
                if (lowWord.endsWith(suf)) {
                    return true;
                }
            }
            return false;
        }
    }

    public static class WidgetReliableWords implements IStopWordsChecker {
        private static final Set<String> WORDS; // = new ArrayList<>();

        static {
            WORDS = new HashSet<String>(Arrays.asList(
                "рождения",
                "встреча",
                "презентация",
                "вебинар",
                "семинар",
                "фестиваль",
                "выставка",
                "заседание",
                "соревнование",
                "конкурс",
                "курс",
                "собрание",
                "совещание",
                "конференция",
                "тренинг",
                "обучение",
                "видеоконференция",
                "мероприятие",
                "олимпиада",
                "мастер-класс",
                "лекция",
                "форум",
                "конгресс",
                "концерт",
                "пресс-конференция"
                ));
        }

        WidgetReliableWords() {
        }

        @Override
        public boolean onWord(final String word) {
            return WORDS.contains(word.toLowerCase(Locale.getDefault()));
        }
    }

    public static class YeildWordTokenizer {
        private int position = 0;
        private final String text;

        YeildWordTokenizer(final String text) {
            this.text = text;
        }

        public void reset() {
            position = 0;
        }

        String next() {
            StringBuilder sb = new StringBuilder();
            while (position < text.length()
                && !Character.isAlphabetic(text.charAt(position)))
            {
                ++position;
            }
            while (position < text.length()
                && (Character.isAlphabetic(text.charAt(position))
                || text.charAt(position) == '-'))
            {
                sb.append(text.charAt(position));
                ++position;
            }
            return sb.toString();
        }
    }

    public interface IMailTypeSearch {
        @SuppressWarnings("unused")
        void addType(final String status, final int type);

        void addType(final String[] status, final int type);

        Integer getTypeOrNull(final String status);
    }

    public static class MailTypeSearchAhoCorasick implements IMailTypeSearch {
        private static class Node {
            private HashMap<Integer, Node> curNode = new HashMap<>();
            private int type = -1;

            Node nextOrCreate(final int v) {
                if (!curNode.containsKey(v)) {
                    curNode.put(v, new Node());
                }
                return curNode.get(v);
            }

            Node nextOrNull(final int v) {
                if (!curNode.containsKey(v)) {
                    return null;
                }
                return curNode.get(v);
            }

            void setType(final int v) {
                type = v;
            }

            int getType() {
                return type;
            }
        }

        private Node root = new Node();

        @Override
        public void addType(final String status, final int type) {
            Node itr = root;
            for (Integer x : getCodes(status)) {
                itr = itr.nextOrCreate(x);
            }
            itr.setType(type);
        }

        @Override
        public void addType(final String[] status, final int type) {
            for (String sx : status) {
                addType(sx, type);
            }
        }

        @Override
        public Integer getTypeOrNull(final String status) {
            Node itr = root;
            for (Integer x : getCodes(status)) {
                itr = itr.nextOrNull(x);
                if (itr == null) {
                    return null;
                }
            }
            return itr.getType();
        }

        @SuppressWarnings("StringSplitter")
        private Integer[] getCodes(final String status) {
            String[] intsStr = status.split("\\.");
            Integer[] intsInt = new Integer[intsStr.length];
            for (int i = 0; i < intsStr.length; ++i) {
                intsInt[i] = Integer.valueOf(intsStr[i]);
            }
            return intsInt;
        }
    }

    public static class MailTypeSearchHashMap implements IMailTypeSearch {
        private HashMap<String, Integer> hs = new HashMap<>();

        @Override
        public void addType(final String status, final int type) {
            hs.put(status, type);
        }

        @Override
        public void addType(final String[] status, final int type) {
            for (String sx : status) {
                addType(sx, type);
            }
        }

        @Override
        public Integer getTypeOrNull(final String status) {
            if (!hs.containsKey(status)) {
                return null;
            }
            return hs.get(status);
        }
    }
}
