package ru.yandex.iex.proxy;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.TimeZone;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public final class XRegexpUtils {
    public static final String DOT = ".";
    public static final String MINUS = "-";
    public static final int MININHOUR = 60;
    public static final int THREE = 3;
    public static final int MSK_GMT = THREE;
    public static final int GROUP3 = THREE;
    public static final String REGEX_DASH = "\\-";
    public static final String REGEX_DOT = "\\.";
    public static final String D_02D_02D = "%02d:%02d:%02d";
    public static final String REGEX_SL = "\\:";
    public static final String STR_CURRENCY = "(₺|£|₤|TL|LT|YTL|€|рублей|RUR|RUB|руб[.]|р[. ]|p[. ]|USD|EUR|\\$|Р)";
    public static final String STR_RU_CURRENCY = "(рублей|RUR|RUB|руб[.]|р[. ]|p[. ]|Р)";
    public static final String IGNORE_CASE = "(?i)";
    private static String dateFormat1 = "yyyy-MM-dd'T'HH:mm:ss";
    private static String dateFormat2 = "dd.MM.yyyy'T'HH:mm:ss";

    private static final TimeZone TIMEZONE = TimeZone.getTimeZone("Europe/Moscow");

    private static final Pattern CV = Pattern.compile(".*[рР]езюме.*|.* cv .*|.* CV .*|.*[вВ]акансия.*");

    private static final Pattern DOUBLE = Pattern.compile("[0-9]+\\.[0-9]+");

    private static final Pattern DOUBLE2 = Pattern.compile("[0-9]+,[0-9]+");

    private static final Pattern INT = Pattern.compile("[0-9]+");

    private static final Pattern RUSWORD = Pattern.compile("[А-Яа-яЁё]+");

    private static final Pattern ENGWORD = Pattern.compile("[A-Za-z]+");

    private static final Pattern TIME = Pattern.compile("[0-9]{1,2}:[0-9]{1,2}:[0-9]{1,2}");

    private static final Pattern TIME2 = Pattern.compile("[0-9]{1,2}:[0-9]{1,2}");

    private static final Pattern GMT = Pattern.compile("([+-])([0-9]{1,2}):([0-9]{1,2})");

    private static final Pattern GMT2 = Pattern.compile("([+-])([0-9]{2})([0-9]{2})");

    private static final Pattern DATE_ISO = // yyyy-MM-dd
        Pattern.compile("((1[0-9]|20)[0-9]{2}|0000)-[0-9]{1,2}-[0-9]{1,2}");

    private static final Pattern DATE_IEX = // dd.MM.yyyy
        Pattern.compile("[0-9]{1,2}.[0-9]{1,2}.(20[0-9]{2}|0)");
    // 0.0.0 - date is valid too

    private static final Pattern DATE_POINT_SEPS = // dd.MM.yyyy
        Pattern.compile("[0-9]{1,2}.[0-9]{1,2}.20[0-9]{2}");

    private static final Pattern DATE_HUMAN_RU_1 =
        Pattern.compile("[0-9]{1,2} (дек|янв|фев|мар|апр|ма|июн|июл|авг|сен|окт|ноя)[а-я]{0,6} 201[0-9]");

    private static final Pattern FLIGHT_NUMBER =
        Pattern.compile("([A-ZА-Я]{2}|[A-Z][0-9]|[0-9][A-Z]|[А-Я][0-9]|[0-9][А-Я])[- ]*([0-9]{1,4})");

    private static final Pattern SEAT = Pattern.compile("(мест[ао][ ,0-9]+|[0-9]+ место)");

    private static final Pattern SEAT_MOVIE = Pattern.compile("[0-9]+ место");

    private static final Pattern HALL = Pattern.compile("[Зз]ал[ №,0-9]+");

    private static final Pattern ROW = Pattern.compile("[рР]яд[ ,0-9]+");

    private static final Pattern ROW_MOVIE = Pattern.compile("[0-9]+ [рР]яд");

    private static final Pattern EMAIL = Pattern.compile(".+@.+");

    private static final Pattern SPECIFIC_DOMAIN = Pattern.compile(".+\\$.+");

    private static final Pattern MESSAGE_ID = Pattern.compile("(?i)Message-Id: (\\S+)");

    private static final Pattern FINAL_RECIPIENT = Pattern.compile("(?i)Final-Recipient: rfc822; (\\S+)");

    private static final Pattern ORIGINAL_RECIPIENT = Pattern.compile("(?i)Original-Recipient: rfc822;(\\S+)");

    private static final Pattern MAIL_STATUS = Pattern.compile("(?i)Status: (\\d\\.\\d{1,3}\\.\\d{1,3})");

    private static final Pattern MAIL_ACTION = Pattern.compile("(?i)Action: ([^\\n]*)");

    private static final Pattern REMOTE_MTA = Pattern.compile("(?i)Remote-MTA: dns; ([^\\n]*)");

    private static final Pattern DIAGNOSTIC_CODE =
        Pattern.compile("(?i)Diagnostic-Code: [^;]*; (.*?(?=\\n[a-zA-Z-]*:))", Pattern.DOTALL);

    private static final Pattern BOUNCE_SPAM =
        Pattern.compile(
            "(?i)(?:spam|sender\\s+domain\\s+must\\s+exist|rejected\\s+due\\s+to\\s+" +
            "(?:DMARC\\s+policy|content\\s+restrictions)|Mail\\s+contents+denied|Sender\\s+denied)",
            Pattern.DOTALL);

    private static final Pattern BOUNCE_REGULAR =
        Pattern.compile("(?i)(?:quota|blocked|inactive|relay)", Pattern.DOTALL);

    private static final Pattern BOUNCE_SMTP =
            Pattern.compile("(?i)(?:\\nX-Mailer: Yamail|Received:\\s+from.*by smtp[^\\n]+?\\((?:nw)?smtp(?:corp)?" +
                "\\/Yandex\\) with E?SMTP)",
                Pattern.DOTALL);

    private static final Pattern BOUNCE_UNKNOWN =
        Pattern.compile(
            "(?i)((?:no\\s+such|unknown|not\\s+stored\\s+this)\\s+user|(?:unknown|bad)\\s+recipient|" +
            "undeliverable\\s+address|(recipient|addressee|user)\\s+(undeliverable|" +
            ".*?\\bunknown\\b|.*?\\bnot\\s+found\\b)|invalid\\s+(mailbox|recipient)|" +
            "unknown\\s+or\\s+illegal\\s+alias|5\\d\\d.*?unknown|no\\s+" +
            "(?:mailbox\\s+here|valid\\s+recipients)|user\\s+doesn\\'t\\s+have\\s+a\\s+account|" +
            "(account|mailbox).*?\\b(unavailable|not\\s+(?:exists?|found))\\b)",
            Pattern.DOTALL);

    private static final Pattern BOUNCE_FORWARD =
            Pattern.compile("(?i)(?:\\bX-Yandex-F(?:orwar|w)d\\:)", Pattern.DOTALL);

    private static final Pattern REPORTING_MTA = Pattern.compile("(?i)Reporting-MTA: dns; (\\S+)");

    private static final Pattern MAIL_HEADER_BODY = Pattern.compile("^(?s)(\\S+.*?)\\s*\\n\\s*\\n\\s*(\\S+.*)$");

    private static final Pattern MAIL_HEADERS_PAIRS =
            Pattern.compile("(?s)(?:^|\\n)([^:\\s]+):[^\\S\\n]*(.*?)(?=\\n\\S+|\\n\\s*\\n|$)");

    private static final Pattern MAIL_FROM = Pattern.compile("(?i)\\nFrom: [^\\n]*<([^\\n]*)>");

    private static final Pattern MAIL_ADDR = Pattern.compile("(?i)[\\w\\+\\-\\.]+\\@(?:(?:[a-z0-9](?:[a-z0-9-]{0," +
            "61}[a-z0-9])?|xn\\-\\-[a-z0-9-]+)\\.)+(?:xn\\-\\-[a-z0-9-]+|[a-z]+)");

    private static final Pattern MAIL_BOUNDARY =
            Pattern.compile("(?i)\\bboundary\\s*=\\s*([\\\"\\'])([^\\\"\\']*?)\\1");

    private static final Pattern MAIL_BOUNDARY_ROUGHLY =
            Pattern.compile("(?i)\\bboundary\\s*=\\s*(?:[\\\"\\'])?(\\S+)(?:[\\\"\\'])?");

    private static final Pattern MAIL_COUNT_LIMIT = Pattern.compile("(?i)messages count limit");

    private static final Pattern SENDER_DOMAIN = Pattern.compile("(?i)[^\\n]+@((?:(?:[a-z0-9](?:[a-z0-9-]{0," +
            "61}[a-z0-9])?|xn\\-\\-[a-z0-9-]+)\\.)+(?:xn\\-\\-[a-z0-9-]+|[a-z]+))");

    private static final Pattern DEP_CITY_SUPERKASSA = Pattern.compile("[А-Я]+, [А-Я]+ -");

    private static final Pattern ARR_CITY_SUPERKASSA = Pattern.compile("- [А-Я]+, [А-Я]+");

    private static final Pattern ONLY_CITY_SUPERKASSA = Pattern.compile("([А-Я]+),");

    private static final Pattern CINEMA_PKPASS_VENUE_NAME = Pattern.compile("\"venue_name\\\" = \\\"([^\"]*)\\\"");

    private static final Pattern LIST_UNSUBSCRIBE = Pattern.compile("(?i)\\nlist-unsubscribe: <([^\\n]*)>");

    private static final Pattern NEWS_SRC = Pattern.compile("src=([^\\s]*)");

    private static final Pattern NEWS_WIDTH = Pattern.compile("w=([^\\s]*)");

    private static final Pattern NEWS_HEIGHT = Pattern.compile("h=([0-9-]*)");

    private static final Pattern HREF_URL = Pattern.compile("href=([^>]*)");

    private static final Pattern ORDER_NUMBER_LAST_SEQ = Pattern.compile("([0-9A-Za-z-]+$)");

    private static final Pattern ORDER_NUMBER_AFTER_TAG = Pattern.compile(">([0-9A-Za-z-]+)");

    private static final Pattern ORDER_WITH_JUNK =
        Pattern.compile(
            "((омер|аш|ние|о|О|№)( вашего)?[ ]*заказ[ае]|Заказ|аказ"
                + " (номер|№[\\\\s]?))[:]?[\\\\n\\\\t\\\\r\\\\s]*№?\\\\s*[0-9a-zA-Z-]+");

    private static final Pattern PRICE = Pattern.compile(IGNORE_CASE + "([0-9]{1}[0-9.,\\p{Z}\\s]*)" + STR_CURRENCY);

    private static final Pattern PRICE_2 =
        Pattern.compile(IGNORE_CASE + STR_CURRENCY + "([\\p{Z}\\s]*[0-9]{1}[0-9.,\\p{Z}\\s]*)");

    private static final Pattern CURRENCY = Pattern.compile(STR_CURRENCY);

    private static final Pattern RU_CURRENCY = Pattern.compile(STR_RU_CURRENCY);

    private XRegexpUtils() {
    }

    public static String getOrderNumberWithJunk(final String text) {
        return parseOnce(ORDER_WITH_JUNK.matcher(text), 0);
    }

    public static boolean isMailCountLimit(final String text) {
        String res = parseOnce(MAIL_COUNT_LIMIT.matcher(text), 0);
        return !res.isEmpty();
    }

    public static boolean isRuCurrency(final String text) {
        return !parseOnce(RU_CURRENCY.matcher(text), 0).isEmpty();
    }

    public static String getOrderNumberLastSeq(final String text) {
        return parseOnce(ORDER_NUMBER_LAST_SEQ.matcher(text), 1);
    }

    public static String getOrderNumberAfterTag(final String text) {
        return parseOnce(ORDER_NUMBER_AFTER_TAG.matcher(text), 1);
    }

    public static String getHrefUrl(final String text) {
        return parseOnce(HREF_URL.matcher(text), 1);
    }

    public static String getNewsSrc(final String text) {
        return parseOnce(NEWS_SRC.matcher(text), 1);
    }

    public static String getNewsWidth(final String text) {
        return parseOnce(NEWS_WIDTH.matcher(text), 1);
    }

    public static String getNewsHeight(final String text) {
        return parseOnce(NEWS_HEIGHT.matcher(text), 1);
    }

    public static String getListUnsubscribe(final String text) {
        return parseOnce(LIST_UNSUBSCRIBE.matcher(text), 1);
    }

    public static String getCinemaVenueName(final String text) {
        return parseOnce(CINEMA_PKPASS_VENUE_NAME.matcher(text), 1);
    }

    public static String getMailHeaders(final String text) {
        return parseOnce(MAIL_HEADER_BODY.matcher(text), 1);
    }

    public static String getMailBody(final String text) {
        return parseOnce(MAIL_HEADER_BODY.matcher(text), 2);
    }

    public static HashMap<String, ArrayList<String>> getMailHeadersMap(final String headers) {
        return parseAsMap(MAIL_HEADERS_PAIRS.matcher(headers));
    }

    public static String getFrom(final String text) {
        return parseOnce(MAIL_FROM.matcher(text), 1);
    }

    public static String getMailAddr(final String headerValue) {
        return parseOnce(MAIL_ADDR.matcher(headerValue), 0);
    }

    public static String getMailBaundary(final String headerContentTypeValue) {
        String boundary = parseOnce(MAIL_BOUNDARY.matcher(headerContentTypeValue), 2);
        if (boundary == null || boundary.isEmpty()) {
            boundary = parseOnce(MAIL_BOUNDARY_ROUGHLY.matcher(headerContentTypeValue), 1);
        }
        return boundary;
    }

    public static String getOriginalRecipient(final String text) {
        return parseOnce(ORIGINAL_RECIPIENT.matcher(text), 1);
    }

    public static ArrayList<String> getAction(final String text) {
        return parseAllStr(MAIL_ACTION.matcher(text), 1);
    }

    public static String getRemoteMTA(final String text) {
        return parseOnce(REMOTE_MTA.matcher(text), 1);
    }

    public static String getCity(final String text) {
        // TODO: make real city extractor
        return parseOnce(ONLY_CITY_SUPERKASSA.matcher(text), 1);
    }

    public static String getDiagnosticCode(final String text) {
        return parseOnce(DIAGNOSTIC_CODE.matcher(text), 1);
    }

    public static String getBounceSpam(final String text) {
        return parseOnce(BOUNCE_SPAM.matcher(text), 0);
    }

    public static String getBounceRegular(final String text) {
        return parseOnce(BOUNCE_REGULAR.matcher(text), 0);
    }

    public static String getBounceSmtp(final String text) {
        return parseOnce(BOUNCE_SMTP.matcher(text), 0);
    }

    public static String getBounceUnknown(final String text) {
        return parseOnce(BOUNCE_UNKNOWN.matcher(text), 0);
    }

    public static String getBounceForward(final String text) {
        return parseOnce(BOUNCE_FORWARD.matcher(text), 0);
    }

    public static String getSenderDomain(final String fromEmail) {
        return parseOnce(SENDER_DOMAIN.matcher(fromEmail), 1);
    }

    public static ArrayList<String> getFinalRecipient(final String text) {
        return parseAllStr(FINAL_RECIPIENT.matcher(text), 1);
    }

    public static String getMailStatus(final String text) {
        return parseOnce(MAIL_STATUS.matcher(text), 1);
    }

    public static String getReportingMTA(final String text) {
        return parseOnce(REPORTING_MTA.matcher(text), 1);
    }

    public static ArrayList<String> getPrices(final String text) {
        ArrayList<String> res = parseAllStr(PRICE.matcher(text), 1);
        res.addAll(parseAllStr(PRICE_2.matcher(text), 2));
        return res;
    }

    public static ArrayList<String> getCurrencies(final String text) {
        return parseAllStr(CURRENCY.matcher(text), 1);
    }

    public static ArrayList<String> getMessageIdMail(final String text) {
        return parseAllStr(MESSAGE_ID.matcher(text), 1);
    }

    public static ArrayList<String> getFlightAsParts(final String flight) {
        Matcher mtf = FLIGHT_NUMBER.matcher(flight);
        ArrayList<String> res = new ArrayList<>();
        if (mtf.find()) {
            res.add(mtf.group(1));
            res.add(mtf.group(2));
        }
        return res;
    }

    public static String getTime(final String s) {
        Matcher matcher = TIME.matcher(s);
        String result = s;
        if (matcher.find()) {
            final String[] allTime = matcher.group(0).split(REGEX_SL);
            result = String.format(
                D_02D_02D,
                Integer.parseInt(allTime[0]),
                Integer.parseInt(allTime[1]),
                Integer.parseInt(allTime[2]));
        }
        matcher = TIME2.matcher(s);
        if (matcher.find()) {
            final String[] allTime = matcher.group(0).split(REGEX_SL);
            result = String.format(
                D_02D_02D,
                Integer.parseInt(allTime[0]),
                Integer.parseInt(allTime[1]),
                0);
        }
        return result;
    }

    public static String getAllHidsInfoWithLengthTwo(final String text) {
        Pattern fromTo =
            Pattern.compile(
                "\\#BEGIN_[0-9]*\\.[0-9]*\\n(.*?)\\#END_",
                Pattern.DOTALL);
        ArrayList<String> dates = parseAllStr(fromTo.matcher(text));
        StringBuilder textResult = new StringBuilder();
        for (String x : dates) {
            textResult.append(x);
        }
        return new String(textResult);
    }

    public static ArrayList<String> getAllTimes(final String s) {
        ArrayList<String> times = parseAllStr(TIME.matcher(s));
        times.addAll(parseAllStr(TIME2.matcher(s)));
        return times;
    }

    public static ArrayList<String> getAllFlightNumbers(final String s) {
        ArrayList<String> flightNumbers = parseAllStr(FLIGHT_NUMBER.matcher(s));
        return flightNumbers;
    }

    public static ArrayList<String> getAllDates(final String s) {
        ArrayList<String> dates = parseAllStr(DATE_HUMAN_RU_1.matcher(s));
        return dates;
    }

    public static ArrayList<String> getAllCommaSepedDates(final String s) {
        ArrayList<String> dates = parseAllStr(DATE_POINT_SEPS.matcher(s));
        return dates;
    }

    public static ArrayList<String> getDepCities(final String s) {
        ArrayList<String> depCities =
            parseAllStr(DEP_CITY_SUPERKASSA.matcher(s));
        return depCities;
    }

    public static ArrayList<String> getArrCities(final String s) {
        ArrayList<String> arrCities =
            parseAllStr(ARR_CITY_SUPERKASSA.matcher(s));
        return arrCities;
    }

    public static ArrayList<String> getRows(final String s) {
        return parseAllStr(ROW.matcher(s));
    }

    public static ArrayList<String> getRowsMovieSpec(final String s) {
        return parseAllStr(ROW_MOVIE.matcher(s));
    }

    public static ArrayList<String> getHalls(final String s) {
        return parseAllStr(HALL.matcher(s));
    }

    public static ArrayList<String> getSeats(final String s) {
        return parseAllStr(SEAT.matcher(s));
    }

    public static ArrayList<String> getSeatsMovieSpec(final String s) {
        return parseAllStr(SEAT_MOVIE.matcher(s));
    }

    public static ArrayList<String> getDoubles(final String s) {
        Matcher matcher = DOUBLE.matcher(s);
        ArrayList<String> dbls = new ArrayList<>(2);
        while (matcher.find()) {
            dbls.add(matcher.group(0));
        }
        return dbls;
    }

    public static ArrayList<String> getDoubles2(final String s) {
        return parseAllStr(DOUBLE2.matcher(s));
    }

    public static ArrayList<Integer> getInts(final String s) {
        Matcher matcher = INT.matcher(s);
        ArrayList<Integer> ints = new ArrayList<>();
        while (matcher.find()) {
            ints.add(Integer.parseInt(matcher.group(0)));
        }
        return ints;
    }

    public static ArrayList<String> getStrInts(final String s) {
        Matcher matcher = INT.matcher(s);
        ArrayList<String> ints = new ArrayList<>();
        while (matcher.find()) {
            ints.add(matcher.group(0));
        }
        return ints;
    }

    public static ArrayList<String> getRusWords(final String s) {
        Matcher matcher = RUSWORD.matcher(s);
        ArrayList<String> ruswords = new ArrayList<>();
        while (matcher.find()) {
            ruswords.add(matcher.group(0));
        }
        return ruswords;
    }

    public static ArrayList<String> getEngWords(final String s) {
        Matcher matcher = ENGWORD.matcher(s);
        ArrayList<String> engwords = new ArrayList<>();
        while (matcher.find()) {
            engwords.add(matcher.group(0));
        }
        return engwords;
    }

    public static String timeToReminderTime(final String s) {
        // int mskGmt = getTimeShiftOnMskInMinutes(gmtInMinutes);
        // int seconds = getTimeInSeconds(getTime(s));
        String time = getTime(s);
        // time = getTimeFromSeconds(seconds + mskGmt * MININHOUR);
        String date = getDateInISO(s);
        date = toISODate(date);
        return new StringBuilder(date).append(" ").append(time).toString();
    }

    public static int getGmtInMinutes(final String s) {
        Matcher matcher = GMT.matcher(s);
        int result = MSK_GMT * MININHOUR; // MSK time default
        if (matcher.find()) {
            String sign = matcher.group(1);
            int h = Integer.parseInt(matcher.group(2));
            int m = Integer.parseInt(matcher.group(GROUP3));
            result = h * MININHOUR + m;
            if (sign.equals(MINUS)) {
                result = -result;
            }
        } else {
            matcher = GMT2.matcher(s);
            if (matcher.find()) {
                String sign = matcher.group(1);
                result = Integer.parseInt(matcher.group(2)) * MININHOUR
                    + Integer.parseInt(matcher.group(GROUP3));
                if (sign.equals(MINUS)) {
                    result = -result;
                }
            }
        }
        return result;
    }

    public static String getDateInISO(final String s) {
        Matcher matcher = DATE_ISO.matcher(s);
        String[] result = {};
        if (matcher.find()) {
            result = matcher.group(0).split(REGEX_DASH);
        } else {
            matcher = DATE_IEX.matcher(s);
            if (matcher.find()) {
                result = matcher.group(0).split(REGEX_DOT);
                for (int i = 0; i < result.length / 2; i++) {
                    XStrAlgo.swap(result, i, result.length - 1 - i);
                }
            }
        }
        if (result.length < THREE) {
            return s;
        }
        return String.format(
            "%04d-%02d-%02d",
            Integer.parseInt(result[0]),
            Integer.parseInt(result[1]),
            Integer.parseInt(result[2]));
    }

    public static long getTimestamp(
        final String time,
        final String date,
        final int gmtInMinutes)
    {
        long res = -1;
        if (time != null && date != null && isStrTime(time)) {
            String format;
            if (isISODate(date)) {
                format = dateFormat1;
            } else if (isIEXDate(date)) {
                format = dateFormat2;
            } else {
                return res;
            }
            try {
                SimpleDateFormat sdf = new SimpleDateFormat(format);
                sdf.setTimeZone(TIMEZONE);
                StringBuilder sb =
                    new StringBuilder(date.length() + time.length() + 1);
                sb.append(date);
                sb.append('T');
                sb.append(time);
                Date d = sdf.parse(new String(sb));
                res =
                    TimeUnit.MILLISECONDS.toSeconds(d.getTime())
                    + ((long) getTimeShiftOnMskInMinutes(gmtInMinutes))
                    * MININHOUR;
            } catch (ParseException e) {
                res = -1;
            }
        }
        return res;
    }

//    public static long getTimestampWithoutGMT(
//        final String time,
//        final String date)
//    {
//        return getTimestamp(time, date, MSK_GMT * MININHOUR);
//    }

    public static boolean isStrContainsCv(final String s) {
        return CV.matcher(s).matches();
    }

    public static boolean isStrTime(final String s) {
        return TIME.matcher(s).matches() || TIME2.matcher(s).matches();
    }

    public static boolean isISODate(final String s) {
        return DATE_ISO.matcher(s).matches();
    }

    public static boolean isIEXDate(final String s) {
        return DATE_IEX.matcher(s).matches();
    }

    public static boolean isEmail(final String s) {
        return EMAIL.matcher(s).matches();
    }

    public static boolean isSpecificDomain(final String s) {
        return SPECIFIC_DOMAIN.matcher(s).matches();
    }

    public static String parseOnce(final Matcher matcher, final int group)
    {
        if (matcher.find()) {
            return matcher.group(group);
        }
        return "";
    }
    /*
    public static int getTimeInSeconds(final String time) {
        int result = 0;
        if (isStrTime(time)) {
            String[] pr = time.split(":");
            int cnv = MININHOUR * MININHOUR;
            for (final String x : pr) {
                result += Integer.parseInt(x) * cnv;
                cnv /= MININHOUR;
            }
        }
        return result;
    }
    */

    /*
    public static String getTimeFromSeconds(final int timeInSeconds) {
        int cnv = MININHOUR * MININHOUR;
        int hours = timeInSeconds / cnv;
        int minutes = (timeInSeconds % cnv) / MININHOUR;
        int seconds = timeInSeconds % MININHOUR;
        return String.format("%02d:%02d:%02d", hours, minutes, seconds);
    }
    */

    public static String toIEXDate(final String s) {
        if (isISODate(s)) {
            String[] del = s.split(REGEX_DASH);
            return new StringBuilder(del[2])
                .append(DOT).append(del[1])
                .append(DOT).append(del[0]).toString();
        }
        return s;
    }

    public static String toISODate(final String s) {
        if (isIEXDate(s)) {
            String[] del = s.split(REGEX_DOT);
            return new StringBuilder(del[2])
                .append(MINUS).append(del[1])
                .append(MINUS).append(del[0]).toString();
        }
        return s;
    }

    public static int getTimeShiftOnMskInMinutes(final int gmtInMinutes) {
        return gmtInMinutes - MSK_GMT * MININHOUR;
    }

    private static ArrayList<String> parseAllStr(final Matcher matcher) {
        ArrayList<String> ress = new ArrayList<>();
        while (matcher.find()) {
            ress.add(matcher.group(0));
        }
        return ress;
    }

    private static ArrayList<String> parseAllStr(
        final Matcher matcher,
        final int group)
    {
        ArrayList<String> ress = new ArrayList<>();
        while (matcher.find()) {
            ress.add(matcher.group(group));
        }
        return ress;
    }

    private static HashMap<String, ArrayList<String>> parseAsMap(final Matcher matcher)
    {
        HashMap<String, ArrayList<String>> ress = new HashMap<>();
        while (matcher.find()) {
            if (matcher.groupCount() > 1) {
                ress.computeIfAbsent(matcher.group(1).toLowerCase(), x -> new ArrayList<>()).add(matcher.group(2));
            }
        }
        return ress;
    }
}
