package ru.yandex.iex.proxy;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;

import ru.yandex.iex.proxy.xutils.StringUtils;

public final class XTimeUtils {
    public static final int HHMMSS_LEN = 100;
    public static final int SEC_IN_MIN_OR_MIN_IN_HOUR = 60;
    public static final int SEC_IN_HOUR =
        SEC_IN_MIN_OR_MIN_IN_HOUR * SEC_IN_MIN_OR_MIN_IN_HOUR;
    public static final int HOURS_IN_DAY = 24;
    public static final String T = "T";
    public static final String DASH = "-";
    public static final int MONTH_COUNT = 12;
    public static final int DAYS_IN_YEAR = 365;
    public static final int TEN_YEARS = 10;
    private static final int MAX_STR_LEN_MONTH = 3;

    private static Map<String, Integer> name2num = new HashMap<>();
    private static Map<String, Integer> nameEng2num = new HashMap<>();
    private static Calendar calendar = Calendar.getInstance();

    static {
        String[] mnth = {
            "янв", "фев",
            "мар", "апр", "май",
            "июн", "июл", "авг",
            "сен", "окт", "ноя",
            "дек"};
        String[] mnthEng = {
            "jan", "feb",
            "mar", "apr", "may",
            "jun", "jul", "aug",
            "sep", "oct", "nov",
            "dec"};
        for (int i = 1; i <= MONTH_COUNT; ++i) {
            name2num.put(mnth[i - 1], i);
            nameEng2num.put(mnthEng[i - 1], i);
        }
        name2num.put("мая", 2 + 2 + 1);
    }

    private static final String TIME_FORMAT = "yyyy-MM-dd HH:mm:ss";
    private static final TimeZone TIMEZONE =
        TimeZone.getTimeZone("Europe/Moscow");

    private XTimeUtils() {
    }

    @SuppressWarnings("unused")
    public static int getHhMmSsOnly() {
        return getHhMmSsOnly(System.currentTimeMillis());
    }

    public static int getHhMmSsOnly(final long timestamp) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeInMillis(timestamp);
        int hh = calendar.get(Calendar.HOUR_OF_DAY);
        int mm = calendar.get(Calendar.MINUTE);
        int ss = calendar.get(Calendar.SECOND);
        return (hh * HHMMSS_LEN + mm) * HHMMSS_LEN + ss;
    }

    public static String makeIexDateFromRuFormat(
        final String date,
        final String time)
    {
        ArrayList<Integer> nums = XRegexpUtils.getInts(date);
        ArrayList<String> words = XRegexpUtils.getRusWords(date);
        int year = calendar.get(Calendar.YEAR);
        int month = calendar.get(Calendar.MONTH);
        int day = calendar.get(Calendar.DAY_OF_MONTH);
        if (nums.size() == 2 && words.size() == 1) {
            month = getMonthNumberFromMonthName(words.get(0));
            day = Math.min(nums.get(0), nums.get(1));
            year = Math.max(nums.get(0), nums.get(1));
        }
        StringBuilder sb = new StringBuilder();
        sb.append(day);
        sb.append('.');
        sb.append(month);
        sb.append('.');
        sb.append(year);
        sb.append(' ');
        sb.append(formatTime(time));
        return sb.toString();
    }

    public static String parseDate(final String date) {
        ArrayList<Integer> nums = XRegexpUtils.getInts(date);
        ArrayList<String> words = XRegexpUtils.getRusWords(date);
        int year = calendar.get(Calendar.YEAR);
        int month = calendar.get(Calendar.MONTH);
        int day = calendar.get(Calendar.DAY_OF_MONTH);
        if (nums.size() == 1 && words.size() == 1) {
            month = getMonthNumberFromMonthName(words.get(0));
            day = nums.get(0);
        }
        // dd.MM.yyyy
        StringBuilder sb = new StringBuilder();
        sb.append(day);
        sb.append('.');
        sb.append(month);
        sb.append('.');
        sb.append(year);
        return sb.toString();
    }

    public static int getMonthNumberFromMonthName(final String name) {
        String monthName = name;
        if (name.length() > MAX_STR_LEN_MONTH) {
            monthName = name.substring(0, MAX_STR_LEN_MONTH);
        }
        final String mnth = monthName.toLowerCase(Locale.getDefault());
        if (name2num.containsKey(mnth)) {
            return name2num.get(mnth);
        }
        return calendar.get(Calendar.MONTH);
    }

    public static Integer getMonthNumberFromMonthNameEng(final String name) {
        String monthName = name;
        if (name.length() > MAX_STR_LEN_MONTH) {
            monthName = name.substring(0, MAX_STR_LEN_MONTH);
        }
        return nameEng2num.get(monthName.toLowerCase(Locale.getDefault()));
    }

    public static void changeTimeFormat(
        final Map<String, Object> x,
        final String key)
    {
        if (x.containsKey(key)) {
            Object value = x.get(key);
            if (value instanceof String) {
                String newValue =
                    XTimeUtils.getTimeInIEXTimeFormat((String) value);
                x.put(key, newValue);
            }
        }
    }

    public static long getTimestampFromDateAndTime(
        final Map<String, Object> fl,
        final String dt,
        final String tm)
    {
        String time = XRegexpUtils.getTime(
            XJsonUtils.getStrValue(fl, tm));
        String date = XRegexpUtils.getDateInISO(
            XJsonUtils.getStrValue(fl, dt));
        int gmt = XRegexpUtils.getGmtInMinutes(
            XJsonUtils.getStrValue(fl, tm));
        return XRegexpUtils.getTimestamp(time, date, gmt);
    }

    public static long getTimestampFromDateAndTimeMask(
        final Map<?, ?> fl,
        final String tm,
        final String dt)
    {
        String time = XRegexpUtils.getTime(
            XJsonUtils.getStrValue(fl, tm));
        String date = XRegexpUtils.getDateInISO(
            XJsonUtils.getStrValue(fl, dt));
        int gmt = XRegexpUtils.getGmtInMinutes(
            XJsonUtils.getStrValue(fl, tm));
        return XRegexpUtils.getTimestamp(time, date, gmt);
    }

    @SuppressWarnings("StringSplitter")
    public static String minusDays(final String iso8601, final int n) {
        try {
            SimpleDateFormat format =
                new SimpleDateFormat("yyyy-MM-dd");
            format.setTimeZone(TIMEZONE);
            Date parsed = format.parse(XRegexpUtils.getDateInISO(iso8601));
            Calendar cal = Calendar.getInstance();
            cal.setTime(parsed);
            cal.add(Calendar.DAY_OF_YEAR, -n);
            return String.format(
                "%04d-%02d-%02d",
                cal.get(Calendar.YEAR),
                cal.get(Calendar.MONTH) + 1,
                cal.get(Calendar.DAY_OF_MONTH))
                + T
                + iso8601.split(T)[1];
        } catch (ParseException e) {
            throw new IllegalArgumentException();
        }
    }

    public static long getTimestampISO8601(final String iso8601) {
        String time = XRegexpUtils.getTime(iso8601);
        String date = XRegexpUtils.getDateInISO(iso8601);
        int gmt = XRegexpUtils.getGmtInMinutes(iso8601);
        return XRegexpUtils.getTimestamp(time, date, gmt);
    }

    public static String getTimeInIEXTimeFormat(final String anyTime) {
        // MM.dd.yyy hh.mm.ss
        String time = XRegexpUtils.getTime(anyTime);
        String date = XRegexpUtils.getDateInISO(anyTime);
        if (time.isEmpty() || date.isEmpty()) {
            return anyTime;
        }
        date = XRegexpUtils.toIEXDate(date);
        return date + ' ' + time;
    }

    public static long dayInSeconds() {
        return HOURS_IN_DAY
            * SEC_IN_HOUR;
    }

    public static String getDateTimestampMillisec(final String depTime)
        throws ParseException
    {
        SimpleDateFormat format = new SimpleDateFormat(TIME_FORMAT);
        format.setTimeZone(TIMEZONE);
        Date date = format.parse(depTime);
        return String.valueOf(date.getTime());
    }

    @SuppressWarnings("unused")
    public static String getHumanReadableDateFromMillisec(final long mill)
        throws ParseException
    {
        SimpleDateFormat format = new SimpleDateFormat(TIME_FORMAT);
        format.setTimeZone(TIMEZONE);
        return format.format(new Date(mill));
    }

    public static long getTimeFromTimestamp(final long timestampInSec) {
        return timestampInSec % dayInSeconds();
    }

    @SuppressWarnings("StringSplitter")
    public static int getTimeInSeconds(final String time) {
        int result = 0;
        if (time != null && !time.isEmpty() && XRegexpUtils.isStrTime(time)) {
            String[] pr = time.split(":");
            int cnv = SEC_IN_HOUR;
            for (final String x : pr) {
                result += Integer.parseInt(x) * cnv;
                cnv /= SEC_IN_MIN_OR_MIN_IN_HOUR;
            }
        }
        return result;
    }

    public static String timeToString(final int timeInSeconds) {
        int tsec = timeInSeconds;
        if (tsec < 0) {
            tsec += dayInSeconds();
        }
        int hours = tsec / SEC_IN_HOUR;
        int minutes = (tsec - hours * SEC_IN_HOUR) / SEC_IN_MIN_OR_MIN_IN_HOUR;
        int seconds = tsec % SEC_IN_MIN_OR_MIN_IN_HOUR;
        return String.format(
            "%02d:%02d:%02d",
            hours,
            minutes,
            seconds);
    }

    public static String getTimeFromTimeAndGMT(final String s) {
        int timeInSeconds = getTimeInSeconds(XRegexpUtils.getTime(s));
        int gmtInMinutes = XRegexpUtils.getGmtInMinutes(s);
        int minutesToAdd =
            XRegexpUtils.getTimeShiftOnMskInMinutes(gmtInMinutes);
        return timeToString(
            timeInSeconds + minutesToAdd * SEC_IN_MIN_OR_MIN_IN_HOUR);
    }

    public static String formatTime(final String time) {
        return XRegexpUtils.getTime(time);
    }

    @SuppressWarnings("StringSplitter")
    public static String formatDate(final String date) {
        String dateInIso = XRegexpUtils.getDateInISO(date);
        String[] dateParts = dateInIso.split(DASH);
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < dateParts.length; ++i) {
            if (i != 0) {
                sb.append('.');
            }
            sb.append(dateParts[dateParts.length - 1 - i]);
        }
        return sb.toString();
    }

    public static String getGtmInRfc3339(final int gtmIntMinutes) {
        int h = gtmIntMinutes / SEC_IN_MIN_OR_MIN_IN_HOUR;
        int m = gtmIntMinutes % SEC_IN_MIN_OR_MIN_IN_HOUR;
        if (h < 0) {
            return String.format("%03d:%02d", h, m);
        }
        return String.format("%02d:%02d", h, m);
    }

    public static int getDefaultGmtInMinutes() {
        return XRegexpUtils.MSK_GMT * XRegexpUtils.MININHOUR;
    }

    public interface TimeFormat {
        String getFormatedTime(
            final String date,
            final String time,
            final int timezone);
    }

    public static class TimeIexFormat implements TimeFormat {
        @Override
        @SuppressWarnings("StringSplitter")
        public String getFormatedTime(
            final String date,
            final String time,
            final int timezone)
        {
            String res = date;
            final String preres = date + time;
            final String[] ar = preres.split(
                ru.yandex.iex.proxy.tickethandlerlegacy
                    .TicketContext.SPACE_AS_STR);
            if (ar.length >= 2) {
                res = ar[0] + ' ' + ar[ar.length - 1];
            }
            return res;
        }
    }

    public static class TimeRfc3339Format implements TimeFormat {
        @Override
        public String getFormatedTime(
            final String date,
            final String time,
            final int timezoneInMinutes)
        {
            final String rdate = XRegexpUtils.getDateInISO(date);
            final String rtime = XTimeUtils.formatTime(time);
            String gtm = XTimeUtils.getGtmInRfc3339(timezoneInMinutes);
            StringBuilder sb = new StringBuilder();
            sb.append(rdate)
                .append('T')
                .append(rtime);
            if (!gtm.isEmpty() && !gtm.startsWith(DASH)) {
                sb.append('+');
            }
            sb.append(gtm);
            return new String(sb);
        }
    }

    public static String completeData(final String data) {
        String defaultYear = "1970";
        String defaultDays = "-1-1";
        String defaultTime = "T00:00:00";
        String res;
        if (data == null || data.isEmpty()) {
            res = defaultYear + defaultDays + defaultTime;
        } else if (data.length() == 2 + 2 && StringUtils.isDigit(data)) {
            res = data + defaultDays + defaultTime;
        } else if (StringUtils.isDigit(data.replace(DASH, ""))) {
            res = data + defaultTime;
        } else {
            res = defaultYear + defaultDays + defaultTime;
        }
        return res;
    }
}
