package ru.yandex.iex.proxy.cacheupdate;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.logging.Level;
import java.util.stream.Collectors;

import org.apache.http.Header;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.http.config.HttpHostConfig;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.iex.proxy.AbstractContext;
import ru.yandex.io.FlushOnCloseWriter;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;

public class UpdateFactCallback
    extends AbstractProxySessionCallback<SearchResult>
{
    private static final int INITIAL_BUF_SIZE = 1024;
    private static final String LOG_PREFIX = "UpdateFactCallback: ";
    private static final String PREFIX = "prefix";
    private static final String DOCS = "docs";
    private static final String FACT_DATA = "fact_data";
    private static final String FACT_NAME = "fact_name";
    private static final String FACT_MID = "fact_mid";

    private AbstractContext context;
    private String xIndexOperationQueueName;
    private PossibleUpdateCacheData possibleUpdateCacheData;
    private final FutureCallback<Object> callback;
    private BasicGenericConsumer<JsonObject, JsonException> jsonConsumer;
    private JsonParser parser;

    //CSOFF: ParameterNumber
    public UpdateFactCallback(
        final AbstractContext context,
        final String xIndexOperationQueueName,
        final PossibleUpdateCacheData possibleUpdateCacheData,
        final FutureCallback<Object> callback)
    {
        super(context.session());
        this.context = context;
        this.xIndexOperationQueueName = xIndexOperationQueueName;
        this.possibleUpdateCacheData = possibleUpdateCacheData;
        this.callback = callback;
        jsonConsumer = new BasicGenericConsumer<>();
        parser = TypesafeValueContentHandler.prepareParser(jsonConsumer);
    }
    //CSON: ParameterNumber

    @Override
    public void completed(final SearchResult result) {
        context.session().logger().info(LOG_PREFIX
            + "got result from hosts, size = " + result.hitsCount());
        StringBuilderWriter sbw = new StringBuilderWriter();

        List<SearchDocument> filteredResult =
            filterSearchResults(result.hitsArray());

        try (JsonWriter writer = new JsonWriter(sbw)) {
            boolean nonEmptyRequest = false;
            boolean isCurrentMidInDocs = false;
            writer.startObject();
            writer.key(PREFIX);
            String uid = Long.toString(possibleUpdateCacheData.uid());
            writer.value(uid);
            writer.key(DOCS);
            writer.startArray();
            for (int i = 0; i < filteredResult.size(); i++) {
                SearchDocument doc = filteredResult.get(i);
                context.session().logger().info(
                    LOG_PREFIX + "this doc is from post actions, process");
                String foundMid = doc.attrs().get(FACT_MID);
                if (foundMid.equals(possibleUpdateCacheData.mid())) {
                    isCurrentMidInDocs = true;
                }
                context.session().logger().info(LOG_PREFIX
                    + "pure fact data: " + doc.attrs().get(FACT_DATA));
                JsonMap factDataJsonMap = getFactDataJsonMap(doc);
                if (factDataJsonMap == null) {
                    continue;
                }
                boolean changed = false;
                for (Map.Entry<String, FieldModificationSettings> entry
                    : possibleUpdateCacheData.fieldSettings().entrySet())
                {
                    if (!entry.getValue().ignoreFirst()
                        && !entry.getValue().ignoreLast()
                        && foundMid.equals(possibleUpdateCacheData.mid()))
                    {
                        continue;
                    }
                    if (entry.getValue().ignoreFirst() && i == 0
                        || entry.getValue().ignoreLast()
                           && i == filteredResult.size() - 1
                           && isCurrentMidInDocs)
                    {
                        continue;
                    }
                    changed = true;
                    factDataJsonMap.put(
                        entry.getKey(),
                        entry.getValue().newValue());
                }
                if (!changed) {
                    continue;
                }
                nonEmptyRequest = true;
                String url = "facts_" + uid + '_' + foundMid
                    + '_' + doc.attrs().get(FACT_NAME);
                writer.startObject();
                writer.key("url");
                writer.value(url);
                factDataJsonMap.put(
                    possibleUpdateCacheData.modifiedMidFieldName(),
                    new JsonString(possibleUpdateCacheData.mid()));
                writer.key(FACT_DATA);
                writer.startString();
                try (JsonWriter writer2 =
                        new JsonWriter(new FlushOnCloseWriter(writer)))
                {
                    factDataJsonMap.writeValue(writer2);
                }
                writer.endString();
                writer.endObject();
            }
            writer.endArray();
            writer.endObject();
            if (nonEmptyRequest) {
                sendUpdateRequest(sbw);
            } else {
                callback.completed(null);
            }
        } catch (IOException e) {
            context.session().logger().log(
                Level.WARNING,
                "Failed to process response from cache",
                e);
        }
    }

    private List<SearchDocument> filterSearchResults(
        final List<SearchDocument> result)
    {
        // filter by searchField, checkFields
        String searchFieldName = possibleUpdateCacheData.searchFieldName();
        Set<Map.Entry<String, String>> checkFields =
            possibleUpdateCacheData.checkFields().entrySet();

        return result.stream().filter(doc -> {
            JsonMap factDataJsonMap = getFactDataJsonMap(doc);
            if (factDataJsonMap == null || !Objects.equals(
                factDataJsonMap.getString(searchFieldName, null),
                possibleUpdateCacheData.searchFieldValue()))
            {
                return false;
            }
            boolean suitable = true;
            for (Map.Entry<String, String> entry: checkFields) {
                if (!Objects.equals(
                    factDataJsonMap.getString(entry.getKey(), null),
                    entry.getValue()))
                {
                    suitable = false;
                    break;
                }
            }
            return suitable;
        }).collect(Collectors.toList());
    }

    private void sendUpdateRequest(final StringBuilderWriter sbw) {
        context.session().logger().info("UPDATE:" + sbw.toString());
        String queueName = context.iexProxy().
            config().factsIndexingQueueName();
        String uid = Long.toString(possibleUpdateCacheData.uid());
        String updateUri = "/update?service=" + queueName + "&facts"
            + "&prefix=" + uid;
        BasicAsyncRequestProducerGenerator generator =
            new BasicAsyncRequestProducerGenerator(
                updateUri,
                sbw.toString()
            );
        generator.addHeader(YandexHeaders.SERVICE, queueName);
        generator.addHeader(
            YandexHeaders.X_INDEX_OPERATION_TIMESTAMP,
            Long.toString(context.operationDateMillis()));
        if (xIndexOperationQueueName != null) {
            generator.addHeader(
                YandexHeaders.X_INDEX_OPERATION_QUEUE,
                xIndexOperationQueueName);
        }
        HttpRequest request = context.session().request();
        Header zooShardId = request.getFirstHeader(
            YandexHeaders.ZOO_SHARD_ID);
        if (zooShardId != null) {
            generator.addHeader(zooShardId);
        }
        AsyncClient producerAsyncClient = context.iexProxy()
            .producerAsyncClient().adjust(context.session().context());
        HttpHostConfig producerAsyncClientConfig = context.
            iexProxy().config().producerAsyncClientConfig();
        HttpHost host = producerAsyncClientConfig.host();
        producerAsyncClient.execute(
            host,
            generator,
            AsyncStringConsumerFactory.OK,
            context.session().listener().
                createContextGeneratorFor(producerAsyncClient),
            callback
        );
    }

    private JsonMap getFactDataJsonMap(final SearchDocument doc) {
        try {
            String factData = doc.attrs().get(FACT_DATA);
            int len = factData.length();
            char[] buf = new char[INITIAL_BUF_SIZE];
            if (len > buf.length) {
                buf = new char[Math.max(len, buf.length << 1)];
            }
            factData.getChars(0, len, buf, 0);
            parser.reset();
            parser.parse(buf, 0, len);
            return jsonConsumer.get().asMap();
        } catch (JsonException e) {
            context.session().logger().log(
                Level.WARNING,
                LOG_PREFIX + "doc processing failed",
                e);
            return null;
        }
    }
}
