package ru.yandex.iex.proxy.calendar;

import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.message.BasicHeader;

import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.config.ImmutableURIConfig;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.HttpExceptionConverter;
import ru.yandex.http.util.NotFoundException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseProducerGenerator;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncPostURIRequestProducerSupplier;
import ru.yandex.iex.proxy.IexProxy;
import ru.yandex.iex.proxy.XJsonUtils;
import ru.yandex.iex.proxy.XMessageToLog;
import ru.yandex.iex.proxy.eventtickethandler.EventCallback;
import ru.yandex.iex.proxy.eventtickethandler.EventTicketContext;
import ru.yandex.iex.proxy.eventtickethandler.EventTicketHandler;
import ru.yandex.iex.proxy.xutils.tjson.TJson;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncDomConsumerFactory;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.mail.search.MailSearchParams;
import ru.yandex.parser.email.types.MessageType;
import ru.yandex.parser.searchmap.User;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;

public class CalendarEventCallback extends EventCallback {
    private static final String ICS = "ics";
    private CalendarInfoParser parser = new CalendarInfoParser();
    private int retryCount = 1;
    private String request = null;

    public CalendarEventCallback(final EventCallback callback) {
        super(callback);
    }

    @Override
    public void act(final EventTicketContext context) {
        parser.init(context);
    }

    @Override
    public void execute(final EventTicketContext context) {
        this.context = context;
        act(context);
        if (parser.getIcsJson() == null) {
            String stid = context.getStid();

            if (retryCount > 0 && stid != null
                && context.getTypes().contains(MessageType.INVITE))
            {
                AsyncClient tikaClient =
                    context.iexProxy().tikaiteClient().adjust(context.session().context());

                User user = MailSearchParams.changeLogUser(context.prefix());
                List<HttpHost> indexerHosts =
                    context.iexProxy().searchMap().indexerHosts(user);
                if (indexerHosts.isEmpty()) {
                    String hostsNotFoundMessage =
                        "Indexer hosts is empty for uid: " + context.prefix()
                        + ", service: " + user.service() + ", shard: " + user.shard();
                    XMessageToLog.warning(context, hostsNotFoundMessage);
                    super.failed(new NotFoundException(hostsNotFoundMessage));
                } else {
                    tikaClient.execute(
                            indexerHosts,
                            tikaiteRequest(context),
                            JsonAsyncDomConsumerFactory.OK,
                            context.session().listener().createContextGeneratorFor(tikaClient),
                            new PushIcsAndRetryIcsEventCallback(context));
                }
            } else {
                completed(null);
            }
            return;
        }
        sendAttachSidRequest();
    }

    private BasicAsyncRequestProducerGenerator tikaiteRequest(
        final EventTicketContext context)
    {
        final StringBuilder uri = new StringBuilder();
        uri.append("/tikaite?json-type=dollar&stid=");
        uri.append(context.getStid());
        BasicAsyncRequestProducerGenerator producerGenerator =
            new BasicAsyncRequestProducerGenerator(new String(uri));
        producerGenerator.addHeader(
            YandexHeaders.X_YA_SERVICE_TICKET,
            context.iexProxy().tikaiteTvm2Ticket());
        producerGenerator.addHeader(
            YandexHeaders.X_SRW_SERVICE_TICKET,
            context.iexProxy().unistorageTvm2Ticket());
        return producerGenerator;
    }

    private void sendAttachSidRequest() {
        IexProxy iexProxy = context.iexProxy();
        AsyncClient client;
        ImmutableURIConfig config;
        String tvmTicket;
        long uid  = parser.uidLong();
        if (BlackboxUserinfo.corp(uid)) {
            client = iexProxy.corpAttachSidClient();
            config = iexProxy.config().corpAttachSidConfig();
            tvmTicket = iexProxy.corpAttachSidTvm2Ticket();
        } else {
            client = iexProxy.attachSidClient();
            config = iexProxy.config().attachSidConfig();
            tvmTicket = iexProxy.attachSidTvm2Ticket();
        }
        try {
            String uri = config.uri().toASCIIString();
            String body = getAttachSidRequestBody();
            ContentType contentType =
                ContentType.TEXT_PLAIN.withCharset(StandardCharsets.UTF_8);
            context.session().logger().info(
                "Send attach sid request: " + uri + ", with body: " + body);
            client.execute(
                new HeaderAsyncRequestProducerSupplier(
                    new AsyncPostURIRequestProducerSupplier(
                        uri,
                        body,
                        contentType),
                    new BasicHeader(
                        YandexHeaders.X_YA_SERVICE_TICKET,
                        tvmTicket)),
                JsonAsyncDomConsumerFactory.OK,
                context.session().listener().createContextGeneratorFor(client),
                new AttachSidCallback());
        } catch (URISyntaxException | IOException e) {
            context.session().logger().log(
                Level.SEVERE,
                "Attach sid request building error",
                e);
            completed(null);
        }
    }

    private String getAttachSidRequestBody() throws IOException {
        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = new JsonWriter(sbw)) {
            writer.startArray();
            writer.startObject();
            writer.key("uid");
            writer.value(context.uid());
            writer.key("downloads");
            writer.startArray();
            writer.startObject();
            writer.key("mid");
            writer.value(context.getMid());
            writer.key("hids");
            writer.startArray();
            writer.value(parser.getHid());
            writer.endArray();
            writer.endObject();
            writer.endArray();
            writer.endObject();
            writer.endArray();
        }
        return sbw.toString();
    }

    private void sendRequestToCalendar(final String sid) {
        if (context == null || sid == null) {
            completed(null);
            return;
        }
        try {
            long uid = parser.uidLong();
            ImmutableHttpHostConfig config;
            AsyncClient client;
            String webattach;
            String calendarRequest;
            String tvmTicket;
            if (BlackboxUserinfo.corp(uid)) {
                config = context.iexProxy().config().calendarToolsConfig();
                client = context.iexProxy().calendarToolsClient();
                webattach = "webattachcorp.mail.yandex.net";
                calendarRequest = "/api/mail/getEventInfoByIcsUrl?";
                tvmTicket = context.iexProxy().calendarToolsTvm2Ticket();
            } else {
                config = context.iexProxy().config().calendarConfig();
                client = context.iexProxy().calendarClient();
                webattach = "webattach.mail.yandex.net";
                calendarRequest = "/api/mail/importEventByIcsUrl?";
                tvmTicket = context.iexProxy().calendarTvm2Ticket();
            }
            HttpHost host = config.host();
            client = client.adjust(context.session().context());

            request = calendarRequest + "uid=" + uid
                  + "&icsUrl=http://" + webattach + "/message_part_real/?sid="
                  + sid + "&lang=" + parser.getLang();
            XMessageToLog.info(
                context,
                "Request to calendar: " + host + request);
            BasicAsyncRequestProducerGenerator producerGenerator =
                new BasicAsyncRequestProducerGenerator(request);
            producerGenerator.addHeader(
                YandexHeaders.X_YA_SERVICE_TICKET,
                tvmTicket);
            client.execute(
                host,
                producerGenerator,
                JsonAsyncDomConsumerFactory.OK,
                context.session().listener()
                    .createContextGeneratorFor(client),
                this);
        } catch (Exception e) {
            XMessageToLog.error(context, "sid error", e);
            completed(null);
        }
    }

    @Override
    public void completed(final Object result) {
        // calendar response is always with status code 200,
        // in case of error: error body in result
        if (context != null && result instanceof Map) {
            TJson jsonCrawler = new TJson(result);
            if (jsonCrawler.get("error") == null) {
                completedResponse(jsonCrawler);
            } else {
                failedResponse(jsonCrawler);
            }
        } else {
            if (callback != null) {
                callback.execute(context);
            }
        }
    }

    private void completedResponse(final TJson jsonCrawler) {
        if (context == null) {
            if (callback != null) {
                callback.execute(null);
            }
            return;
        }
        Map<String, Object> solution =
            parser.parseCalendarResponse(jsonCrawler);
        HashMap<String, Object> rawData = new HashMap<>();
        rawData.put("request", request);
        rawData.put(EventTicketHandler.DATA, jsonCrawler.getRoot().get("root"));
        try {
            XJsonUtils.putAll(parser.getIcsJson(), rawData);
        } catch (JsonUnexpectedTokenException ignored) {
        }
        solution.put(EventTicketHandler.RAW_DATA, rawData);
        context.setNewSolution(XJsonUtils.mergeJson(
            solution,
            context.getOutputJson()));
        CalendarFactMidsSearcher searcher = new CalendarFactMidsSearcher(
            context.iexProxy(),
            context.session(),
            new SearchOtherMidsCallback(context, callback));
        searcher.search(
            Long.parseLong(parser.getUid()),
            XJsonUtils.asStringOrNull(solution.get("externalEventId")),
            null);
    }

    private void failedResponse(final TJson jsonCrawler) {
        String errorName = XJsonUtils.asStringOrNull(
            jsonCrawler.get("error.name"));
        String errorMsg = XJsonUtils.asStringOrNull(
            jsonCrawler.get("error.message"));
        BasicAsyncResponseProducerGenerator generator =
            new BasicAsyncResponseProducerGenerator(
                HttpStatus.SC_BAD_REQUEST,
                "Error name: " + errorName + "; error message: "
                + errorMsg);
        HttpException ex = HttpExceptionConverter.toHttpException(
            new BasicAsyncRequestProducerGenerator(request),
            generator.get().generateResponse());
        XMessageToLog.error(
            context,
            "Request failed: " + request
            + ", error name: " + errorName
            + ", error message: " + errorMsg,
            ex);
        if (callback != null) {
            callback.execute(context);
        }
    }

    private class PushIcsAndRetryIcsEventCallback
        implements FutureCallback<Object>
    {
        private EventTicketContext context;

        PushIcsAndRetryIcsEventCallback(final EventTicketContext context) {
            this.context = context;
        }

        @Override
        public void cancelled() {
            XMessageToLog.warning(context, "Tikaite request cancelled");
            --retryCount;
            execute(context);
        }

        @Override
        public void failed(final Exception e) {
            XMessageToLog.error(context, "Tikaite request failed", e);
            --retryCount;
            execute(context);
        }

        @Override
        public void completed(final Object result) {
            --retryCount;
            if (!context.getInputJson().containsKey(ICS)) {
                String hid = findCalendarHid(result);
                if (hid != null) {
                    Map<String, Object> res = new HashMap<>();
                    res.put(EventTicketHandler.STID, context.getStid());
                    res.put(EventTicketHandler.LANG, "ru");
                    res.put(EventTicketHandler.UID, context.uid());
                    res.put(EventTicketHandler.HID, hid);
                    context.getInputJson().put(ICS, res);
                }
            }
            execute(context);
        }

        private String findCalendarHid(final Object res) {
            if (res instanceof List) {
                for (Object head: (List) res) {
                    String hid = getHidIfMimetypeCalendar(head);
                    if (hid != null) {
                        return hid;
                    }
                }
            }
            return null;
        }

        private String getHidIfMimetypeCalendar(final Object h) {
            if (h instanceof Map) {
                Map<?, ?> m = (Map<?, ?>) h;
                String mimetype = (String) m.get("mimetype");
                if (mimetype != null && mimetype.equals("text/calendar")) {
                    return (String) m.get(EventTicketHandler.HID);
                }
            }
            return null;
        }
    }

    private static class SearchOtherMidsCallback
        implements FutureCallback<SearchResult>
    {
        private EventTicketContext context;
        private EventCallback callback;

        SearchOtherMidsCallback(
            final EventTicketContext context,
            final EventCallback callback)
        {
            this.context = context;
            this.callback = callback;
        }

        @Override
        public void completed(final SearchResult result) {
            if (result != null && result.hitsCount() > 0) {
                SearchDocument firstDoc = result.hitsArray().get(0);
                String mid = firstDoc.attrs().get("fact_mid");
                if (!Objects.equals(mid, context.getMid())) {
                    XMessageToLog.info(context, "There is other msg " + mid
                        + " about the same calendar event,"
                        + " which is older than current " + context.getMid());
                    context.getOutputJson().put(
                        "calendarMailType",
                        "event_update");
                }
            }
            if (callback != null) {
                callback.execute(context);
            }
        }

        @Override
        public void failed(final Exception e) {
            XMessageToLog.error(context, "Search request failed", e);
            if (callback != null) {
                callback.execute(context);
            }
        }

        @Override
        public void cancelled() {
            if (callback != null) {
                callback.execute(context);
            }
        }
    }

    private final class AttachSidCallback implements FutureCallback<Object> {
        private ProxySession session;

        private AttachSidCallback() {
            this.session = context.session();
        }

        @Override
        public void completed(final Object response) {
            if (response instanceof Map) {
                Object errorObj = ((Map<?, ?>) response).get("error");
                if (errorObj instanceof Map) {
                    Object reason = ((Map<?, ?>) errorObj).get("reason");
                    if (reason instanceof String) {
                        session.logger().log(
                            Level.SEVERE,
                            "Attach sid request error: " + reason);
                        sendRequestToCalendar(null);
                        return;
                    }
                } else {
                    Object res = ((Map<?, ?>) response).get("result");
                    if (res instanceof List && !((List<?>) res).isEmpty()) {
                        Object first = ((List<?>) res).get(0);
                        if (first instanceof Map) {
                            Object sids = ((Map<?, ?>) first).get("sids");
                            if (sids instanceof List
                                && !((List<?>) sids).isEmpty()) {
                                Object sid = ((List<?>) sids).get(0);
                                if (sid instanceof String) {
                                    sendRequestToCalendar((String) sid);
                                    return;
                                }
                            }
                        }
                    }
                }
            }
            session.logger().log(Level.SEVERE, "Attach sid parsing error");
            sendRequestToCalendar(null);
        }

        @Override
        public void failed(final Exception e) {
            session.logger().log(Level.SEVERE, "Attach sid request error", e);
            sendRequestToCalendar(null);
        }

        @Override
        public void cancelled() {
            session.logger().log(Level.WARNING, "Attach sid request cancelled");
            sendRequestToCalendar(null);
        }
    }
}
