package ru.yandex.iex.proxy.calendar;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.ISODateTimeFormat;

import ru.yandex.iex.proxy.XJsonUtils;
import ru.yandex.iex.proxy.XMessageToLog;
import ru.yandex.iex.proxy.eventtickethandler.EventTicketContext;
import ru.yandex.iex.proxy.eventtickethandler.EventTicketHandler;
import ru.yandex.iex.proxy.xutils.tjson.TJson;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.util.string.StringUtils;

public class CalendarInfoParser {
    private static final String INFO = "eventInfo.";
    private static final String NAME = "name";
    private static final String ACTIONS = "actions";
    private static final String ATTENDEES = "attendees";
    private static final String CALENDAR_URL = "calendarUrl";
    private static final String DECISION = "decision";
    private static final String END_DATE_RFC = "end_date_rfc";
    private static final String END_DATE_TS = "end_date_ts";
    private static final String EXTERNAL_ID = "externalId";
    private static final String ICS = "ics";
    private static final String IS_ALL_DAY = "isAllDay";
    private static final String IS_CANCELLED = "isCancelled";
    private static final String LOCATION = "location";
    private static final String ORGANIZER = "organizer";
    private static final String PROD_ID = "prodId";
    private static final String RECURRENCE_EVENT_ID = "recurrenceEventId";
    private static final String RECURRENCE_ID = "recurrenceId";
    private static final String REPETITION_DESCRIPTION =
        "repetitionDescription";
    private static final String START_DATE_RFC = "start_date_rfc";
    private static final String START_DATE_TS = "start_date_ts";

    private static final DateTimeFormatter DATE_TIME_FORMATTER =
        ISODateTimeFormat.dateTimeNoMillis()
            .withZone(DateTimeZone.forID("Europe/Moscow"));

    private EventTicketContext context;
    private Object icsInJson = null;
    private String uid;
    private String stid;
    private String lang;
    private String hid;
    private Map<String, Object> solution = new HashMap<>();

    public void init(final EventTicketContext context) {
        this.context = context;
        Object icsJson = XJsonUtils.getIfKeyExists(context.getInputJson(), ICS);
        if (icsJson instanceof Map) {
            try {
                Map<?, ?> mp = ValueUtils.asMap(icsJson);
                uid = ValueUtils.asString(mp.get(EventTicketHandler.UID));
                stid = ValueUtils.asString(mp.get(EventTicketHandler.STID));
                lang = ValueUtils.asString(mp.get(EventTicketHandler.LANG));
                hid = ValueUtils.asString(mp.get(EventTicketHandler.HID));
                icsInJson = icsJson;
            } catch (JsonUnexpectedTokenException e) {
                XMessageToLog.error(context, "context.ics parse error", e);
            }
        }
    }

    public Map<String, Object> parseCalendarResponse(final TJson crawler) {
        if (!context.getOutputJson().containsKey(EventTicketHandler.SEATS)) {
            solution.put(EventTicketHandler.WIDGET_SUBTYPE, "calendar");
        }
        solution.put(EventTicketHandler.ORIGIN, ICS);
        putIfNotNull(EventTicketHandler.DOMAIN, context.getDomain());
        putIfNotNull("externalEventId", crawler.get(INFO + EXTERNAL_ID));
        putIfNotNull(RECURRENCE_EVENT_ID, crawler.get(INFO + RECURRENCE_ID));
        solution.putIfAbsent(RECURRENCE_EVENT_ID, "");
        putIfNotNull(EventTicketHandler.TITLE, crawler.get(INFO + NAME));
        putIfNotNull(START_DATE_TS, crawler.get(INFO + "start"));
        putIfNotNull(START_DATE_RFC, crawler.get(INFO + "startDt"));
        putIfNotNull(END_DATE_TS, crawler.get(INFO + "end"));
        putIfNotNull(END_DATE_RFC, crawler.get(INFO + "endDt"));
        putIfNotNull(IS_CANCELLED, crawler.get(INFO + IS_CANCELLED));
        putIfNotNull(CALENDAR_URL, crawler.get(INFO + CALENDAR_URL));
        putIfNotNull(PROD_ID, crawler.get(INFO + PROD_ID));
        putIfNotNull(
            REPETITION_DESCRIPTION,
            crawler.get(INFO + REPETITION_DESCRIPTION));
        putIfNotNull(IS_ALL_DAY, crawler.get(INFO + IS_ALL_DAY));
        putIfNotNull(ACTIONS, crawler.get(INFO + ACTIONS));
        putIfNotNull(DECISION, crawler.get(INFO + DECISION));

        Object attendees = crawler.get(INFO + ATTENDEES);
        putIfNotNull(ATTENDEES, attendees);
        putPeople(attendees);
        putLocation(crawler.get(INFO + LOCATION), attendees);

        putIfNotNull(ORGANIZER, crawler.get(INFO + ORGANIZER));

        Object latestInstanceTs = crawler.get(INFO + "latestInstanceTs");
        putIfNotNull("dateEventLastRepetitionTs", latestInstanceTs);
        if (latestInstanceTs instanceof Long) {
            putIfNotNull(
                "dateEventLastRepetition",
                DATE_TIME_FORMATTER.print((Long) latestInstanceTs));
        }
        Object dueTs = crawler.get(INFO + "dueTs");
        putIfNotNull("dateEventRepeatUntilTs", dueTs);
        if (dueTs instanceof Long) {
            putIfNotNull(
                "dateEventRepeatUntil",
                DATE_TIME_FORMATTER.print((Long) dueTs));
        }

        solution.put(
            EventTicketHandler.SPECIAL_PARTS,
            Collections.singletonList(hid));
        return solution;
    }

    private void putPeople(final Object attendees) {
        long attendeesSize = 0;
        if (attendees instanceof List) {
            for (Object attendee : (List) attendees) {
                if (!isAttendeeResource(attendee)) {
                    ++attendeesSize;
                }
            }
        }
        solution.put("people", attendeesSize + 1);
    }

    private void putLocation(final Object location, final Object attendees) {
        if (location instanceof String && !((String) location).isEmpty()) {
            solution.put(LOCATION, location);
        } else {
            List<String> locations = new ArrayList<>();
            if (attendees instanceof List) {
                for (Object attendee : (List) attendees) {
                    if (isAttendeeResource(attendee)) {
                        String name = getNameOrEmpty(attendee);
                        if (!name.isEmpty()) {
                            locations.add(name);
                        }
                    }
                }
            }
            solution.put(LOCATION, StringUtils.join(locations, ", "));
        }
    }

    private boolean isAttendeeResource(final Object attendee) {
        return Objects.equals(
            XJsonUtils.getNodeByPathOrNullEless(attendee, "type"),
            "resource");
    }

    private String getNameOrEmpty(final Object attendee) {
        Object name = XJsonUtils.getNodeByPathOrNullEless(attendee, NAME);
        if (name instanceof String) {
            return (String) name;
        }
        return "";
    }

    private void putIfNotNull(final String key, final Object value) {
        if (value != null) {
            solution.put(key, value);
        }
    }

    public Object getIcsJson() {
        return icsInJson;
    }

    public String getUid() {
        return uid;
    }

    public long uidLong() {
        return context.prefix();
    }

    public String getStid() {
        return stid;
    }

    public String getHid() {
        return hid;
    }

    public String getLang() {
        return lang;
    }
}
