package ru.yandex.iex.proxy.calendar;

import java.io.IOException;
import java.util.List;
import java.util.Locale;
import java.util.logging.Level;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.dbfields.ChangeType;
import ru.yandex.dbfields.OracleFields;
import ru.yandex.dbfields.PgFields;
import ru.yandex.function.GenericFunction;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.iex.proxy.IexProxy;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.mail.search.MailSearchParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;

public class CalendarLogHandler
    implements HttpAsyncRequestHandler<List<CalendarLog>>
{
    private static final String FACT_MID = "fact_mid";
    private static final String SERVICE = "service";
    private static final String PREFIX = "prefix";
    private static final String MDB = "mdb";
    private static final String PG = "pg";
    private static final double THOUSAND = 1000;

    private final IexProxy proxy;

    public CalendarLogHandler(final IexProxy proxy) {
        this.proxy = proxy;
    }

    @Override
    public HttpAsyncRequestConsumer<List<CalendarLog>> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException
    {
        if (!(request instanceof HttpEntityEnclosingRequest)) {
            throw new BadRequestException("Payload expected");
        }
        return new CalendarLogAsyncConsumer<>(
            ((HttpEntityEnclosingRequest) request).getEntity(),
            new LogConsumer(),
            StringCollectorsFactory.INSTANCE,
            BasicContainerFactory.INSTANCE,
            proxy);
    }

    @Override
    public void handle(
        final List<CalendarLog> payload,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        ProxySession session = new BasicProxySession(proxy, exchange, context);
        if (payload == null || payload.isEmpty()) {
            session.response(HttpStatus.SC_OK);
            return;
        }
        final MultiFutureCallback<Object> multiCallback =
            new MultiFutureCallback<>(
                new FinishMultiCallbacks(session));
        for (CalendarLog log: payload) {
            for (Long uid : log.getUids()) {
                SearchResultCallback callback = new SearchResultCallback(
                    uid,
                    session,
                    proxy,
                    multiCallback.newCallback());
                CalendarFactMidsSearcher searcher =
                    new CalendarFactMidsSearcher(proxy, session, callback);
                searcher.search(
                    uid,
                    log.getExternalId(),
                    log.getRecurrenceId());
            }
        }
        multiCallback.done();
    }

    private static class LogConsumer
        implements GenericFunction<JsonObject, CalendarLog, JsonException>
    {
        @Override
        public CalendarLog apply(final JsonObject jsonObject)
            throws JsonException
        {
            return CalendarLogParser.getInstance().parse(jsonObject);
        }
    }

    private static class SearchResultCallback
        implements FutureCallback<SearchResult>
    {
        private ProxySession session;
        private final Long uid;
        private final IexProxy proxy;
        private final FutureCallback<Object> callback;

        //CSOFF: ParameterNumber
        SearchResultCallback(
            final Long uid,
            final ProxySession session,
            final IexProxy proxy,
            final FutureCallback<Object> callback)
        {
            this.session = session;
            this.uid = uid;
            this.proxy = proxy;
            this.callback = callback;
        }
        //CSON: ParameterNumber

        @Override
        public void completed(final SearchResult result) {
            if (result != null && result.hitsArray().size() > 0) {
                SearchDocument lastDoc =
                    result.hitsArray().get(result.hitsArray().size() - 1);
                String url = getUrl();
                String body = getBody(lastDoc.attrs().get(FACT_MID));
                BasicAsyncRequestProducerGenerator generator =
                    new BasicAsyncRequestProducerGenerator(url, body);
                session.logger().info(
                    "Send notify request, url: " + url + " body: " + body);
                proxy.producerAsyncClient().execute(
                    proxy.config().producerAsyncClientConfig().host(),
                    generator,
                    EmptyAsyncConsumerFactory.ANY_GOOD,
                    callback);
            } else {
                session.logger().info(
                    "Search mids result is empty: " + result + ", uid: " + uid);
                callback.completed(null);
            }
        }

        @Override
        public void failed(final Exception e) {
            callback.failed(e);
        }

        @Override
        public void cancelled() {
            callback.cancelled();
        }

        private String getUrl() {
            QueryConstructor url = new QueryConstructor("/notify?");
            try {
                url.append(SERVICE, MailSearchParams.changeLogService(uid));
                url.append(PREFIX, uid);
                url.append(MDB, PG);
                url.append("update_cache", "event-ticket");
                url.append(
                    PgFields.CHANGE_TYPE,
                    ChangeType.IEX_UPDATE.name()
                        .toLowerCase(Locale.ROOT).replace('_', '-'));
            } catch (BadRequestException e) {
                proxy.logger().log(Level.SEVERE, "Error parsing param", e);
            }
            return url.toString();
        }

        private String getBody(final String mid) {
            double operationDate = System.currentTimeMillis() / THOUSAND;
            StringBuilderWriter sbw =
                new StringBuilderWriter(new StringBuilder());
            try (JsonWriter writer = new JsonWriter(sbw)) {
                writer.startObject();
                writer.key(PgFields.UID);
                writer.value(uid);
                writer.key(PgFields.CHANGE_TYPE);
                writer.value(ChangeType.IEX_UPDATE
                    .name().toLowerCase(Locale.ROOT).replace('_', '-'));
                writer.key(OracleFields.OPERATION_DATE);
                writer.value(operationDate);
                writer.key(PgFields.CHANGED);
                writer.startArray();
                writer.startObject();
                writer.key("mid");
                writer.value(mid);
                writer.endObject();
                writer.endArray();
                writer.endObject();
            } catch (IOException e) {
                proxy.logger().log(Level.SEVERE, "Request body build error", e);
            }
            return sbw.toString();
        }
    }

    private static class FinishMultiCallbacks
        extends AbstractProxySessionCallback<List<Object>>
    {
        FinishMultiCallbacks(final ProxySession session) {
            super(session);
        }

        @Override
        public void completed(final List<Object> results) {
            session.response(HttpStatus.SC_OK);
        }
    }
}
