package ru.yandex.iex.proxy.combomailhandler;

import java.io.IOException;
import java.util.Collections;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.iex.proxy.IexProxy;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.mail.search.MailSearchParams;
import ru.yandex.search.document.mail.MailMetaInfo;

public class ComboMailHandler implements HttpAsyncRequestHandler<JsonObject> {

    private final IexProxy proxy;

    public ComboMailHandler(final IexProxy proxy) throws IOException {
        this.proxy = proxy;
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException
    {
        if (!(request instanceof HttpEntityEnclosingRequest)) {
            throw new BadRequestException("Payload expected");
        }
        return new JsonAsyncTypesafeDomConsumer(
            ((HttpEntityEnclosingRequest) request).getEntity(),
            StringCollectorsFactory.INSTANCE,
            BasicContainerFactory.INSTANCE);
    }

    @Override
    public void handle(
        final JsonObject payload,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        ProxySession session = new BasicProxySession(proxy, exchange, context);
        String stid = session.params().getString(MailSearchParams.STID);
        Long uid = session.params().getLong(MailSearchParams.UID);

        if (!BlackboxUserinfo.corp(uid)) {
            BasicAsyncRequestProducerGenerator requestGenerator =
                new BasicAsyncRequestProducerGenerator(
                    "/mail/handler?json-type=dollar&ultra-fast-mode&stid="
                    + stid);
            requestGenerator.addHeader(
                YandexHeaders.X_YA_SERVICE_TICKET,
                proxy.tikaiteTvm2Ticket());
            requestGenerator.addHeader(
                YandexHeaders.X_SRW_SERVICE_TICKET,
                proxy.unistorageTvm2Ticket());
            AsyncClient client = proxy.tikaiteClient().adjust(context);
            client.execute(
                proxy.tikaiteHost(),
                requestGenerator,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                session.listener().createContextGeneratorFor(client),
                new TikaiteCallback(session, uid, stid, context));
        } else {
            session.response(
                HttpStatus.SC_OK,
                JsonType.NORMAL.toString(Collections.emptyMap()));
        }
    }

    private final class TikaiteCallback implements FutureCallback<JsonObject> {
        private final Logger logger;
        private final Long uid;
        private final String mid;
        private final String stid;
        private final ProxySession session;
        private final HttpContext context;

        // CSOFF: ParameterNumber
        private TikaiteCallback(
            final ProxySession session,
            final Long uid,
            final String stid,
            final HttpContext context)
            throws BadRequestException
        {
            this.logger = session.logger();
            this.session = session;
            this.uid = uid;
            this.stid = stid;
            this.mid = session.params().getString(MailSearchParams.MID);
            this.context = context;
        }
        // CSON: ParameterNumber

        @Override
        public void completed(final JsonObject result) {
            logger.info("in tikaite completed ");
            try {
                boolean isValidQuery = false;
                String from = "";
                String subject = "";
                String body = "";
                for (JsonObject docObj: result.get("docs").asList()) {
                    JsonMap doc = docObj.asMap();
                    final String contentType =
                        doc.getOrNull(MailMetaInfo.CONTENT_TYPE);
                    if ((contentType.equals("text/html")
                           || contentType.equals("text/plain"))
                           && doc.getOrNull(MailMetaInfo.ATTACHSIZE) == null)
                    {
                        isValidQuery = true;
                        from = doc.getOrNull("hdr_from_email").trim();
                        logger.info("from = " + from);
                        subject = doc.getOrNull("hdr_subject").trim();
                        logger.info("subject = " + subject);
                        body = doc.getOrNull("pure_body").trim();
                        logger.info("body = " + body);

                        if (!proxy.isValidMediaFiscal(from, body)) {
                            logger.info("ignoring fiscal from " + from);
                            isValidQuery = false;
                        }
                        break;
                    }
                }
                if (isValidQuery) {
                    session.logger().
                        info("Media: valid request, will send to media_wrapper");
                    StringBuilderWriter sbw =
                        new StringBuilderWriter(new StringBuilder(""));
                    try (JsonWriter writer = new JsonWriter(sbw)) {
                        writer.startObject();
                        writer.key("address");
                        writer.value(from);
                        writer.key("text");
                        writer.value(body);
                        writer.key("title");
                        writer.value(subject);
                        writer.endObject();

                        BasicAsyncRequestProducerGenerator generator =
                            new BasicAsyncRequestProducerGenerator("" + '/', sbw.toString());
                        generator.addHeader(
                            HttpHeaders.CONTENT_TYPE,
                            ContentType.APPLICATION_JSON.toString());
                        generator.addHeader(
                            HttpHeaders.ACCEPT,
                            ContentType.APPLICATION_JSON.toString());

                        AsyncClient mediaClient =
                            proxy.mediaClient().adjust(
                                session.context());
                        mediaClient.execute(
                            proxy.config().mediaConfig().host(),
                            generator,
                            JsonAsyncTypesafeDomConsumerFactory.OK,
                            session.listener().
                                createContextGeneratorFor(mediaClient),
                            new ErrorSuppressingFutureCallback<>(
                                new MediaResultCallback(session),
                                y -> RequestErrorType.HTTP,
                                JsonMap.EMPTY)
                        );

                    } catch (IOException e) {
                        session.logger().
                            info("Media: exception while processing media request");
                        session.response(
                                HttpStatus.SC_OK,
                                JsonType.NORMAL.toString(Collections.emptyMap()));
                    }

                } else {
                    session.logger().
                        info("Media: not valid query for " + from);
                    session.response(
                            HttpStatus.SC_OK,
                            JsonType.NORMAL.toString(null));
                }

            } catch (JsonException je) {
                failed(je);
            }
        }

        @Override
        public void failed(final Exception e) {
            logger.info("Tikaite failed");
            logger.log(Level.WARNING, "Tikaite failed with", e);
        }

        @Override
        public void cancelled() {
            logger.warning("Cancelled");
        }
    }

    private static class MediaResultCallback implements FutureCallback<JsonObject> {
        private final ProxySession session;

        MediaResultCallback(final ProxySession session)
        {
            this.session = session;
        }

        @Override
        public void cancelled() {
            session.logger().info("MediaResultCallback: cancelled ");
            session.response(
                    HttpStatus.SC_OK,
                    JsonType.NORMAL.toString(null));
        }

        @Override
        public void failed(final Exception e) {
            session.logger().info("MediaResultCallback: failed ");
            session.response(
                    HttpStatus.SC_OK,
                    JsonType.NORMAL.toString(null));
        }

        @Override
        public void completed(final JsonObject result) {
            session.logger().info("MediaResultCallback: completed with "
                    + JsonType.NORMAL.toString(result));
            try {
                JsonMap map = result.asMap();
                String name = map.getOrNull("name");
                String type = map.getOrNull("type");
                if (name != null && name.length() < 20 && type != null && type.length() < 30) {
                    session.logger().info("MediaResultCallback: valid response, save");
                    session.response(
                            HttpStatus.SC_OK,
                            JsonType.NORMAL.toString(result));
                } else {
                    if (name == null || type == null) {
                        session.logger().info("MediaResultCallback: name or type is absent, ignore");
                    } else {
                        session.logger().info("MediaResultCallback: too long response, ignore");
                    }
                    session.response(
                            HttpStatus.SC_OK,
                            JsonType.NORMAL.toString(null));
                }

            } catch (JsonBadCastException e) {
                session.logger().info("JsonBadCastException "
                    + "in constructing media response");
                session.response(
                        HttpStatus.SC_OK,
                        JsonType.NORMAL.toString(null));
            }
        }
    }

}

