package ru.yandex.iex.proxy.complaints;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import ru.yandex.client.so.shingler.config.ShinglerClientsConfig;
import ru.yandex.client.so.shingler.config.ShinglerClientsConfigBuilder;
import ru.yandex.http.config.AbstractURIConfigBuilder;
import ru.yandex.http.config.URIConfig;
import ru.yandex.logger.LoggerConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.DurationParser;
import ru.yandex.stater.StaterConfig;
import ru.yandex.stater.StaterConfigBuilder;

public class ComplaintsConfigBuilder extends AbstractURIConfigBuilder<ComplaintsConfigBuilder>
    implements ComplaintsConfig
{
    private Map<Route, File> rulesDictFiles;
    private Map<Route, Map<Integer, String>> rulesDictionaries;
    private LoggerConfigBuilder complLog;
    private LoggerConfigBuilder ytLog;
    private StaterConfigBuilder complaintsLagStaterConfig;
    private StaterConfigBuilder soSearchRequestsStaterConfig;
    private int dailyComplaintsLimit;
    private long messageExpirationPeriod;
    private boolean shinglersDryRun;
    private boolean shinglersAddStatByActions;
    private ShinglerClientsConfig shinglersConfig;
    private boolean useSolog;
    private boolean useSologger;

    @SuppressWarnings("unused")
    public ComplaintsConfigBuilder() {
        this(ComplaintsConfigDefaults.INSTANCE);
    }

    public ComplaintsConfigBuilder(final ComplaintsConfig config) {
        super(config);
        rulesDictFiles = config.rulesDictFiles();
        rulesDictionaries = config.rulesDictionaries();
        complLog = new LoggerConfigBuilder(config.complLog());
        ytLog = new LoggerConfigBuilder(config.ytLog());
        complaintsLagStaterConfig = new StaterConfigBuilder(config.complaintsLagStaterConfig());
        soSearchRequestsStaterConfig = new StaterConfigBuilder(config.soSearchRequestsStaterConfig());
        dailyComplaintsLimit = config.dailyComplaintsLimit();
        messageExpirationPeriod = config.messageExpirationPeriod();
        shinglersDryRun = config.shinglersDryRun();
        shinglersAddStatByActions = config.shinglersAddStatByActions();
        shinglersConfig = new ShinglerClientsConfigBuilder(config.shinglersConfig());
        useSolog = config.useSolog();
        useSologger = config.useSologger();
    }

    public ComplaintsConfigBuilder(final URIConfig config) {
        super(config);
        rulesDictFiles = ComplaintsConfigDefaults.INSTANCE.rulesDictFiles();
        rulesDictionaries = ComplaintsConfigDefaults.INSTANCE.rulesDictionaries();
        complLog = new LoggerConfigBuilder(ComplaintsConfigDefaults.INSTANCE.complLog());
        ytLog = new LoggerConfigBuilder(ComplaintsConfigDefaults.INSTANCE.ytLog());
        complaintsLagStaterConfig =
            new StaterConfigBuilder(ComplaintsConfigDefaults.INSTANCE.complaintsLagStaterConfig());
        soSearchRequestsStaterConfig =
            new StaterConfigBuilder(ComplaintsConfigDefaults.INSTANCE.soSearchRequestsStaterConfig());
        dailyComplaintsLimit = ComplaintsConfigDefaults.INSTANCE.dailyComplaintsLimit();
        messageExpirationPeriod = ComplaintsConfigDefaults.INSTANCE.messageExpirationPeriod();
        shinglersDryRun = ComplaintsConfigDefaults.INSTANCE.shinglersDryRun();
        shinglersAddStatByActions = ComplaintsConfigDefaults.INSTANCE.shinglersAddStatByActions();
        shinglersConfig = ComplaintsConfigDefaults.INSTANCE.shinglersConfig();
        useSolog = ComplaintsConfigDefaults.INSTANCE.useSolog();
        useSologger = ComplaintsConfigDefaults.INSTANCE.useSologger();
    }

    public ComplaintsConfigBuilder(final URIConfig config, final ComplaintsConfig complaintsConfig) {
        super(complaintsConfig == null ? config : complaintsConfig);
        if (config != null) {
            uri(config.uri());
        }
        if (complaintsConfig != null) {
            rulesDictFiles = complaintsConfig.rulesDictFiles();
            rulesDictionaries = complaintsConfig.rulesDictionaries();
            complLog = new LoggerConfigBuilder(complaintsConfig.complLog());
            ytLog = new LoggerConfigBuilder(complaintsConfig.ytLog());
            complaintsLagStaterConfig = new StaterConfigBuilder(complaintsConfig.complaintsLagStaterConfig());
            soSearchRequestsStaterConfig = new StaterConfigBuilder(complaintsConfig.soSearchRequestsStaterConfig());
            dailyComplaintsLimit = complaintsConfig.dailyComplaintsLimit();
            messageExpirationPeriod = complaintsConfig.messageExpirationPeriod();
            shinglersDryRun = complaintsConfig.shinglersDryRun();
            shinglersAddStatByActions = complaintsConfig.shinglersAddStatByActions();
            shinglersConfig = new ShinglerClientsConfigBuilder(complaintsConfig.shinglersConfig());
            useSolog = complaintsConfig.useSolog();
            useSologger = complaintsConfig.useSologger();
        }
    }

    public ComplaintsConfigBuilder(final IniConfig config)
        throws ConfigException
    {
        this(config, ComplaintsConfigDefaults.INSTANCE);
    }

    public ComplaintsConfigBuilder(final IniConfig config, final ComplaintsConfig defaults)
        throws ConfigException
    {
        super(config, defaults);
        rulesDictFiles = new HashMap<>(defaults.rulesDictFiles());
        rulesDictionaries = new ConcurrentHashMap<>(defaults.rulesDictionaries());
        for (final Route route: Route.values()) {
            File rulesDictFile = null;
            try {
                rulesDictFile = config.getInputFile(
                    "rules-dictionary.so-" + route.name().toLowerCase(Locale.ROOT) + ".file",
                    null);
            } catch (Exception e) {
                System.err.println("ComplaintsConfigBuilder failed to get dictionary file route=" + route.name());
                /* do nothing */
            }
            if (rulesDictFile != null) {
                rulesDictFiles.put(route, rulesDictFile);
            }
            if (rulesDictFiles.containsKey(route) && rulesDictFiles.get(route) != null) {
                ConcurrentHashMap<Integer, String> rulesDictionary = loadRulesDictionary(rulesDictFiles.get(route));
                rulesDictionaries.put(route, rulesDictionary);
            } else if (!rulesDictionaries.containsKey(route) || rulesDictionaries.get(route) == null) {
                System.err.println("ComplaintsConfigBuilder failed to find dictionary file for route=" + route.name());
            }
        }
        complLog = new LoggerConfigBuilder(config.section("compl-log"), defaults.complLog());
        ytLog = new LoggerConfigBuilder(config.section("yt-log"), defaults.ytLog());
        complaintsLagStaterConfig =
            new StaterConfigBuilder(config.section("lag.stat"), defaults.complaintsLagStaterConfig());
        soSearchRequestsStaterConfig =
            new StaterConfigBuilder(config.section("search.stat"), defaults.soSearchRequestsStaterConfig());
        dailyComplaintsLimit = config.getInt("daily-complaints-limit", defaults.dailyComplaintsLimit());
        messageExpirationPeriod =
            config.get("message-expiration-period", defaults.messageExpirationPeriod(), DurationParser.POSITIVE_LONG);
        shinglersDryRun = config.getBoolean("shinglers.dry-run", defaults.shinglersDryRun());
        shinglersAddStatByActions = config.getBoolean("shinglers.add-stat-by-actions", defaults.shinglersDryRun());
        shinglersConfig = new ShinglerClientsConfigBuilder(
            config.section(ShinglerClientsConfigBuilder.SECTION),
            defaults.shinglersConfig());
        useSolog = config.getBoolean("solog", defaults.useSolog());
        useSologger = config.getBoolean("sologger", defaults.useSologger());
    }

    public static Map<Route, Map<Integer, String>> loadRulesDictionaries(final Map<Route, File> rulesDictFilePaths)
        throws ConfigException
    {
        Map<Route, Map<Integer, String>> rulesDictionaries =
            new ConcurrentHashMap<>(ComplaintsConfigDefaults.INSTANCE.rulesDictionaries());
        for (final Route route: Route.values()) {
            if (rulesDictFilePaths != null && rulesDictFilePaths.containsKey(route)) {
                ConcurrentHashMap<Integer, String> rulesDictionary = loadRulesDictionary(rulesDictFilePaths.get(route));
                rulesDictionaries.put(route, rulesDictionary);
            }
        }
        return rulesDictionaries;
    }

    @SuppressWarnings("StringSplitter")
    private static ConcurrentHashMap<Integer, String> loadRulesDictionary(final File file)
        throws ConfigException
    {
        System.err.println("loadRulesDictionary: loading file=" + file);
        if (file != null) {
            try (BufferedReader reader =
                new BufferedReader(new InputStreamReader(new FileInputStream(file), StandardCharsets.UTF_8));
            ) {
                ConcurrentHashMap<Integer, String> rules = new ConcurrentHashMap<>();
                String line;
                while ((line = reader.readLine()) != null) {
                    line = line.trim();
                    if (line.startsWith("#") || line.startsWith(";") || line.startsWith("\\\\") || line.isEmpty()) {
                        continue;
                    }
                    try {
                        final String[] fields = line.split("[ \t]+");
                        if (fields.length > 0 && !fields[0].isEmpty() && !fields[1].isEmpty()) {
                            rules.put(Integer.parseInt(fields[0]), fields[1]);
                        }
                    } catch (NumberFormatException e) {
                        throw new ConfigException("Cannot parse rule's index for line: " + line, e);
                    }
                }
                return rules;
            } catch (IOException e) {
                throw new ConfigException("Cannot load rules dictionary from file: " + file, e);
            }
        }
        return new ConcurrentHashMap<>();
    }

    @Override
    public Map<Route, File> rulesDictFiles() {
        return rulesDictFiles;
    }

    public ComplaintsConfigBuilder rulesDictFiles(final Map<Route, File> rulesDictFiles) {
        this.rulesDictFiles = new HashMap<>(rulesDictFiles);
        return self();
    }

    @Override
    public Map<Route, Map<Integer, String>> rulesDictionaries() {
        return rulesDictionaries;
    }

    public ComplaintsConfigBuilder rulesDictionaries(final Map<Route, Map<Integer, String>> rulesDictionaries) {
        this.rulesDictionaries = new HashMap<>(rulesDictionaries);
        return self();
    }

    @Override
    public LoggerConfigBuilder complLog() {
        return complLog;
    }

    @SuppressWarnings("unused")
    public void complLog(LoggerConfigBuilder complLog) {
        this.complLog = complLog;
    }

    @Override
    public LoggerConfigBuilder ytLog() {
        return ytLog;
    }

    @SuppressWarnings("unused")
    public void ytLog(LoggerConfigBuilder ytLog) {
        this.ytLog = ytLog;
    }

    @Override
    public ComplaintsConfigBuilder self() {
        return this;
    }

    public ImmutableComplaintsConfig build() throws ConfigException
    {
        return new ImmutableComplaintsConfig(this);
    }

    @Override
    public StaterConfigBuilder complaintsLagStaterConfig() {
        return complaintsLagStaterConfig;
    }

    @SuppressWarnings("unused")
    public ComplaintsConfigBuilder complaintsLagStaterConfig(final StaterConfig complaintsLagStaterConfig) {
        this.complaintsLagStaterConfig = new StaterConfigBuilder(complaintsLagStaterConfig);
        return self();
    }

    @Override
    public StaterConfigBuilder soSearchRequestsStaterConfig() {
        return soSearchRequestsStaterConfig;
    }

    public ComplaintsConfigBuilder soSearchRequestsStaterConfig(final StaterConfig soSearchRequestsStaterConfig) {
        this.soSearchRequestsStaterConfig = new StaterConfigBuilder(soSearchRequestsStaterConfig);
        return self();
    }

    @Override
    public int dailyComplaintsLimit() {
        return dailyComplaintsLimit;
    }

    public ComplaintsConfigBuilder dailyComplaintsLimit(final int dailyComplaintsLimit) {
        this.dailyComplaintsLimit = dailyComplaintsLimit;
        return this.self();
    }

    @Override
    public long messageExpirationPeriod() {
        return messageExpirationPeriod;
    }

    public ComplaintsConfigBuilder messageExpirationPeriod(final long messageExpirationPeriod) {
        this.messageExpirationPeriod = messageExpirationPeriod;
        return this.self();
    }

    @Override
    public boolean shinglersDryRun() { return shinglersDryRun; }

    public ComplaintsConfigBuilder shinglersDryRun(final boolean shinglersDryRun) {
        this.shinglersDryRun = shinglersDryRun;
        return this.self();
    }

    @Override
    public boolean shinglersAddStatByActions() {
        return shinglersAddStatByActions;
    }

    public ComplaintsConfigBuilder shinglersAddStatByActions(final boolean shinglersAddStatByActions) {
        this.shinglersAddStatByActions = shinglersAddStatByActions;
        return this.self();
    }

    @Override
    public ShinglerClientsConfig shinglersConfig() {
        return shinglersConfig;
    }

    public ComplaintsConfigBuilder shinglersConfig(final ShinglerClientsConfig shinglersConfig) {
        this.shinglersConfig = new ShinglerClientsConfigBuilder(shinglersConfig);
        return self();
    }

    @Override
    public boolean useSolog() {
        return useSolog;
    }

    public ComplaintsConfigBuilder useSolog(final boolean useSolog) {
        this.useSolog = useSolog;
        return this.self();
    }

    @Override
    public boolean useSologger() {
        return useSologger;
    }

    public ComplaintsConfigBuilder useSologger(final boolean useSologger) {
        this.useSologger = useSologger;
        return this.self();
    }
}
