package ru.yandex.iex.proxy.complaints;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.logging.Level;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.james.mime4j.MimeException;
import org.apache.james.mime4j.stream.BodyDescriptor;
import org.apache.james.mime4j.stream.EntityState;
import org.apache.james.mime4j.stream.Field;
import org.apache.james.mime4j.stream.RecursionMode;
import org.apache.james.mime4j.util.MimeUtil;

import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.function.ByteArrayProcessable;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.iex.proxy.AbstractEntityContext;
import ru.yandex.iex.proxy.IexProxy;
import ru.yandex.iex.proxy.move.UpdateDataHolder;
import ru.yandex.io.IOStreamUtils;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.mail.mime.BodyDecoder;
import ru.yandex.parser.mail.errors.ErrorInfo;
import ru.yandex.parser.mail.received.ReceivedChainParser;
import ru.yandex.parser.mail.senders.SenderInfo;
import ru.yandex.parser.mail.senders.SendersContext;
import ru.yandex.search.document.mail.MailMetaInfo;
import ru.yandex.search.document.mail.SafeMimeTokenStream;
import ru.yandex.util.timesource.TimeSource;

public class EMLContext extends AbstractEntityContext implements AbstractUserActionContext {
    public static final String EML = "eml";

    private final static String FBL_ADDR = "fbl-arf@yandex.ru";
    private static final Pattern RE_SPECIAL_MAILLIST =
        Pattern.compile("(exchange|crm)-(spam|ham)-reports?@(?:mail\\.)?yandex-team.ru");
    private static final Pattern RE_SO_MAILLIST = Pattern.compile("so@(?:mail\\.)?yandex-team.ru");
    private final String eml;
    private MailMessageContext msgContext = null;
    private Long mid;
    private String suid = null;
    private String stid;
    private Long karma = null;
    private String login = null;
    private String sourceDomain = null;
    private Sources source = null;
    private final long complaintDate;
    private final Map<String, List<String>> headersMap;
    private Boolean showTabs = null;
    private String folder;
    private List<SenderInfo> senders = null;
    protected final long startTime;

    public EMLContext(final IexProxy iexProxy, final ProxySession session, final Map<?, ?> json)
        throws HttpException, JsonUnexpectedTokenException
    {
        super(iexProxy, session, json);
        startTime = TimeSource.INSTANCE.currentTimeMillis();
        mid = session.params().getLong(MailIndexFields.MID, 0L);
        stid = session.params().getString(MailIndexFields.STID, "");
        complaintDate =
            session.params().getLong(MailIndexFields.RECEIVED_DATE, startTime / MILLIS);
        folder = session.params().getString(MailIndexFields.FOLDER_NAME);
        headersMap = new HashMap<>();
        msgContext = new MailMessageContext(this, UserAction.OTHER, complaintDate);
        Object emlObject = json.get(EML);
        if (emlObject instanceof Map) {
            Map<?, ?> eml = ValueUtils.asMap(emlObject);
            Object msg = eml.get("message_body");
            if (msg != null) {
                this.eml = ValueUtils.asString(msg);
                parseEML();
            } else {
                this.eml = null;
            }
        } else {
            this.eml = null;
        }
    }

    @Override
    public long startTime() {
        return startTime;
    }

    public String eml() {
        return eml;
    }

    public MailMessageContext msgContext() {
        return msgContext;
    }

    public Long mid() {     // mid of forwarded message, not original
        return mid;
    }

    public String stid() {  // stid of forwarded message, not original
        return stid;
    }

    @Override
    public String suid() {
        return suid;
    }

    @Override
    public void setSuid(final String suid) {
        this.suid = suid;
    }

    public void setUid(final Long uid) {
        this.uid = uid;
    }

    @Override
    public Long karma() {
        return karma;
    }

    @Override
    public void setKarma(final Long karma) {
        this.karma = karma;
    }

    @Override
    public long actionDate() {
        return complaintDate;
    }

    @Override
    public String login() {
        return login;
    }

    @Override
    public void setLogin(final String login) {
        this.login = login;
    }

    @Override
    public Sources source() {
        return source;
    }

    @Override
    public UserAction action() {
        return msgContext.action();
    }

    public void setAction(final UserAction action) {
        msgContext.setAction(action);
    }

    public String sourceDomain() {
        return sourceDomain;
    }

    @Override
    public Boolean showTabs() {
        return showTabs;
    }

    @Override
    public void setShowTabs(final boolean showTabs) {
        this.showTabs = showTabs;
    }

    @Override
    public Map<Long, UpdateDataHolder> messages() {
        final Map<Long, UpdateDataHolder> messages = new HashMap<>();
        final long mid = msgContext.mid() == null ? 0L : msgContext.mid();
        messages.put(mid, new UpdateDataHolder(msgContext, sourceDomain, senders, source));
        return messages;
    }

    @Override
    public void messages(final List<UpdateDataHolder> docs) {
    }

    public List<SenderInfo> senders() {
        return senders;
    }

    @Override
    public Map<String, Long> recipients() {
        return msgContext.recipients();
    }

    @Override
    public Map<String, Long> senderUids() {
        final Map<String, Long> senderUids = new HashMap<>();
        senderUids.put(msgContext.senderEmail(), msgContext.senderUid());
        return senderUids;
    }

    @Override
    public Map<UserAction, List<Long>> actions() {
        final Map<UserAction, List<Long>> actions = new HashMap<>();
        actions.put(msgContext.action(), List.of(msgContext.mid()));
        return actions;
    }

    @Override
    public void response() {
        String response = "{\"delivered\":1}";
        try {
            JsonMap responseObj = TypesafeValueContentHandler.parse(response).asMap();
            if (source != null) {
                responseObj.put("source_type", new JsonString(source.name()));
            }
            if (msgContext.action() != null) {
                responseObj.put("spam_type", new JsonString(msgContext.action().name()));
            }
            if (sourceDomain != null) {
                responseObj.put("source_domain", new JsonString(sourceDomain));
            }
            if (msgContext.senderHost() != null) {
                responseObj.put("sender_host", new JsonString(msgContext.senderHost()));
            }
            if (msgContext.senderEmail() != null) {
                responseObj.put("sender_email", new JsonString(msgContext.senderEmail()));
            }
            if (msgContext.recipientEmail() != null) {
                responseObj.put("recipient_email", new JsonString(msgContext.recipientEmail()));
            }
            long msgDate = msgContext.action() == UserAction.OTHER ? msgContext.actionDate() : msgContext.messageDate();
            if (msgDate != 0) {
                responseObj.put("received_date", new JsonLong(msgDate));
            }
            response = JsonType.NORMAL.toString(responseObj);
        } catch(JsonException e) {
            session.logger().log(Level.SEVERE, "EMLContext.response failed to construct full response", e);
        }
        session.response(HttpStatus.SC_OK, response);
    }

    private void parseEML() {
        SafeMimeTokenStream tokenStream = new SafeMimeTokenStream();
        tokenStream.setRecursionMode(RecursionMode.M_NO_RECURSE);
        try (ByteArrayInputStream in = new ByteArrayInputStream(this.eml.getBytes(StandardCharsets.UTF_8))) {
            tokenStream.parse(in);
            EntityState state = tokenStream.getState();
            MailMetaInfo msgMeta = new MailMetaInfo(-1, -1, iexProxy.yandexNets());
            String fieldName;
            while (state != EntityState.T_END_OF_STREAM) {
                if (state == EntityState.T_FIELD) {
                    Field field = tokenStream.getField();
                    msgMeta.add(field);
                } else if (state == EntityState.T_BODY) {
                    BodyDescriptor bd = tokenStream.getBodyDescriptor();
                    if (MimeUtil.isMessage(bd.getMimeType())) {
                        MailMetaInfo entityMeta = new MailMetaInfo(-1, -1, iexProxy.yandexNets());
                        try (InputStream is = BodyDecoder.INSTANCE.apply(
                                tokenStream.getInputStream(),
                                bd.getTransferEncoding()))
                        {
                            ByteArrayProcessable result = IOStreamUtils.consume(is).toByteArrayProcessable();
                            StringBuilder sb = new StringBuilder();
                            sb.append('\n');
                            SafeMimeTokenStream entityStream = new SafeMimeTokenStream();
                            String to = msgMeta.get(MailMetaInfo.HDR + MailMetaInfo.TO + MailMetaInfo.NORMALIZED);
                            if (to == null) {
                                session.logger().warning(
                                    "EMLContext.parseEML: absent To header => unable to determine type of message");
                                return;
                            }
                            to = to.trim();
                            setUpSourceAndSpamType(to);
                            ReceivedChainParser chainParser = new ReceivedChainParser(iexProxy.yandexNets(), source == Sources.FBL);
                            entityStream.parse(result.content());
                            while (entityStream.getState() != EntityState.T_END_HEADER) {
                                if (entityStream.next() == EntityState.T_FIELD) {
                                    Field field = entityStream.getField();
                                    entityMeta.add(field);
                                    fieldName = field.getName().toLowerCase(Locale.ROOT);
                                    sb.append(fieldName);
                                    sb.append(':');
                                    sb.append(' ');
                                    sb.append(field.getBody());
                                    sb.append('\n');
                                    headersMap.computeIfAbsent(fieldName, x -> new ArrayList<>())
                                        .add(field.getBody().trim());
                                    if (fieldName.equals(MailMetaInfo.RECEIVED)) {
                                        chainParser.process(field.getBody());
                                    }
                                }
                            }
                            ErrorInfo errorInfo = chainParser.errorInfo();
                            if (errorInfo != null) {
                                session.logger().warning("Failed to parse Received chain: " + errorInfo);
                            }
                            try {
                                msgContext.headersMap(headersMap).mailMetaInfo(entityMeta);
                                session.logger().info("EMLContext.parseEML: queueid=" + msgContext.queueId()
                                    + ", smtpId=" + chainParser.yandexSmtpId() + ", recipient="
                                    + chainParser.recipients());
                                senders = new SendersContext(entityMeta.getSender(), new String(sb)).extractSenders();
                                if (msgContext.queueId() == null || msgContext.queueId().isEmpty()) {
                                    msgContext.setQueueId(chainParser.yandexSmtpId());
                                    msgContext.allSmtpIds().addAll(chainParser.allYandexSmtpIds());
                                }
                                if (chainParser.fullSenderHost() != null) {
                                    msgContext.setSenderHost(chainParser.fullSenderHost());
                                    if (MailMessageContext.isYandexEmail(msgContext.senderEmail())) {
                                        msgContext.setSenderHost("yandex.net");
                                    }
                                }
                                if ((msgContext.recipientEmail() == null || msgContext.recipientEmail().isEmpty())
                                        && !chainParser.recipients().isEmpty())
                                {
                                    msgContext.setRecipientEmail(chainParser.recipients().iterator().next());
                                }
                                if (chainParser.sourceDomain() != null) {
                                    sourceDomain = chainParser.sourceDomain();
                                }
                                msgContext.setFolder(folder);
                                msgContext.setSeen(true);
                            } catch (Exception e) {
                                session.logger().log(Level.SEVERE, "EMLContext.parseEML exception", e);
                            }
                            session.logger().info("EMLContext.parseEML: To: " + to + ". SenderHost: "
                                + msgContext.senderHost() + ". SenderEmail: " + msgContext.senderEmail()
                                + ". RecipientEmail: " + msgContext.recipientEmail() + ". SenderType: "
                                + source.name() + ". QueueID: " + msgContext.queueId() + ". MsgID: "
                                + msgContext.msgId());
                        } catch(IOException | MimeException e) {
                            session.logger().log(Level.SEVERE, "EMLContext.parseEML failed to parse msg entity", e);
                        }
                        return;
                    }
                }
                state = tokenStream.next();
            }
        } catch (IOException | MimeException e) {
            session.logger().log(Level.SEVERE, "EMLContext.parseEML failed to parse EML", e);
        }
    }

    private void setUpSourceAndSpamType(String to) {
        source = Sources.UNKNOWN;
        if (to.equals(FBL_ADDR)) {
            source = Sources.FBL;
            msgContext.setAction(UserAction.SPAM);
            msgContext.setSoRes("ham");
        } else {
            Matcher mto = RE_SPECIAL_MAILLIST.matcher(to);
            if (mto.find()) {
                if (mto.group(1).equals("exchange")) {
                    source = Sources.EXCHANGE;
                } else if (mto.group(1).equals("crm")) {
                    source = Sources.CRM;
                }
                if (mto.group(2).equals("spam")) {
                    msgContext.setAction(UserAction.SPAM);
                } else if (mto.group(2).equals("ham")) {
                    msgContext.setAction(UserAction.HAM);
                }
            } else {
                Matcher soto = RE_SO_MAILLIST.matcher(to);
                if (soto.find()) {
                    source = Sources.SO_MAILLIST;
                    msgContext.setAction(UserAction.SPAM);
                }
            }
        }
        if (source != Sources.UNKNOWN) {
            msgContext.setSource(source);
            msgContext.skipReasons().remove(SkipReason.AUTOMATION);
        }
    }
}
