package ru.yandex.iex.proxy.complaints;

import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.blackbox.BlackboxDbfield;
import ru.yandex.blackbox.BlackboxUserIdType;
import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.blackbox.BlackboxUserinfoRequest;
import ru.yandex.blackbox.SingleUserinfoBlackboxCallback;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.iex.proxy.AbstractEntityCallback;
import ru.yandex.iex.proxy.AbstractEntityHandler;
import ru.yandex.iex.proxy.IexProxy;
import ru.yandex.iex.proxy.move.UpdateDataExecutor;
import ru.yandex.iex.proxy.move.UpdateDataHolder;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.search.document.mail.MailMetaInfo;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;

public class EMLHandler extends AbstractEntityHandler<EMLContext> {
    public static final Pattern YANDEX_CORP_DOMAIN = Pattern.compile("(?i)(?:mail\\.)?yandex-team\\.[a-z]+$");;

    private final TimeFrameQueue<Long> crmComplaints;
    private final TimeFrameQueue<Long> exchangeSpamComplaints;
    private final TimeFrameQueue<Long> exchangeHamComplaints;
    private final TimeFrameQueue<Long> fblComplaints;
    private final TimeFrameQueue<Long> soMaillistComplaints;
    private final TimeFrameQueue<Long> unknownComplaints;

    public EMLHandler(final IexProxy iexProxy) {
        super(iexProxy);
        crmComplaints = new TimeFrameQueue<>(iexProxy.config().metricsTimeFrame());
        exchangeSpamComplaints = new TimeFrameQueue<>(iexProxy.config().metricsTimeFrame());
        exchangeHamComplaints = new TimeFrameQueue<>(iexProxy.config().metricsTimeFrame());
        fblComplaints = new TimeFrameQueue<>(iexProxy.config().metricsTimeFrame());
        soMaillistComplaints = new TimeFrameQueue<>(iexProxy.config().metricsTimeFrame());
        unknownComplaints = new TimeFrameQueue<>(iexProxy.config().metricsTimeFrame());
        iexProxy.registerStater(
            new PassiveStaterAdapter<>(
                crmComplaints,
                new NamedStatsAggregatorFactory<>("complaints-crm-spam_ammm", CountAggregatorFactory.INSTANCE)));
        iexProxy.registerStater(
            new PassiveStaterAdapter<>(
                exchangeSpamComplaints,
                new NamedStatsAggregatorFactory<>(
                    "complaints-exchange-spam_ammm",
                    CountAggregatorFactory.INSTANCE)));
        iexProxy.registerStater(
            new PassiveStaterAdapter<>(
                exchangeHamComplaints,
                new NamedStatsAggregatorFactory<>(
                    "complaints-exchange-ham_ammm",
                    CountAggregatorFactory.INSTANCE)));
        iexProxy.registerStater(
            new PassiveStaterAdapter<>(
                fblComplaints,
                new NamedStatsAggregatorFactory<>("complaints-fbl-spam_ammm", CountAggregatorFactory.INSTANCE)));
        iexProxy.registerStater(
            new PassiveStaterAdapter<>(
                soMaillistComplaints,
                new NamedStatsAggregatorFactory<>(
                    "complaints-so-maillist-spam_ammm",
                    CountAggregatorFactory.INSTANCE)));
        iexProxy.registerStater(
            new PassiveStaterAdapter<>(
                unknownComplaints,
                new NamedStatsAggregatorFactory<>("complaints-unknown_ammm", CountAggregatorFactory.INSTANCE)));
    }

    @Override
    protected EMLContext createContext(final IexProxy iexProxy, final ProxySession session, final Map<?, ?> json)
        throws HttpException, JsonUnexpectedTokenException
    {
        return new EMLContext(iexProxy, session, json);
    }

    @Override
    protected void handle(final EMLContext context)
    {
        if (context.eml() == null) {
            context.session().logger().warning("EMLHandler: no message_body found");
            context.response();
        } else {
            if (context.source() == null || context.source() == Sources.UNKNOWN
                    || context.msgContext().action() == UserAction.OTHER)
            {
                context.session().logger().warning("EMLHandler: undetermined complaint's source type!");
                context.response();
            } else {
                context.session().logger().info("EMLHandler.handle: sourceType=" + context.source());
                switch (context.source()) {
                    case CRM:
                        crmComplaints(1);
                        break;
                    case EXCHANGE:
                        if (context.msgContext().action() == UserAction.SPAM) {
                            exchangeSpamComplaints(1);
                        } else if (context.msgContext().action() == UserAction.HAM) {
                            exchangeHamComplaints(1);
                        }
                        break;
                    case FBL:
                        fblComplaints(1);
                        break;
                    case SO_MAILLIST:
                        soMaillistComplaints(1);
                        break;
                    default:
                        unknownComplaints(1);
                }
                if (context.source() == Sources.EXCHANGE || context.source() == Sources.CRM) {
                    requestUserUid(context, context.msgContext().recipientEmail());
                } else if (context.source() == Sources.FBL) {
                    requestUserUid(context, context.msgContext().senderEmail());
                } else {
                    context.response();
                }
            }
        }
    }

    private static class MessageInfoCallback implements FutureCallback<JsonObject>, UserActionHandler {
        private final EMLContext context;
        private final MultiFutureCallback<String> multiCallback;

        public MessageInfoCallback(final EMLContext context, final MultiFutureCallback<String> multiCallback) {
            this.context = context;
            this.multiCallback = multiCallback;
        }

        @Override
        public void completed(final JsonObject result) {
            if (result instanceof JsonMap) {
                try {
                    JsonList hits = result.asMap().getList("hitsArray");
                    if (hits != null && hits.size() > 0) {
                        JsonMap jsonMap = hits.get(0).asMap();
                        if (jsonMap.containsKey(MailMetaInfo.STID)) {
                            context.msgContext().setStid(jsonMap.getString(MailMetaInfo.STID));
                        }
                        if (jsonMap.containsKey(MailMetaInfo.UID)) {
                            Long uid = Long.parseLong(jsonMap.getString(MailMetaInfo.UID));
                            context.msgContext().setUid(uid);
                            context.setUid(uid);
                        }
                        if (jsonMap.containsKey(MailMetaInfo.MID)) {
                            context.msgContext().setMid(Long.parseUnsignedLong(jsonMap.getString(MailMetaInfo.MID)));
                        }
                        if (context.msgContext().stid() != null && context.msgContext().mid() != null
                                && context.msgContext().uid() != null)
                        {
                            TraceFutureCallback.wrap(
                                new ComplaintProcessCallback(
                                    context,
                                    TraceFutureCallback.wrap(
                                        new UserActionCountersUpdater(
                                            context,
                                            multiCallback,
                                            context.source() == Sources.FBL
                                                ? Set.of(UpdateDataExecutor.ABUSES)
                                                : Set.of(UpdateDataExecutor.values())),
                                        context,
                                        "UserActionCountersUpdate")),
                                context, "ComplaintProcess").completed(List.of(
                                    new UpdateDataHolder(
                                        context.msgContext(),
                                        context.sourceDomain(),
                                        context.senders(),
                                        context.source())));
                            return;
                        } else {
                            context.session().logger().info(
                                "MessageInfoCallback.completed: stid=" + context.msgContext().stid() + ", mid="
                                    + context.msgContext().mid() + ", uid=" + context.msgContext().uid());
                        }
                    }
                } catch (JsonException e) {
                    context.session().logger().log(
                        Level.SEVERE,
                        "EMLHandler.MessageInfoCallback failed to parse JSON: " + e.toString(),
                        e);
                }
            } else {
                context.session().logger().warning("EMLHandler.MessageInfoCallback: unable find message info!");
            }
            finalProcessing(context, multiCallback);
            multiCallback.done();
        }

        @Override
        public void failed(Exception e) {
            context.session().logger().log(
                Level.SEVERE,
                "EMLHandler.MessageInfoCallback failed: " + e.toString(),
                e);
            finalProcessing(context, multiCallback);
            multiCallback.done();
        }

        @Override
        public void cancelled() {
            context.session().logger().log(Level.SEVERE, "EMLHandler.MessageInfoCallback cancelled: "
                + context.session().listener().details());
            finalProcessing(context, multiCallback);
            multiCallback.done();
        }
    }

    private static void finalProcessing(final EMLContext context, final MultiFutureCallback<String> multiCallback) {
        context.session().logger().info("EMLHandler.finalProcessing: writing of compl-log");
        UserActionHandler.doFinalComplaintProcessing(context, context.msgContext());
        UserActionHandler.writeComplYtLogRow(context, context.msgContext());
        if (context.source() == Sources.UNKNOWN) {
            context.response();
        } else {
            if (context.source() != Sources.FBL) {
                updatePersonalFilters(
                    context,
                    TraceFutureCallback.wrap(multiCallback.newCallback(), context, "UpdatePF_multi"));
            }
            updateAbuses(
                context,
                TraceFutureCallback.wrap(multiCallback.newCallback(), context, "UpdateAbuses_multi"));
        }
    }

    public static void requestUserUid(final EMLContext context, final String complainer)
    {
        final MultiFutureCallback<String> multiFutureCallback =
            new MultiFutureCallback<>(
                TraceFutureCallback.wrap(
                    new ResponseCallback<>(context, "Updating of shingles & PF filters for complaint"),
                    context,
                    "Response callback")
            );
        if (complainer == null || complainer.isEmpty()) {
            finalProcessing(context, multiFutureCallback);
            multiFutureCallback.done();
            return;
        }
        String domain = "";
        String login = complainer;
        Matcher m = MailMessageContext.RE_EMAIL.matcher(complainer);
        if (m.find()) {
            domain = m.group(2).toLowerCase(Locale.ROOT);
            if (MailMessageContext.RE_YANDEX_DOMAIN.matcher(domain).matches()) {
                login = m.group(1).toLowerCase(Locale.ROOT);
            }
        }
        context.setLogin(login);
        BlackboxClient client;
        BlackboxUserinfoRequest request = new BlackboxUserinfoRequest(BlackboxUserIdType.LOGIN, login);
        if (YANDEX_CORP_DOMAIN.matcher(domain).matches()) {
            client = context.iexProxy().corpBlackboxClient();
            request.addHeader(YandexHeaders.X_YA_SERVICE_TICKET, context.iexProxy().corpBlackboxTvm2Ticket());
        } else {
            client = context.iexProxy().blackboxDirectClient();
            request.addHeader(YandexHeaders.X_YA_SERVICE_TICKET, context.iexProxy().blackboxTvm2Ticket());
        }
        client = client.adjust(context.session().context());
        client.userinfo(
            request.sid("smtp").dbfields(BlackboxDbfield.SUID),
            context.session().listener().createContextGeneratorFor(client),
            TraceFutureCallback.wrap(
                new SingleUserinfoBlackboxCallback(
                    TraceFutureCallback.wrap(
                        new BlackboxCallback(context, multiFutureCallback),
                        context,
                        "BB for EMLHandler")),
                context,
                "SingleUserinfoBB for EMLHandler"));
        context.session().logger().info("EMLHandler: request BB for login=" + login);
    }

    public static void updatePersonalFilters(final EMLContext context, final FutureCallback<String> callback) {
        UpdateDataExecutor.PFILTERS.updateCounters(
            context,
            List.of(
                new UpdateDataHolder(
                    context.msgContext(),
                    context.sourceDomain(),
                    context.senders(),
                    context.source())),
            callback,
            "");
        context.session().logger().info("EMLHandler: request for updating personal filters");
    }

    public static void updateAbuses(final EMLContext context, final FutureCallback<String> callback) {
        UpdateDataExecutor.ABUSES.updateCounters(
            context,
            List.of(
                new UpdateDataHolder(
                    context.msgContext(),
                    context.sourceDomain(),
                    context.senders(),
                    context.source())),
            callback,
            "");
        context.session().logger().info("EMLHandler: request for updating abuses");
    }

    private static class BlackboxCallback implements FutureCallback<BlackboxUserinfo>
    {
        private final MultiFutureCallback<String> multiCallback;
        private final EMLContext context;

        BlackboxCallback(final EMLContext context, final MultiFutureCallback<String> multiCallback) {
            this.context = context;
            this.multiCallback = multiCallback;
        }

        @Override
        public void completed(final BlackboxUserinfo userinfo) {
            if (context.source() == Sources.FBL) {
                context.msgContext().setSenderUid(userinfo.uid());
            } else {
                context.msgContext().setUid(userinfo.uid());
                context.setUid(userinfo.uid());
                context.setSuid(userinfo.dbfields().get(BlackboxDbfield.SUID));
                context.setKarma((long) userinfo.karma());
            }
            context.session().logger().info("EMLHandler.BlackboxCallback: obtained UID: " + userinfo.uid()
                + " for login: "
                + (context.source() == Sources.FBL
                    ? context.msgContext().senderEmail() : context.msgContext().recipientEmail()));
            UserActionHandler.requestMessageInfo(
                context,
                TraceFutureCallback.wrap(new MessageInfoCallback(context, multiCallback), context, "MessageInfo"));
        }

        @Override
        public void cancelled() {
            multiCallback.cancelled();
        }

        @Override
        public void failed(Exception e) {
            if (context.source() == Sources.FBL) {
                context.msgContext().flags().add(Flags._XX);
                context.msgContext().skipReasons().add(SkipReason.FBL_EXT);
            }
            context.session().logger().info("EMLHandler.BlackboxCallback.failed: writing of compl-log");
            finalProcessing(context, multiCallback);
            multiCallback.done();
        }
    }

    private static class ResponseCallback<T> extends AbstractEntityCallback<EMLContext, T>
    {
        private final String prompt;

        ResponseCallback(final EMLContext context, final String prompt) {
            super(context);
            this.prompt = prompt;
        }

        @Override
        public void completed(final T unused) {
            context.session().logger().info("EMLHandler: successfully finished of " + context.source() + " " + prompt);
            context.response();
        }
    }

    public void crmComplaints(long count) {
        crmComplaints.accept(count);
    }

    public void exchangeSpamComplaints(long count) {
        exchangeSpamComplaints.accept(count);
    }

    public void exchangeHamComplaints(long count) {
        exchangeHamComplaints.accept(count);
    }

    public void fblComplaints(long count) {
        fblComplaints.accept(count);
    }

    public void soMaillistComplaints(long count) {
        soMaillistComplaints.accept(count);
    }

    public void unknownComplaints(long count) {
        unknownComplaints.accept(count);
    }
}
