package ru.yandex.iex.proxy.complaints;

import java.io.File;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import ru.yandex.client.so.shingler.AbstractShinglerClient;
import ru.yandex.client.so.shingler.config.ImmutableShinglerClientsConfig;
import ru.yandex.client.so.shingler.config.ShinglerType;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.config.ImmutableURIConfig;
import ru.yandex.iex.proxy.IexProxy;
import ru.yandex.logger.ImmutableLoggerConfig;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.DuplexStaterFactory;
import ru.yandex.stater.ImmutableStaterConfig;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.stater.RequestsStater;
import ru.yandex.stater.StaterConfig;
import ru.yandex.stater.StatersConfig;

public class ImmutableComplaintsConfig extends ImmutableURIConfig implements ComplaintsConfig {
    private final Map<Route, File> rulesDictFiles;
    private volatile Map<Route, Map<Integer, String>> rulesDictionaries;
    private final ImmutableLoggerConfig complLoggerConfig;
    private final ImmutableLoggerConfig ytLoggerConfig;
    private final ImmutableStaterConfig complaintsLagStaterConfig;
    private final ImmutableStaterConfig soSearchRequestsStaterConfig;
    private final int dailyComplaintsLimit;
    private final long messageExpirationPeriod;
    private final boolean shinglersDryRun;
    private final boolean shinglersAddStatByActions;
    private final boolean useSolog;
    private final boolean useSologger;
    private final ImmutableShinglerClientsConfig shinglersConfig;
    private final TimeFrameQueue<Long> complaintsLagStater;
    private final RequestsStater soSearchRequestsStater;
    private final Map<UserAction, RequestsStater> actionStaters;
    private final Map<ShinglerType, Map<UserAction, RequestsStater>> shinglersByUserActionStaters;
    private final Map<Flags, Map<UserAction, TimeFrameQueue<Long>>> flagsStaters;
    private final Map<ComplaintProcessingError, Map<UserAction, TimeFrameQueue<Long>>> errorsStaters;
    private Map<ShinglerType, AbstractShinglerClient<?, ?>> shinglerClients = null;

    public ImmutableComplaintsConfig(final ComplaintsConfig config)
        throws ConfigException
    {
        super(config);
        //rulesDictFiles = Collections.unmodifiableMap(new HashMap<>(config.rulesDictFiles()));
        //rulesDictionaries = Collections.unmodifiableMap(new HashMap<>(config.rulesDictionaries()));
        rulesDictFiles = new ConcurrentHashMap<>(config.rulesDictFiles());
        rulesDictionaries = new ConcurrentHashMap<>(config.rulesDictionaries());
        complLoggerConfig = new ImmutableLoggerConfig(config.complLog());
        ytLoggerConfig = new ImmutableLoggerConfig(config.ytLog());
        complaintsLagStaterConfig = new ImmutableStaterConfig(config.complaintsLagStaterConfig());
        complaintsLagStater = new TimeFrameQueue<>(complaintsLagStaterConfig.metricsTimeFrame());
        soSearchRequestsStaterConfig = new ImmutableStaterConfig(config.soSearchRequestsStaterConfig());
        soSearchRequestsStater = soSearchRequestsStaterConfig.build();
        dailyComplaintsLimit = config.dailyComplaintsLimit();
        messageExpirationPeriod = config.messageExpirationPeriod();
        shinglersDryRun = config.shinglersDryRun();
        shinglersAddStatByActions = config.shinglersAddStatByActions();
        actionStaters = new HashMap<>();
        shinglersByUserActionStaters = new HashMap<>();
        flagsStaters = new HashMap<>();
        errorsStaters = new HashMap<>();
        shinglersConfig = new ImmutableShinglerClientsConfig(config.shinglersConfig());
        useSolog = config.useSolog();
        useSologger = config.useSologger();
    }

    @Override
    public Map<Route, File> rulesDictFiles() {
        return rulesDictFiles;
    }

    public void reloadRulesDictionaries() throws ConfigException {
        rulesDictionaries = ComplaintsConfigBuilder.loadRulesDictionaries(rulesDictFiles);
    }

    @Override
    public Map<Route, Map<Integer, String>> rulesDictionaries() {
        return rulesDictionaries;
    }

    @Override
    public ImmutableLoggerConfig complLog() {
        return complLoggerConfig;
    }

    @Override
    public ImmutableLoggerConfig ytLog() {
        return ytLoggerConfig;
    }

    @Override
    public ImmutableStaterConfig complaintsLagStaterConfig() {
        return complaintsLagStaterConfig;
    }

    @Override
    public ImmutableStaterConfig soSearchRequestsStaterConfig() {
        return soSearchRequestsStaterConfig;
    }

    @Override
    public int dailyComplaintsLimit() {
        return dailyComplaintsLimit;
    }

    @Override
    public long messageExpirationPeriod() {
        return messageExpirationPeriod;
    }

    @Override
    public boolean shinglersDryRun() {
        return shinglersDryRun;
    }

    @Override
    public boolean shinglersAddStatByActions() {
        return shinglersAddStatByActions;
    }

    public void flagStater(final Flags flag, final UserAction action) {
        flagsStaters.get(flag).get(action).accept(1L);
    }

    @Override
    public ImmutableShinglerClientsConfig shinglersConfig() {
        return shinglersConfig;
    }

    @Override
    public boolean useSolog() {
        return useSolog;
    }

    @Override
    public boolean useSologger() {
        return useSologger;
    }

    public void registerComponents(final IexProxy iexProxy) throws ConfigException {
        shinglerClients = shinglersConfig.prepareShinglerClients(iexProxy);
        StaterConfig staterConfig;
        StatersConfig statersConfig;
        RequestsStater requestsStater;
        if (shinglersAddStatByActions) {
            Map<UserAction, RequestsStater> actionsStaters;
            for (Map.Entry<ShinglerType, AbstractShinglerClient<?, ?>> entry : shinglerClients.entrySet()) {
                statersConfig = shinglersConfig.shinglerClientConfig(entry.getKey()).statersConfig();
                if (statersConfig == null) {
                    continue;
                }
                staterConfig = statersConfig.staters().asterisk();
                actionsStaters = shinglersByUserActionStaters.computeIfAbsent(entry.getKey(), x -> new HashMap<>());
                for (UserAction action : UserAction.values()) {
                    if (action == UserAction.OTHER) {
                        continue;
                    }
                    requestsStater =
                        new RequestsStater(
                            iexProxy.config().metricsTimeFrame(),
                            staterConfig.prefix() + (staterConfig.prefix().isEmpty() ? "" : '-') + action.lowerName(),
                            staterConfig.metrics(),
                            staterConfig.category(),
                            staterConfig.title());
                    iexProxy.registerStater(requestsStater);
                    actionsStaters.put(action, requestsStater);
                }
            }
        }
        TimeFrameQueue<Long> stater;
        statersConfig = statersConfig();
        if (statersConfig != null) {
            staterConfig = statersConfig.staters().asterisk();
            for (UserAction action : UserAction.values()) {
                requestsStater =
                    new RequestsStater(
                        iexProxy.config().metricsTimeFrame(),
                        staterConfig.prefix() + (staterConfig.prefix().isEmpty() ? "" : '-') + action.lowerName(),
                        List.of(),
                        staterConfig.category(),
                        staterConfig.title());
                iexProxy.registerStater(requestsStater);
                actionStaters.put(action, requestsStater);
            }
        }
        for (Flags flag : Flags.values()) {
            stater = new TimeFrameQueue<>(iexProxy.config().metricsTimeFrame());
            flagsStaters.computeIfAbsent(flag, x -> new HashMap<>()).put(UserAction.OTHER, stater);
            iexProxy.registerStater(
                new PassiveStaterAdapter<>(
                    stater,
                    new NamedStatsAggregatorFactory<>(
                        "complaints-" + flag.name().substring(1) + "_ammm",
                        CountAggregatorFactory.INSTANCE)));
            for (UserAction action : UserAction.values()) {
                if (action == UserAction.OTHER) {
                    continue;
                }
                stater = new TimeFrameQueue<>(iexProxy.config().metricsTimeFrame());
                flagsStaters.computeIfAbsent(flag, x -> new HashMap<>()).put(action, stater);
                iexProxy.registerStater(
                    new PassiveStaterAdapter<>(
                        stater,
                        new NamedStatsAggregatorFactory<>(
                            "complaints-" + action.lowerName() + "-" + flag.name().substring(1) + "_ammm",
                            CountAggregatorFactory.INSTANCE)));
            }
        }
        for (ComplaintProcessingError error : ComplaintProcessingError.values()) {
            stater = new TimeFrameQueue<>(iexProxy.config().metricsTimeFrame());
            errorsStaters.computeIfAbsent(error, x -> new HashMap<>()).put(UserAction.OTHER, stater);
            iexProxy.registerStater(
                new PassiveStaterAdapter<>(
                    stater,
                    new NamedStatsAggregatorFactory<>(
                        "complaints-" + error.signalName() + "_ammm",
                        CountAggregatorFactory.INSTANCE)));
            for (UserAction action : UserAction.values()) {
                if (action == UserAction.OTHER) {
                    continue;
                }
                stater = new TimeFrameQueue<>(iexProxy.config().metricsTimeFrame());
                errorsStaters.computeIfAbsent(error, x -> new HashMap<>()).put(action, stater);
                iexProxy.registerStater(
                    new PassiveStaterAdapter<>(
                        stater,
                        new NamedStatsAggregatorFactory<>(
                            "complaints-" + action.lowerName() + "-" + error.signalName() + "_ammm",
                            CountAggregatorFactory.INSTANCE)));
            }
        }
        iexProxy.registerStater(
            new PassiveStaterAdapter<>(
                complaintsLagStater,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "complaints-lag-time_ammm",
                        IntegralSumAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "complaints-lag-total_ammm",
                        CountAggregatorFactory.INSTANCE))));
        iexProxy.registerStater(soSearchRequestsStater);
    }

    public AbstractShinglerClient<?, ?> shinglerClient(final ShinglerType shinglerType) {
        return shinglerClients.get(shinglerType);
    }

    public RequestsStater soSearchRequestsStater() {
        return soSearchRequestsStater;
    }

    public TimeFrameQueue<Long> complaintsLagStater() {
        return complaintsLagStater;
    }

    public RequestsStater actionStater(final UserAction action) {
        return actionStaters.get(action);
    }

    public Map<ShinglerType, Map<UserAction, RequestsStater>> shinglersByUserActionStaters() {
        return shinglersByUserActionStaters;
    }

    public void complaintsError(ComplaintProcessingError error, UserAction action) {
        errorsStaters.get(error).get(action).accept(1L);
    }
}
