package ru.yandex.iex.proxy.complaints;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.client.so.shingler.ActivityScheme;
import ru.yandex.client.so.shingler.ActivityShingles;
import ru.yandex.client.so.shingler.ComplScheme;
import ru.yandex.client.so.shingler.ComplShingles;
import ru.yandex.client.so.shingler.FreemailScheme;
import ru.yandex.client.so.shingler.FreemailShingles;
import ru.yandex.client.so.shingler.GeneralShingleInfo;
import ru.yandex.client.so.shingler.MassShinglerResult;
import ru.yandex.client.so.shingler.SenderScheme;
import ru.yandex.client.so.shingler.SenderShingles;
import ru.yandex.client.so.shingler.ShingleException;
import ru.yandex.client.so.shingler.ShingleType;
import ru.yandex.client.so.shingler.ShinglersData;
import ru.yandex.client.so.shingler.Shingles;
import ru.yandex.client.so.shingler.UrlShingles;
import ru.yandex.client.so.shingler.config.ShinglerType;
import ru.yandex.dbfields.MailIndexFields;
import ru.yandex.digest.Fnv;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.iex.proxy.AbstractHandlersContext;
import ru.yandex.iex.proxy.IexProxy;
import ru.yandex.iex.proxy.IndexationContext;
import ru.yandex.iex.proxy.move.UpdateDataHolder;
import ru.yandex.parser.mail.received.ReceivedChainParser;
import ru.yandex.search.document.mail.MailMetaInfo;

public class MailMessageContext implements AbstractHandlersContext {
    public static final String LOG_TOPIC = "mail-so-compl-log";
    public static final String MESSAGE_ID = "message-id";
    public static final String RETURN_PATH = "return-path";
    public static final String X_YANDEX_PERSONAL_SPAM = "x-yandex-personal-spam";
    public static final String AUTHENTICATION_RESULTS = "authentication-results";
    public static final String SEEN = "seen";
    public static final String RECEIVED = "received";
    public static final String DOMAIN = "domain";
    public static final String MX = "mx";
    public static final String RCPT = "rcpt";
    public static final String FORWARD = "forward";
    public static final String SMTP = "smtp";
    public static final String CORP = "corp";
    public static final String X_MAILER = "x-mailer";
    public static final String X_YANDEX_FRONT = "x-yandex-front";
    public static final String PERSONAL_CORRECT = "PERSONAL_CORRECT";
    public static final long MILLIS = 1000L;
    public static final DateTimeZone TIMEZONE = DateTimeZone.forID("Europe/Moscow");
    public static final Pattern RE_DOMAIN =
        Pattern.compile("(?:(?:[a-z0-9](?:[a-z0-9-]{0,61}[a-z0-9])?|xn--[a-z0-9-]+)\\.)+(?:xn--[a-z0-9-]+|[a-z]+)");
    public static final Pattern RE_EMAIL = Pattern.compile("(?i)([^\"@<>,\\s]+)@(" + RE_DOMAIN.pattern() + ")$");
    public static final Pattern RE_AUTH_RESULTS_DOMAIN1 =
        Pattern.compile("(?i)\\bdomain of\\s+(" + RE_DOMAIN.pattern() + ")\\s+");
    public static final Pattern RE_AUTH_RESULTS_DOMAIN2 =
        Pattern.compile("(?i)\\bheader.i=[^\\s@]+?@(" + RE_DOMAIN.pattern() + ")\\b");
    public static final Pattern RE_YANDEX_NET = Pattern.compile("(?i).*\\.yandex\\.net\\b");
    public static final Pattern RE_YANDEX_DOMAIN =
        Pattern.compile("(?i)\\b([a-z-]+)[\\w-]+?\\.(?:mail|ld)\\.yandex(?:-team)?\\.[a-z]+\\b");
    public static final Pattern RE_YANDEX_DOMAIN_QLOUD =
        Pattern.compile("(?i)\\bby [a-z0-9-]+\\.qloud-c\\.yandex\\.net with HTTP;");
    public static final Pattern RE_YANDEX_EMAIL =
        Pattern.compile("(?i)(\\S+)@((?:(?:mail|ld)\\.)?yandex(?:-team)?\\.[a-z]+)$");
    public static final Set<String> freemailHosts = Set.of(
        "km.ru", "freemail.ru", "girlmail.ru", "bossmail.ru", "megabox.ru", "boymail.ru", "safebox.ru",
        "qip.ru", "pochta.ru", "fromru.com", "front.ru", "hotbox.ru", "hotmail.ru", "krovatka.su", "land.ru",
        "mail15.com", "mail333.com", "newmail.ru", "nightmail.ru", "nm.ru", "pisem.net", "pop3.ru", "rbcmail.ru",
        "smtp.ru", "5ballov.ru", "aeterna.ru", "ziza.ru", "memori.ru", "photofile.ru", "photoplenka.ru",
        "aol.com",
        "yahoo.com",
        "yandex.ru", "ya.ru", "narod.ru",
        "mail.ru", "inbox.ru", "bk.ru", "list.ru",
        "gmail.com",
        "rambler.ru",
        "hotmail.com", "live.ru", "outlook.com"
    );

    enum SoResolution {
        SPAM,
        HAM,
        SKIP;

        private final String lowerName;

        SoResolution() {
            lowerName = name().toLowerCase(Locale.ROOT);
        }

        public String lowerName() {
            return lowerName;
        }
    }

    enum TimestampFormat {
        ISO("yyyy-MM-dd HH:mm:ss"),
        CUSTOM("dd.MM.yyyy HH:mm:ss"),
        MSG_HEADER("EE, dd MMM yyyy HH:mm:ss");

        private final DateTimeFormatter formatter;

        TimestampFormat(final String format) {
            formatter = DateTimeFormat.forPattern(format);
        }

        public DateTimeFormatter formatter() {
            return formatter;
        }

        public String apply(final Long unixtime) {
            return (unixtime == null || unixtime < 1 ? ""
                : formatter.print(new DateTime(unixtime * MILLIS, TIMEZONE)));
        }
    }

    protected static final String SENDERTYPE = "sendertype";

    private final IexProxy iexProxy;
    private final ProxySession session;
    private final Map<?, ?> json;
    private UserAction action;
    private Long mid = null;
    private String stid = null;
    private String subject = null;
    private String queueId;
    private final List<String> allSmtpIds;
    private String senderHost;
    private final long actionDate;
    private long messageDate = 0L;
    private String from = null;
    private String fromDomain = null;
    private final Set<String> rules;
    private Map<String, List<String>> headersMap = null;
    private final Map<String, Long> recipients; // map: recipient's email -> recipient's UID
    private Long uid;                           // recipient's UID
    private String recipientEmail;
    private String spamSampleData = null;
    private long senderUid;
    private String senderEmail = null;
    private String senderName = null;
    private Sources source;
    private final String sessionKey;
    private SoResolution soRes = null;
    private String mailBackend = "";
    private Route route;
    private final Set<Flags> flags;
    private final Set<SkipReason> skipReasons;
    private String returnPath = null;
    private String msgId = null;
    private String geoZone = "";
    private String senderIp = "";
    private String folder = "";
    private Boolean seen = null;
    private Boolean spfDkim = null;
    private final ShinglersData shinglersData;
    private boolean isForward = false;
    private boolean isTestLetter = false;
    private boolean isRetry;

    public MailMessageContext(final AbstractHandlersContext context, final UserAction action, final long actionDate) {
        iexProxy = context.iexProxy();
        session = context.session();
        uid = context.prefix();
        json = context.json();
        this.action = action;
        this.actionDate = actionDate;
        allSmtpIds = new ArrayList<>();
        recipients = new HashMap<>();
        rules = new HashSet<>();
        flags = new HashSet<>();
        skipReasons = new HashSet<>();
        shinglersData = new ShinglersData();
        route = context.corp() ? Route.CORP : Route.IN;
        senderUid = 0L;
        isRetry = false;
        source = source(json);
        sessionKey = sessionKey(json);
        if (sessionKey == null || sessionKey.isEmpty()) {
            skipReasons.add(SkipReason.AUTOMATION);
        }
    }

    public MailMessageContext(
        final AbstractHandlersContext context,
        final UserAction action,
        final long actionDate,
        final String queueId,
        final List<String> allSmtpIds,
        final String senderHost,
        final String recipientEmail)
    {
        this(context, action, actionDate);
        this.queueId = queueId;
        if (allSmtpIds != null) {
            this.allSmtpIds.addAll(allSmtpIds);
        }
        this.senderHost = senderHost;
        this.recipientEmail = recipientEmail;
    }

    public MailMessageContext(
        final AbstractHandlersContext context,
        final UserAction action,
        final long actionDate,
        final ReceivedChainParser chainParser)
    {
        this(context, action, actionDate);
        queueId = chainParser.yandexSmtpId();
        allSmtpIds.addAll(chainParser.allYandexSmtpIds());
        senderHost = chainParser.fullSenderHost();
        if (chainParser.sourceIps().size() > 0) {
            senderIp = chainParser.sourceIps().get(0).getHostAddress();
        }
        setMailBackend(chainParser.mailFront());
        if (chainParser.recipients().size() > 0) {
            recipientEmail = chainParser.recipients().iterator().next();
            isForward = chainParser.recipients().size() > 1;
        }
        isTestLetter = chainParser.cmail();
    }

    @Override
    public IexProxy iexProxy() {
        return iexProxy;
    }

    @Override
    public ProxySession session() {
        return this.session;
    }

    @Override
    public long prefix() {
        return uid;
    }

    @Override
    public Map<?, ?> json() {
        return this.json;
    }

    @Override
    public String uid() {
        return Long.toString(uid);
    }

    public void setUid(final long uid) {
        this.uid = uid;
    }

    public Long mid() {
        return mid;
    }

    public void setMid(final long mid) {
        this.mid = mid;
    }

    public String stid() {
        return stid;
    }

    public void setStid(final String stid) {
        this.stid = stid;
    }

    public Route route() {
        return route;
    }

    public void setRoute(final Route route) {
        this.route = route;
    }

    public void setRoute(final String route) {
        this.route = route == null || route.isEmpty()
            ? Route.IN : Route.valueOf(route.toUpperCase(Locale.ROOT));
    }

    public SoResolution soRes() {
        return soRes;
    }

    public void setSoRes(final SoResolution soRes) {
        this.soRes = soRes;
    }

    public void setSoRes(final String soRes) {
        if (soRes == null) {
            this.soRes = null;
        } else {
            this.soRes = soRes.isBlank() ? SoResolution.SKIP : SoResolution.valueOf(soRes.toUpperCase(Locale.ROOT));
        }
    }

    public String mailBackend() {
        return mailBackend;
    }

    public void setMailBackend(final String mailBackend) {
        this.mailBackend = mailBackend == null ? "" : mailBackend;
    }

    public Sources source() {
        return source;
    }

    public void setSource(final Sources source) {
        this.source = source;
    }

    public UserAction action() {
        return action;
    }

    public void setAction(final UserAction action) {
        this.action = action;
    }

    public Boolean isSpam() {
        if (action == UserAction.SPAM || action == UserAction.HAM) {
            return action == UserAction.SPAM;
        }
        return null;
    }

    public String queueId() {
        return queueId;
    }

    public void setQueueId(final String queueId) {
        this.queueId = queueId;
    }

    public List<String> allSmtpIds() {
        return allSmtpIds;
    }

    public String sessionKey() {
        return sessionKey;
    }

    public String senderHost() {
        return senderHost;
    }

    public void setSenderHost(final String senderHost) {
        this.senderHost = senderHost;
    }

    public String senderEmail() {
        return senderEmail;
    }

    public long senderUid() {
        return senderUid;
    }

    public void setSenderUid(final long senderUid) {
        this.senderUid = senderUid;
    }

    public Map<String, Long> recipients() {
        return recipients;
    }

    public long messageDate() {
        return messageDate;
    }

    public long actionDate() {
        return actionDate;
    }

    public String subject() {
        return subject;
    }

    public String recipientEmail() {
        return recipientEmail;
    }

    public void setRecipientEmail(final String recipientEmail) {
        this.recipientEmail = recipientEmail;
    }

    public String spamSampleData() {
        return spamSampleData;
    }

    public void spamSampleData(final String spamSampleData) {
        this.spamSampleData = spamSampleData;
    }

    public String from() {
        return from;
    }

    public String fromDomain() {
        return fromDomain;
    }

    public String msgId() {
        return msgId;
    }

    public String returnPath() {
        return returnPath;
    }

    public String senderName() {
        return senderName;
    }

    public Set<Flags> flags() {
        return flags;
    }

    public Set<SkipReason> skipReasons() {
        return skipReasons;
    }

    public String geoZone() {
        return geoZone;
    }

    public void setGeoZone(final String geoZone) {
        this.geoZone = geoZone == null ? "" : geoZone;
    }

    public String senderIp() {
        return senderIp;
    }

    public void setSenderIp(final String senderIp) {
        this.senderIp = senderIp;
    }

    public String folder() {
        return folder;
    }

    public void setFolder(final String folder) {
        this.folder = folder;
    }

    public Boolean seen() {
        return seen;
    }

    public void setSeen(final boolean seen) {
        this.seen = seen;
    }

    public static String pureEmail(final String headerValue) {
        String email = headerValue;
        if (email != null) {
            if (email.startsWith("<")) {
                email = email.substring(1).trim();
            }
            if (email.endsWith(">")) {
                email = email.substring(0, email.length() - 1).trim();
            }
        }
        return email;
    }

    public MailMessageContext headersMap(final Map<String, List<String>> headersMap) {
        this.headersMap = headersMap;
        if (headersMap != null) {
            if (headersMap.containsKey(MESSAGE_ID)) {
                msgId = headersMap.get(MESSAGE_ID).get(0);
            }
            if (headersMap.containsKey(RETURN_PATH)) {
                returnPath = pureEmail(headersMap.get(RETURN_PATH).get(0));
            }
            if (headersMap.containsKey(MailMetaInfo.FROM)) {
                senderEmail = pureEmail(headersMap.get(MailMetaInfo.FROM).get(0));
            }
            if (returnPath != null && (senderEmail == null || senderEmail.isEmpty())) {
                senderEmail = returnPath;
            }
            if (headersMap.containsKey(X_YANDEX_PERSONAL_SPAM)) {
                flags.add(Flags._PF);
            }
            if (isTestLetter) {
                skipReasons.add(SkipReason.TEST_LETTER);
            }
            parseRoute();
        }
        return self();
    }

    public MailMessageContext indexationContext(final IndexationContext<UpdateDataHolder> indexationContext) {
        if (uid == null) {
            uid = Long.parseLong(indexationContext.uid());
        }
        mid = indexationContext.mid() == null ? null : Long.parseLong(indexationContext.mid());
        stid = indexationContext.stid();
        messageDate = indexationContext.receivedDate();
        final long expirationPeriod = iexProxy.config().complaintsConfig().messageExpirationPeriod();
        if ((action == UserAction.SPAM || action == UserAction.HAM)
                && messageDate < actionDate - expirationPeriod / MILLIS)
        {
            skipReasons.add(SkipReason.OLDER_30_DAYS);
        }
        return mailMetaInfo(indexationContext.meta());
    }

    public MailMessageContext mailMetaInfo(final MailMetaInfo metaInfo) {
        if (metaInfo == null) {
            senderName = null;
            senderEmail = returnPath;
            subject = null;
        } else {
            if (messageDate == 0L) {
                if (metaInfo.get(MailMetaInfo.RECEIVED_DATE) != null) {
                    messageDate = (long) Double.parseDouble(metaInfo.get(MailMetaInfo.RECEIVED_DATE).trim());
                } else if (metaInfo.get(MailMetaInfo.GATEWAY_RECEIVED_DATE) != null) {
                    messageDate = (long) Double.parseDouble(metaInfo.get(MailMetaInfo.GATEWAY_RECEIVED_DATE).trim());
                }
            }
            subject = metaInfo.get(MailMetaInfo.HDR + MailMetaInfo.SUBJECT);
            if (subject != null) {
                subject = subject.replaceAll("\\s+", " ").trim();
            }
            String hdrFrom = metaInfo.get(MailMetaInfo.HDR + MailMetaInfo.FROM + MailMetaInfo.NORMALIZED);
            if (hdrFrom != null && !hdrFrom.trim().isEmpty()) {
                senderEmail = from = hdrFrom.trim();
            } else if (returnPath != null) {
                senderEmail = returnPath;
            }
            String to = metaInfo.get(MailMetaInfo.HDR + MailMetaInfo.TO + MailMetaInfo.NORMALIZED);
            if (to != null && !to.trim().isEmpty()) {
                to = to.trim();
                if (recipientEmail == null || recipientEmail.isEmpty() || !to.contains("\n")) {
                    recipientEmail = to;
                }
                for (final String recipientItem : to.split("\\s+")) {
                    recipients.put(recipientItem.trim(), recipientItem.trim().equals(recipientEmail) ? uid : 0L);
                }
            }
            senderName = metaInfo.get(MailMetaInfo.HDR + MailMetaInfo.FROM + MailMetaInfo.DISPLAY_NAME);
            if (senderName != null) {
                senderName = senderName.replaceAll("\\s+", " ").trim();
            }
        }
        fromDomain = domain(from);
        spfDkim = spfDkim();
        return self();
    }

    public MailMessageContext actionInfo(final Map<?, ?> actionInfo) {
        if (actionInfo == null) {
            session.logger().info("MailMessageContext: actionInfo is absent for uid=" + uid);
        } else {
            Object seen = actionInfo.get(SEEN);
            if (seen != null) {
                this.seen = (Boolean) seen;
            }
            if (actionInfo.containsKey(MailIndexFields.FOLDER_NAME)) {
                folder = actionInfo.get(MailIndexFields.FOLDER_NAME).toString();
            }
        }
        return self();
    }

    public MailMessageContext self() {
        return this;
    }

    public boolean isTestLetter() {
        return isTestLetter;
    }

    @SuppressWarnings("unused")
    public void thisIsTestLetter() {
        isTestLetter = true;
    }

    public boolean isForward() {
        return isForward;
    }

    @SuppressWarnings("unused")
    public void thisIsForward() {
        isForward = true;
    }

    public Set<String> rules() {
        return rules;
    }

    public Map<String, List<String>> headersMap() {
        return headersMap;
    }

    public ShinglersData shinglersData() {
        return shinglersData;
    }

    public boolean isRetry() {
        return isRetry;
    }

    public void thisIsRetry() {
        isRetry = true;
    }

    public boolean isPersonal() {
        return rules.contains(PERSONAL_CORRECT);
    }

    public static String domain(final String emailAddr) {
        if (emailAddr == null) {
            return null;
        }
        Matcher m = RE_EMAIL.matcher(emailAddr);
        return m.find() ? m.group(2) : null;
    }

    public static String login(final String emailAddr) {
        if (emailAddr == null) {
            return null;
        }
        Matcher m = RE_EMAIL.matcher(emailAddr);
        return m.find() ? m.group(1) : null;
    }

    public Boolean spfDkim() {
        if (from == null || !headersMap.containsKey(AUTHENTICATION_RESULTS)) {
            return null;
        }
        return spfDkim(from, headersMap.get(AUTHENTICATION_RESULTS).get(0));
    }

    public static boolean spfDkim(final String from, final String authResults) {
        String domain = domain(from);
        if (domain != null && !domain.isEmpty()) {
            boolean spfPass = authResults.contains(" spf=pass ");
            boolean dkimPass = authResults.contains(" dkim=pass ");
            if (spfPass || dkimPass) {
                Matcher m1 = RE_AUTH_RESULTS_DOMAIN1.matcher(authResults);
                Matcher m2 = RE_AUTH_RESULTS_DOMAIN2.matcher(authResults);
                return m1.find() && spfPass && m1.group(1).equals(domain)
                        || m2.find() && dkimPass && m2.group(1).equals(domain);
            }
        }
        return false;
    }

    private void parseRoute() {
        if (mailBackend.endsWith(CORP) || source == Sources.EXCHANGE || source == Sources.CRM
                || source == Sources.SO_MAILLIST)
        {
            setRoute(Route.CORP);
            return;
        } else if (source == Sources.FBL || mailBackend.startsWith(SMTP)
                || headersMap.containsKey(X_MAILER) && headersMap.get(X_MAILER).get(0).startsWith("Yamail"))
        {
            setRoute(Route.OUT);
            return;
        }
        Matcher m = null;
        if (headersMap.containsKey(X_YANDEX_FRONT)) {
            m = RE_YANDEX_DOMAIN.matcher(headersMap.get(X_YANDEX_FRONT).get(0));
            if (!m.find() && headersMap.containsKey(AUTHENTICATION_RESULTS)) {
                m = RE_YANDEX_DOMAIN.matcher(headersMap.get(AUTHENTICATION_RESULTS).get(0));
            }
        }
        if (m != null && m.find()) {
            if (m.group(1).endsWith(CORP)) {
                setRoute(Route.CORP);
                return;
            }
            if (m.group(1).equals("mxback")) {
                if (mailBackend.equals(SMTP) || mailBackend.equals("web")) {
                    setRoute(Route.OUT);
                    return;
                }
                for (final String received : headersMap.get(RECEIVED)) {
                    if (RE_YANDEX_DOMAIN_QLOUD.matcher(received).find()) {
                        setRoute(Route.OUT);
                        return;
                    }
                }
            }
        }
        setRoute(Route.IN);
    }

    public void fillFlags() {
        // complaint types as defined in https://wiki.yandex-team.ru/AntiSpam/Complaints/#oboznachenietipovzhalob
        if (rules.size() < 1) {
            flags.add(Flags._BR);
        }
        if (rules.contains("HIDDEN_DLVR")) {
            flags.add(Flags._HD);
        }
        if (rules.contains("DL_FBR")) {
            flags.add(Flags._DL);
        }
        if (rules.contains("FREE_MAIL_COND")) {
            flags.add(Flags._FM);
        }
        if (rules.contains("DSN_NO_SENT_BY_YAMAIL") || rules.contains("CL_BOUNCE")) {
            flags.add(Flags._BN);
        }
        if (mailBackend.equals("yaback")) {
            flags.add(Flags._YB);
        } else if (mailBackend.equals("outback")) {
            flags.add(Flags._OB);
        } else if (rules.contains("BY_YANDEX_HTTP")) {
            flags.add(Flags._YW);
        } else if (rules.contains("BY_YANDEX_SMTP")) {
            flags.add(Flags._YS);
        } else if (rules.contains("ALLTRUSTEDIP") && source != Sources.FBL) {
            flags.add(Flags._ZY);
        }
        if (rules.contains("TR_GEO_RCP")) {
            flags.add(Flags._ZT);
        }
        if (rules.contains("USR_UA")) {
            flags.add(Flags._ZU);
        }
        if (rules.contains("USR_BY")) {
            flags.add(Flags._ZB);
        }
        if (rules.contains("__CSO_FROM_FORWARD")) {
            flags.add(Flags._DF);
        }
        if (rules.contains("YANDEX_MAILER") || rules.contains("YANDEX_SMTP")) {
            flags.add(Flags._DC);
        }
        if (rules.contains("__POP3_AUTH")) {
            flags.add(Flags._P3);
        }
        if (rules.contains("SH_27_1111")) {
            flags.add(Flags._1K);
        }
        if (rules.contains("__IS_FORWARD")) {
            flags.add(Flags._FW);
        }
        if (rules.contains("PAY_SENDER")) {
            flags.add(Flags._PS);
        }
        if (rules.contains("YA_DEVNULL_RP")) {
            flags.add(Flags._DN);
        }
        if (rules.contains("SHIN2_FAIL")) {
            flags.add(Flags._S2);
        }
        if (rules.contains("PDD")) {
            flags.add(Flags._PD);
            if (rules.contains("PDD_MAILBOX_5K")) {
                flags.add(Flags._PO);
            }
        }
        if (rules.contains("BY_YANDEX_HTTP_SPAM")) {
            flags.add(Flags._YC);
        }
        if (rules.contains("TR_GEO_USER")) {
            flags.add(Flags._OT);
        }
        if (rules.contains("REC_IPV6")) {
            flags.add(Flags._V6);
        }
        if (rules.contains("__IMAP_AUTH")) {
            flags.add(Flags._IM);
        }
        if (rules.contains("__BORN_DATE_0_30")) {
            flags.add(Flags._FR);
        }
        if (rules.contains("NEWS_NO_UNSIB_LIST")) {
            flags.add(Flags._NU);
        }
        if (seen != null && seen) {
            flags.add(Flags._SN);
        }
        if (rules.contains("__SPF_PASS")) {
            flags.add(Flags._SP);
        }
        if (rules.contains("SPF_FAIL")) {
            flags.add(Flags._SF);
        }
        if (rules.contains("__YA_DKIM_PASS")) {
            flags.add(Flags._DP);
        }
        if (rules.contains("DOMN_ROLL")) {
            flags.add(Flags._DR);
        }
        if (rules.contains("YA_POP3")) {
            flags.add(Flags._YP);
        }
        if (rules.contains("YA_IMAP")) {
            flags.add(Flags._YM);
        }
        if (corp()) {
            flags.add(Flags._YT);
        }
        if (rules.contains("RDSL_FR")) {
            flags.add(Flags._DS);
        }
        //    if (rules.contains("Z_MARKET")) {
        //        flags.add(Flags._MZ);
        //    }
        //    if (rules.contains("CL_SNDR_ESHOP")) {
        //        flags.add(Flags._ES);
        //    }
        if (rules.contains("U_PF")) {
            flags.add(Flags._UF);
        }
        if (rules.contains(PERSONAL_CORRECT)) {
            flags.add(Flags._PF);
        }
        if (rules.contains("SP_LIST_YTEAM")) {
            flags.add(Flags._LY);
        }
        if (rules.contains("ABUK_PERC_1_50") || rules.contains("ABUK_PERC_50_90")
                || rules.contains("ABUK_PERC_90_MAX")) {
            flags.add(Flags._AB);
        }
        if (rules.contains("FAKE_RESOLV") || rules.contains("FAKE_RESOLV_V6") || rules.contains("FRNR")) {
            flags.add(Flags._FA);
        }
        if (rules.contains("ACT_US")) {
            flags.add(Flags._AU);
        }
        if (rules.contains("CLNDR_CMPL")) {
            flags.add(Flags._CA);
        }
        if (rules.contains("MAILISH")) {
            flags.add(Flags._MA);
        }
        if (rules.contains("RPTN_MAN")) {
            flags.add(Flags._RP);
        }
        if (route == Route.OUT) {
            if (rules.contains("MALIC_ML")) {
                flags.add(Flags._ML);
            }
            if (rules.contains("PDD_LOCAL")) {
                flags.add(Flags._PL);
            }
            if (rules.contains("YA_CAPTCHA")) {
                flags.add(Flags._CY);
            }
            if (rules.contains("YA_CAPTCHA_BAD")) {
                flags.add(Flags._CN);
            }
        }
        if (rules.contains("SEO_ABUSE")) {
            flags.add(Flags._SA);
        }
        if (soRes == SoResolution.SKIP) {
            flags.add(Flags._SK);
        }
        if (soRes == SoResolution.SPAM && action == UserAction.SPAM) {
            flags.add(Flags._WS);
            flags.add(Flags._XX);
            skipReasons.add(SkipReason.FAKE_FOO);
        }
        if (soRes == SoResolution.HAM && action == UserAction.HAM) {
            flags.add(Flags._WH);
            flags.add(Flags._XX);
            skipReasons.add(SkipReason.FAKE_FOO);
        }
        if (source == Sources.CRM) {
            flags.add(Flags._CR);
        }
        if (source == Sources.EXCHANGE) {
            flags.add(Flags._EX);
        }
        if (source == Sources.UNSUBSCRIBE) {
            flags.add(Flags._UN);
        }
        if (source == Sources.IMAP) {
            flags.add(Flags._IC);
            skipReasons.add(SkipReason.IMAP);
        }
        if (source == Sources.MOBILE) {
            flags.add(Flags._MC);
        }
        if (source == Sources.FURITA) {
            flags.add(Flags._FC);
        }
        if (source == Sources.FBL) {
            flags.add(Flags._FB);
        }
        if (source == Sources.FASTSRV) {
            flags.add(Flags._FS);
        }
        if (skipReasons.size() > 0) {   // NOTE: call this checking up after all other modifications of skipReasons
            flags.add(Flags._XX);
            if (skipReasons.contains(SkipReason.MAXED)) {
                flags.add(Flags._MD);
            }
            session.logger().warning("MailMessageContext: skipReasons=" + skipReasons);
        }
        for (int j = 0; j < 10; j++) {
            if (rules.contains("LOGTYPE_A" + j)) {
                flags.add(Flags.valueOf("_A" + j));
            }
            if (rules.contains("FLAG_" + j)) {
                flags.add(Flags.valueOf("_F" + j));
            }
        }
    }

    public static String formatParameter(final String p) {
        return (p == null || p.isEmpty() ? emptyParameter() : p.trim().replace("\t", "").replace("\n", ""));
    }

    public static String formatDate(final Long unixtime, final TimestampFormat timestampFormat) {
        return (unixtime == null || unixtime == 0L ? emptyParameter() : timestampFormat.apply(unixtime));
    }

    public static String formatLong(final Long value) {
        return value == null ? emptyParameter() : Long.toString(value);
    }

    public static String emptyParameter() {
        return "-";
    }

    public String getComplLogRow() {
        ArrayList<String> fields = new ArrayList<>();
        fields.add(formatParameter(returnPath));
        fields.add(formatParameter(from));
        fields.add(action.name().substring(0, 1));
        fields.add(formatDate(messageDate, TimestampFormat.CUSTOM));
        fields.add(formatDate(actionDate, TimestampFormat.CUSTOM));
        fields.add(formatParameter(flags().stream().map(Flags::name).collect(Collectors.joining())));
        fields.add(formatParameter(recipientEmail));
        fields.add(formatParameter(msgId));
        fields.add(formatParameter(senderIp));
        fields.add(formatParameter(senderHost));
        fields.add(formatParameter(geoZone));
        fields.add(formatParameter(source == null || source == Sources.FBL ? "" : uid()));
        fields.add(formatParameter(queueId));
        fields.add(formatParameter(senderName));
        fields.add(formatParameter(subject));
        fields.add(formatParameter(soRes == null ? "" : soRes.lowerName()));
        return String.join("\t", fields);
    }

    public String getYtLogRow(final String suid, final Long karma, final Boolean showTabs) {
        HashMap<String, String> fields = new HashMap<>();
        fields.put("unixtime", formatLong(messageDate));
        fields.put("actdate", formatDate(actionDate, TimestampFormat.ISO));
        fields.put("msgdate", formatDate(messageDate, TimestampFormat.ISO));
        String actionStr = "";
        if (action != null) {
            switch (action) {
                case SPAM:
                    actionStr = "foo";
                    break;
                case HAM:
                    actionStr = "antifoo";
                    break;
                case DELETE:
                    actionStr = "delete";
                    break;
                default:
                    actionStr = "";
            }
        }
        fields.put(MailIndexFields.TYPE, formatParameter(actionStr));
        fields.put(MailIndexFields.UID, formatLong(uid));
        fields.put(MailIndexFields.SUID, formatParameter(suid));
        fields.put(MailIndexFields.STID, formatParameter(stid));
        fields.put(MailIndexFields.MID, formatLong(mid));
        fields.put("karma", formatLong(karma));
        fields.put("ip", emptyParameter());     // IP-address of complainant
        fields.put("queueid", formatParameter(queueId));
        fields.put("source", formatParameter(source == null ? null : source.name().toLowerCase(Locale.ROOT)));
        fields.put("from", formatParameter(from));
        fields.put("sender_ip", formatParameter(senderIp));
        fields .put("rcpt", formatParameter(domain(recipientEmail)));
        String soResStr = "";
        if (soRes != null) {
            switch (soRes) {
                case HAM:
                    soResStr = "1";
                    break;
                case SPAM:
                    soResStr = "2";
                    break;
                default:
                    soResStr = "";
            }
        }
        fields.put("so_res", formatParameter(soResStr));
        fields.put("route", formatParameter(route == null ? null : route.lowerName()));
        fields.put(
            "flags",
            formatParameter(flags.stream().map(x -> x.name().replace("_", "")).collect(Collectors.joining(";"))));
        fields.put("geo", formatParameter(geoZone));
        fields.put("folder", formatParameter(folder));
        fields.put(SEEN, formatParameter(seen == null ? null : (seen ? "yes" : "no")));
        fields.put("spf_dkim", formatParameter(spfDkim == null ? null : (spfDkim ? "1" : "0")));
        fields.put(
            "skipped",
            formatParameter(skipReasons.stream().map(SkipReason::lowerName).collect(Collectors.joining(";"))));
        fields.put("client", emptyParameter());     // we don't know client now
        fields.put("rules", formatParameter(rules.stream().sorted().collect(Collectors.joining(";"))));
        fields.put("show_tabs", formatParameter(showTabs == null ? null : (showTabs ? "on" : "off")));
        final StringBuilder sb = new StringBuilder("tskv\ttskv_format=" + LOG_TOPIC);
        for (final Map.Entry<String, String> entry : fields.entrySet()) {
            sb.append("\t");
            sb.append(entry.getKey());
            sb.append("=");
            sb.append(entry.getValue());
        }
        return sb.toString();
    }

    @SuppressWarnings("unchecked")
    public boolean setUpFreemailShingles() {
        if (senderUid != 0 && (action == UserAction.HAM || action == UserAction.SPAM)) {
            try {
                final Set<FreemailScheme> schemes = new HashSet<>();
                FreemailShingles addonShingles = null;
                final Map<String, Object> counters = new HashMap<>(Map.of(
                    "uuid", FreemailShingles.hash(Long.toString(senderUid), true)
                ));
                boolean isSpam = action == UserAction.SPAM;
                if (route == Route.OUT) {
                    schemes.add(FreemailScheme.TIME);
                    schemes.add(FreemailScheme.COUNTERS);
                    schemes.add(FreemailScheme.COMPLAINT);
                    counters.put("complaint_" + (isSpam ? "spam" : "ham"), 1L);
                    counters.put("spam", isSpam);
                    counters.put("hash", FreemailShingles.hash(uid(), false));
                }
                if (isFreemail() || route == Route.OUT) {
                    schemes.add(FreemailScheme.TIME);
                    schemes.add(FreemailScheme.COUNTERS);
                    schemes.add(FreemailScheme.ACTIVE);
                    schemes.add(FreemailScheme.RECIPIENTS_MAX);
                    counters.put("send_" + (isSpam ? "spam" : "ham"), 1L);
                    counters.put("recepients_count", recipients.size());
                    counters.put("active_days", 1);
                    if (geoZone != null && !geoZone.isEmpty()) {
                        schemes.add(FreemailScheme.GEO);
                        counters.put("geo", geoZone.split(" ")[0]);
                    }
                }
                if (recipients.size() > 0) {
                    schemes.add(FreemailScheme.COUNTERS);
                    final String key = "receive_" + (isSpam ? "spam" : "ham");
                    final Map<String, Object> counters2 = new HashMap<>();
                    for (final Map.Entry<String, Long> recipientInfo : recipients.entrySet()) {
                        if (recipientInfo.getValue() != 0L) {
                            ((List<Object>) counters2.computeIfAbsent("uuid", x -> new ArrayList<>()))
                                .add(FreemailShingles.hash(Long.toString(recipientInfo.getValue()),true));
                            ((List<Object>) counters2.computeIfAbsent(key, x -> new ArrayList<>())).add(1L);
                        }
                    }
                    if (counters2.size() > 0) {
                        addonShingles = new FreemailShingles(new HashSet<>(Set.of(FreemailScheme.COUNTERS)), counters2);
                    }
                }
                final FreemailShingles shingles = new FreemailShingles(schemes, counters);
                if (addonShingles != null) {
                    shingles.setAddonShingles(addonShingles);
                }
                if (schemes.size() > 0 || addonShingles != null) {
                    shinglersData().put(ShinglerType.FREEMAIL, shingles);
                }
                return true;
            } catch (Exception e) {
                session().logger().log(Level.SEVERE, "MailMessageContext failed to set up Freemail-shingler's data: "
                    + e, e);
            }
        }
        return false;
    }

    public boolean setUpDefaultActivityShingles() {
        final Set<Long> uids = new HashSet<>(recipients.values());
        uids.add(prefix());
        try {
            shinglersData().put(
                ShinglerType.ACTIVITY,
                new ActivityShingles(
                    new HashSet<>(Set.of(ActivityScheme.COMPLS)),
                    uids.stream().map(x -> new HashMap<String, Object>(Map.of("uid", x, "complaints", 1)))
                        .collect(Collectors.toList())));
            shinglersData().get(ShinglerType.ACTIVITY).addAll(
                new ActivityShingles(
                    new HashSet<>(Set.of(ActivityScheme.INC_COMPL_DAYS)),
                    uids.stream().map(x -> new HashMap<String, Object>(Map.of(
                        "uid", x, "days_with_complaints", 1))).collect(Collectors.toList())));
            session().logger().info("MailMessageContext.setUpDefaultActivityShingles: activityShingles="
                + shinglersData().get(ShinglerType.ACTIVITY));
            return true;
        } catch (Exception e) {
            session().logger().log(Level.SEVERE, "MailMessageContext failed to create Activity-shingler's default "
                + "data: " + e, e);
        }
        return false;
    }

    public void setUpMassShingles(final MassShinglerResult shinglerResult) {
        if (isSpam() != null) {
            shinglersData.put(
                route == Route.OUT ? ShinglerType.MASS_OUT : ShinglerType.MASS_IN,
                shinglerResult.setSpam(isSpam(), isPersonal()));
        }
    }

    public void setUpComplShingles(final Shingles shingles) throws ShingleException  {
        if (isSpam() != null) {
            setUpComplShingles(new ComplShingles(shingles));
        }
    }

    public void setUpComplShingles(final ComplShingles complShingles) {
        if (isSpam() != null) {
            String spamKey = "complaint_" + (isSpam() ? "spam" : "ham");
            List<Object> one = new ArrayList<>();
            one.add(1);
            for (final Map.Entry<ShingleType, Map<Long, GeneralShingleInfo<ComplScheme>>> entry
                    : complShingles.entrySet())
            {
                for (final GeneralShingleInfo<ComplScheme> shingleInfo : entry.getValue().values()) {
                    shingleInfo.computeIfAbsent(ComplScheme.TODAY_ABUSES, x -> new HashMap<>())
                        .put(spamKey, one);
                    shingleInfo.computeIfAbsent(ComplScheme.HISTORY_ABUSES, x -> new HashMap<>())
                        .put(spamKey, one);
                    shingleInfo.computeIfAbsent(ComplScheme.HISTORY_ABUSES_DAYS, x -> new HashMap<>())
                        .put("day_count_with_complaint", one);
                }
            }
            shinglersData.put(ShinglerType.COMPL, complShingles);
        }
    }

    public void setUpSenderShingles(final Shingles massShingles, final String psndrRule) throws ShingleException {
        if (isSpam() != null) {
            SenderShingles senderShingles = new SenderShingles();
            Long domainShingle = null;
            Long emailShingle = null;
            Long psndrShingle = null;
            Map<String, Object> totalCounters = new HashMap<>();
            Map<String, Object> weeksCounters = new HashMap<>();
            totalCounters.put(isSpam() ? "cs" : "ch", 1L);
            if (rules.contains("__YA_DKIM_PASS")) {
                totalCounters.put("dkim_complaint_spam", 1L);
            }
            weeksCounters.put(isSpam() ? "compl_spam" : "compl_ham", 1L);
            if (massShingles.containsKey(ShingleType.FROM_ADDR_SHINGLE)) {
                emailShingle = massShingles.get(ShingleType.FROM_ADDR_SHINGLE).keySet().iterator().next();
            } else if (senderEmail != null && !senderEmail.isEmpty()) {
                emailShingle = Fnv.fnv64(senderEmail);
            }
            if (massShingles.containsKey(ShingleType.FROM_DOMAIN_SHINGLE)) {
                domainShingle = massShingles.get(ShingleType.FROM_DOMAIN_SHINGLE).keySet().iterator().next();
            } else if (senderEmail != null && !senderEmail.isEmpty()) {
                domainShingle = Fnv.fnv64(domain(senderEmail));
            }
            if (psndrRule != null) {
                psndrShingle = Fnv.fnv64(psndrRule);
            }
            if (emailShingle != null) {
                Map<String, Object> emailTotalCounters = new HashMap<>(totalCounters);
                Map<String, Object> emailWeeksCounters = new HashMap<>(weeksCounters);
                emailTotalCounters.put(SenderScheme.SENDER_TOTALS.shingleField(), emailShingle);
                emailTotalCounters.put(SENDERTYPE, (byte) 1);
                emailWeeksCounters.put(SenderScheme.SENDER_TOTALS.shingleField(), emailShingle);
                emailWeeksCounters.put(SENDERTYPE, (byte) 1);
                senderShingles.addShingleInfo(
                    ShingleType.FROM_ADDR_SHINGLE,
                    emailShingle,
                    SenderScheme.SENDER_TOTALS,
                    emailTotalCounters);
                senderShingles.addShingleInfo(
                    ShingleType.FROM_ADDR_SHINGLE,
                    emailShingle,
                    SenderScheme.SENDER_2WEEKS,
                    emailWeeksCounters);
            }
            if (domainShingle != null) {
                Map<String, Object> domainTotalCounters = new HashMap<>(totalCounters);
                Map<String, Object> domainWeeksCounters = new HashMap<>(weeksCounters);
                domainTotalCounters.put(SenderScheme.SENDER_TOTALS.shingleField(), domainShingle);
                domainTotalCounters.put(SENDERTYPE, (byte) 2);
                domainWeeksCounters.put(SenderScheme.SENDER_TOTALS.shingleField(), domainShingle);
                domainWeeksCounters.put(SENDERTYPE, (byte) 2);
                senderShingles.addShingleInfo(
                    ShingleType.FROM_DOMAIN_SHINGLE,
                    domainShingle,
                    SenderScheme.SENDER_TOTALS,
                    domainTotalCounters);
                senderShingles.addShingleInfo(
                    ShingleType.FROM_DOMAIN_SHINGLE,
                    domainShingle,
                    SenderScheme.SENDER_2WEEKS,
                    domainWeeksCounters);
            }
            if (psndrShingle != null) {
                Map<String, Object> psndrTotalCounters = new HashMap<>(totalCounters);
                Map<String, Object> psndrWeeksCounters = new HashMap<>(weeksCounters);
                psndrTotalCounters.put(SenderScheme.SENDER_TOTALS.shingleField(), psndrShingle);
                psndrTotalCounters.put(SENDERTYPE, (byte) 3);
                psndrWeeksCounters.put(SenderScheme.SENDER_TOTALS.shingleField(), psndrShingle);
                psndrWeeksCounters.put(SENDERTYPE, (byte) 3);
                senderShingles.addShingleInfo(psndrShingle, SenderScheme.SENDER_TOTALS, psndrTotalCounters);
                senderShingles.addShingleInfo(psndrShingle, SenderScheme.SENDER_2WEEKS, psndrWeeksCounters);
            }
            shinglersData.put(ShinglerType.SENDER, senderShingles);
        }
    }

    public void setUpUrlShingles(final Shingles shingles) throws ShingleException {
        if (isSpam() != null) {
            shinglersData.put(ShinglerType.URL, new UrlShingles(shingles).setSpam(isSpam()));
        }
    }

    public static boolean isFreemail(final String email) {
        final int i = email.lastIndexOf('@');
        if (i > -1) {
            final String login = email.substring(0, i);
            final String domain = email.substring(i + 1);
            if (login.equals("mailer-daemon")) {
                return false;
            } else if (freemailHosts.contains(domain)) {
                return true;
            } else {
                final String[] parts = domain.split("\\.");
                final int n = parts.length - 1;
                return n > 0 && parts[n].equals("yahoo")
                    || n > 1 && parts[n - 1].equals("yahoo") && (parts[n].equals("co") || parts[n].equals("com"));
            }
        }
        return false;
    }

    public boolean isFreemail() {
        return (from != null) && isFreemail(from);
    }

    public static boolean isYandexEmail(final String email) {
        return email != null && RE_YANDEX_EMAIL.matcher(email).find();
    }

    public static Sources source(final Map<?, ?> json) {
        Object dbUser = json == null || json.isEmpty() ? null : json.get("db_user");
        try {
            return dbUser == null ? Sources.USER : Sources.valueOf(dbUser.toString().toUpperCase(Locale.ROOT));
        } catch (Exception e) {
            return Sources.USER;
        }
    }

    public static String sessionKey(final Map<?, ?> json) {
        Object sKey = json == null || json.isEmpty() ? null : json.get("session_key");
        return sKey == null ? null : (String) sKey;
    }
}
