package ru.yandex.iex.proxy.complaints;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.iex.proxy.AbstractContext;
import ru.yandex.iex.proxy.IexProxy;
import ru.yandex.iex.proxy.move.UpdateDataHolder;
import ru.yandex.util.timesource.TimeSource;

public class UserActionContext implements AbstractUserActionContext {
    protected final IexProxy iexProxy;
    protected final ProxySession session;
    protected final Long uid;
    protected final ImmutableComplaintsConfig complaintsConfig;
    protected volatile Map<Long, UpdateDataHolder> messages;
    protected final Map<UserAction, List<Long>> actions;
    protected String suid;
    protected Long karma = null;
    protected final Map<?, ?> json;
    protected UserAction action;
    protected final long actionDate;
    protected String login;
    protected volatile Map<String, Long> recipients;
    protected volatile Map<String, Long> senderUids;
    protected Sources source;
    protected Boolean showTabs = null;
    protected final long startTime;

    public UserActionContext(final AbstractContext context) {
        iexProxy = context.iexProxy();
        session = context.session();
        complaintsConfig = context.iexProxy().config().complaintsConfig();
        uid = context.prefix();
        suid = context.suid();
        json = context.json();
        action = UserAction.OTHER;
        source = MailMessageContext.source(json);
        messages = new ConcurrentHashMap<>();
        actions = new HashMap<>();
        actionDate = Math.round(context.operationDate());
        recipients = new ConcurrentHashMap<>();
        senderUids = new ConcurrentHashMap<>();
        BlackboxUserinfo userinfo = context.userInfo();
        login = userinfo == null ? null : userinfo.login();
        startTime = TimeSource.INSTANCE.currentTimeMillis();
    }

    public UserActionContext(final AbstractContext context, UserAction action) {
        this(context);
        this.action = action;
        final long now = Math.round(startTime / 1000.0);
        final long weekAgo = now - 7 * 86400;
        if (context.operationDate() > now + 1 || context.operationDate() < weekAgo) {
            complaintsConfig.complaintsError(ComplaintProcessingError.WrongDate, action);
            context.session().logger().warning("UserActionContext: wrong date for move detected!");
        }
    }

    @Override
    public IexProxy iexProxy() {
        return iexProxy;
    }

    @Override
    public ProxySession session() {
        return this.session;
    }

    @Override
    public long prefix() {
        return uid;
    }

    @Override
    public Map<?, ?> json() {
        return this.json;
    }

    @Override
    public long startTime() {
        return startTime;
    }

    @Override
    public String uid() {
        return Long.toString(uid);
    }

    @Override
    public String suid() {
        return suid;
    }

    @Override
    public void setSuid(final String suid) {
        this.suid = suid;
    }

    @Override
    public Long karma() {
        return karma;
    }

    @Override
    public void setKarma(final Long karma) {
        this.karma = karma;
    }

    @Override
    public UserAction action() {
        return action;
    }

    public UserActionContext action(final UserAction action) {
        this.action = action;
        return self();
    }

    @Override
    public Sources source() {
        return source;
    }

    public void setSource(final Sources source) {
        this.source = source;
    }

    public Map<String, Long> recipients() {
        return recipients;
    }

    public Map<String, Long> senderUids() {
        return senderUids;
    }

    @Override
    public long actionDate() {
        return actionDate;
    }

    @Override
    public String login() {
        return login;
    }

    @Override
    public void setLogin(final String login) {
        this.login = login;
    }

    @Override
    public Boolean showTabs() {
        return showTabs;
    }

    @Override
    public void setShowTabs(final boolean showTabs) {
        this.showTabs = showTabs;
    }

    @Override
    public Map<Long, UpdateDataHolder> messages() {
        return messages;
    }

    @Override
    public void messages(final List<UpdateDataHolder> docs) {
        if (docs == null || docs.size() < 1) {
            return;
        }
        for (final UpdateDataHolder doc : docs) {
            if (doc == null) {
                continue;
            }
            messages.put(doc.context().mid(), doc);
            actions.computeIfAbsent(doc.context().action(), x -> new ArrayList<>()).add(doc.context().mid());
            for (final Map.Entry<String, Long> recipientInfo : doc.context().recipients().entrySet()) {
                if (recipients.containsKey(recipientInfo.getKey()) && recipientInfo.getValue() != 0L) {
                    recipients.put(recipientInfo.getKey(), recipientInfo.getValue());
                } else {
                    recipients.put(recipientInfo.getKey(), 0L);
                }
            }
            if (doc.context().senderEmail() == null) {
                session.logger().warning("UserActionContext: senderEmail=null for mid=" + doc.context().mid());
            } else {
                senderUids.put(doc.context().senderEmail(), doc.context().senderUid());
            }
        }
        if (actions.size() > 0) {
            action = actions.keySet().iterator().next();
        }
    }

    @Override
    public Map<UserAction, List<Long>> actions() {
        return actions;
    }

    private UserActionContext self() {
        return this;
    }
}
