package ru.yandex.iex.proxy.complaints;

import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;

import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;

import ru.yandex.dbfields.PgFields;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncPostURIRequestProducerSupplier;
import ru.yandex.iex.proxy.move.UpdateDataExecutor;
import ru.yandex.iex.proxy.move.UpdateDataHolder;

import static java.util.Map.entry;

public class UserActionCountersUpdater implements FutureCallback<List<UpdateDataHolder>> {
    protected static final String UNKNOWN = "unknown";
    protected static final String SO_WEB_GETBYID_URL = "https://web.so.yandex-team.ru/web_fgbd.py?";
    protected static final Map<String, String> labels = Map.ofEntries(
        entry("CL_BOUNCE",       "bounce"),
        entry("CL_PERSONALNEWS", "personalnews"),
        entry("CL_ESHOP",        "eshop"),
        entry("CL_ETICKET",      "eticket"),
        entry("CL_NEWS",         "news"),
        entry("CL_NOTIFICATION", "notification"),
        entry("CL_PEOPLE",       "people"),
        entry("CL_REGISTRATION", "registration"),
        entry("__CL_JOB",        "job"),
        entry("__CL_COUNT",      "count"),
        entry("CL_HOTELBOOK",    "hotel"),
        entry("CL_CANCEL",       "cancel"),
        entry("CL_SCHEMA",       "schema"),
        entry("CL_YAMONEY",      "yamoney"),
        entry("CL_PHISHING",     "phishing"),
        entry("CL_INVITE",       "invite"),
        entry("CL_FIRSTMAIL",    "firstmail")
    );

    protected final AbstractUserActionContext context;
    protected final MultiFutureCallback<String> multiCallback;
    protected final Set<UpdateDataExecutor> updateExecutors;

    public UserActionCountersUpdater(
        final AbstractUserActionContext context,
        final MultiFutureCallback<String> multiCallback)
    {
        this.context = context;
        this.multiCallback = multiCallback;
        updateExecutors = Set.of(UpdateDataExecutor.values());
    }

    public UserActionCountersUpdater(
        final AbstractUserActionContext context,
        final MultiFutureCallback<String> multiCallback,
        final Set<UpdateDataExecutor> updateExecutors)
    {
        this.context = context;
        this.multiCallback = multiCallback;
        this.updateExecutors = updateExecutors;
    }

    @Override
    public void completed(List<UpdateDataHolder> docs) {
        if (docs == null || docs.size() < 1) {
            context.session().logger().info("UserActionCountersUpdater processing skipped");
            context.stat(YandexHttpStatus.SC_OK);
            multiCallback.done();
            return;
        }
        final String addonParams = "&operation-id=" + context.json().get(PgFields.OPERATION_ID);
        if (docs.size() == 1) {
            UpdateDataHolder updateDataHolder = docs.get(0);
            context.session().logger().info("UserActionCountersUpdater: senderDomain="
                + updateDataHolder.senderDomain() + ", senders=" + updateDataHolder.senders() + ", isSpam="
                + updateDataHolder.isSpam());
        }
        for (final UpdateDataExecutor updateDataExecutor : updateExecutors) {
            context.session().logger().info("UserActionCountersUpdater: type = " + updateDataExecutor.name()
                + ", docsCnt = " + docs.size());
            updateDataExecutor.updateCounters(
                context,
                docs,
                TraceFutureCallback.wrap(multiCallback.newCallback(), context, "UpdateDataExecutor_multi"),
                addonParams);
        }
        for (final UpdateDataHolder doc : docs) {
            UserAction action = doc.context().action();
            if (action == UserAction.SPAM || action == UserAction.HAM) {
                FutureCallback<String> callback2 =
                    TraceFutureCallback.wrap(
                        multiCallback.newCallback(),
                        context,
                        "UpdateDataExecutor_FBL-out_multi");
                try {
                    requestFblOutHandler(
                        context,
                        prepareEML(context, doc),
                        action == UserAction.SPAM,
                        callback2);
                } catch (Exception e) {
                    context.session().logger().log(
                        Level.SEVERE,
                        "UserActionCountersUpdater failed to request FBL-out handler: " + e,
                        e);
                    callback2.completed(null);
                }
            } else {
                context.session().logger().info("requestFblOutHandler request skipped because of action="
                    + action.name());
            }
        }
        multiCallback.done();
        postUpdateActions(docs);
        context.stat(YandexHttpStatus.SC_OK);
    }

    @Override
    public void cancelled() {
        context.stat(YandexHttpStatus.SC_CLIENT_CLOSED_REQUEST);
        multiCallback.cancelled();
    }

    @Override
    public void failed(final Exception e) {
        if (e instanceof BadResponseException) {
            context.stat(((BadResponseException) e).statusCode());
        } else {
            context.stat(YandexHttpStatus.SC_REMOTE_CLOSED_REQUEST);
        }
        multiCallback.failed(e);
    }

    public void postUpdateActions(List<UpdateDataHolder> docs) {
        if (docs != null && docs.size() > 0) {
            try {
                if (updateExecutors.contains(UpdateDataExecutor.ABUSES)) {
                    context.iexProxy().abusesPutRequestsStater(docs.size());
                }
                context.session().logger().info("UserActionCountersUpdater: Updated counters for uid="
                    + docs.get(0).uid() + ", action=" + docs.get(0).context().action().name());
            } catch (Exception e) {
                context.session().logger().log(Level.SEVERE, "UserActionCountersUpdater failed: ", e);
            }
        } else {
            context.session().logger().info(
                "UserActionCountersUpdater: there are no items for updating counters for it");
        }
    }

    private static void requestFblOutHandler(
        final AbstractUserActionContext context,
        final String eml,
        final boolean isSpam,
        final FutureCallback<String> callback)
        throws URISyntaxException
    {
        if (eml == null || eml.isEmpty()) {
            context.session().logger().warning("requestFblOutHandler: empty data!");
            callback.completed(null);
            return;
        }
        context.session().logger().info("requestFblOutHandler: preparing request for uid=" + context.uid());
        AsyncClient complaintsClient = context.iexProxy().complaintsClient().adjust(context.session().context());
        String uri = context.iexProxy().config().complaintsConfig().uri().toString() + "?footype="
            + (isSpam ? "foo" : "antifoo");
        AsyncPostURIRequestProducerSupplier complaintsPost =
            new AsyncPostURIRequestProducerSupplier(uri, eml, ContentType.APPLICATION_OCTET_STREAM);
        complaintsClient.execute(
            complaintsPost,
            AsyncStringConsumerFactory.OK,
            context.session().listener().createContextGeneratorFor(complaintsClient),
            TraceFutureCallback.wrap(new AbstractFilterFutureCallback<>(callback) {
                @Override
                public void completed(final String something) {
                    callback.completed(something);
                }

                @Override
                public void cancelled() {
                    context.session().logger().warning(
                        "requestFblOutHandler request cancelled: " + context.session().listener().details());
                    callback.completed(null);
                }

                @Override
                public void failed(final Exception e) {
                    context.session().logger().log(Level.WARNING, "requestFblOutHandler request failed: "
                        + context.session().listener().details() + " because of exception: " + e.toString(), e);
                    callback.completed(null);
                }
            }, context, "AbstractFilterFuture for requestFblOutHandler request for FBL-out"));
    }

    private static String prepareEML(final AbstractUserActionContext context, final UpdateDataHolder doc) {
        final MailMessageContext msg = doc.context();
        final String boundary = "=_NextPart_" + msg.uid() + "_" + msg.stid();
        StringBuilder sb = new StringBuilder();
        sb.append("IY-Complaint: 2\n");
        sb.append("From: ").append(safe(msg.headersMap().get("from") == null || msg.headersMap().get("from").size() < 1
            ? null : msg.headersMap().get("from").get(0))).append('\n');
        if (msg.headersMap().containsKey("to")) {
            sb.append("To: ").append(String.join(", ", msg.headersMap().get("to"))).append('\n');
        } else {
            context.session().logger().warning("prepareEML: Absent 'To' header in complaint's message!");
            return null;
        }
        if (msg.headersMap().containsKey("cc")) {
            sb.append("Cc: ").append(String.join(", ", msg.headersMap().get("cc"))).append('\n');
        }
        if (msg.headersMap().containsKey("subject")) {
            sb.append("Subject: ").append(msg.headersMap().get("subject")).append('\n');
        }
        if (msg.headersMap().containsKey("message-id")) {
            sb.append("Message-ID: ").append(msg.headersMap().get("message-id")).append('\n');
        }
        if (msg.headersMap().containsKey("return-path")) {
            sb.append("Return-Path: ").append(msg.headersMap().get("return-path")).append('\n');
        }
        sb.append("Date: ").append(
            MailMessageContext.formatDate(msg.actionDate(), MailMessageContext.TimestampFormat.MSG_HEADER))
            .append('\n');
        sb.append("MIME-Version: 1.0\n");
        sb.append("Content-Type: multipart/mixed;\n");
        sb.append("\tboundary=\"").append(boundary).append("\"\n");
        sb.append("IY-ComplRequest: &source=").append(msg.source().name().toLowerCase(Locale.ROOT)).append("&type=")
            .append(msg.action() == UserAction.SPAM ? "foo" : "antifoo").append("&uid=").append(msg.uid()).append("\n");
        if (context.karma() != null) {
            sb.append("X-Karma: ").append(context.karma()).append("\n");
        }
        if (msg.queueId() != null) {
            sb.append("X-Yandex-QueueID: ").append(msg.queueId()).append("\n");
        }
        if (msg.route() != null) {
            sb.append("X-Yandex-Route: ").append(msg.route()).append(" (").append(msg.mailBackend()).append(")")
                .append("\n");
        }
        if (msg.mailBackend() != null) {
            sb.append("X-Yandex-So-Front: ").append(msg.mailBackend()).append("\n");
        }
        final List<String> msgLabels = getLabels(msg);
        if (msgLabels.size() > 0) {
            sb.append("X-SO-Label: ").append(String.join(" ", msgLabels)).append("\n");
        }
        if (msg.geoZone() != null) {
            sb.append("X-SO-Geo: ").append(msg.geoZone()).append("\n");
        }
        sb.append("\nThis is a multi-part message in MIME format.\n\n--").append(boundary).append("\n");
        sb.append("Content-Type: text/html; charset=\"koi8-r\"\n\n<html>\n<body>\n");
        if (msg.queueId() != null) {
            sb.append("<a href=\"" + SO_WEB_GETBYID_URL + "qid=").append(msg.queueId()).append("\">")
                .append(SO_WEB_GETBYID_URL).append("qid=").append(msg.queueId()).append("</a><br>\n");
        }
        if (msg.msgId() != null) {
            sb.append("<a href=\"" + SO_WEB_GETBYID_URL + "msgid=").append(msg.msgId()).append("\">")
                .append(SO_WEB_GETBYID_URL).append("msgid=").append(msg.msgId()).append("</a><br>\n");
        }
        sb.append("\n</body>\n</html>\n\n--").append(boundary).append("\nContent-Type: message/rfc822;\n");
        sb.append("Content-Disposition: attachment;\n");
        String login = context.login() == null ? msg.recipientEmail() : context.login();
        if (login == null) {
            login = "_";
        }
        if (login.contains("\n")) {
            login = login.substring(0, login.indexOf('\n'));
        }
        sb.append("\tfilename=\"").append(login).append("_").append(msg.uid()).append(".eml\"\n\n");
        for (final Map.Entry<String, List<String>> headerInfo : msg.headersMap().entrySet()) {
            StringBuilder header = new StringBuilder();
            for (String h : headerInfo.getKey().split("-")) {
                if (!header.toString().isEmpty()) {
                    header.append('-');
                }
                header.append(h.substring(0, 1).toUpperCase(Locale.ROOT)).append(h.substring(1));
            }
            for (final String headerVal : headerInfo.getValue()) {
                sb.append(header).append(": ").append(headerVal).append("\n");
            }
        }
        sb.append("\n\nFAKE BODY\n\n\n--").append(boundary).append("--\n\n");
        return new String(sb);
    }

    private static String safe(final String s) {
        return s == null ? UserActionCountersUpdater.UNKNOWN : s;
    }

    private static List<String> getLabels(final MailMessageContext msg) {
        final List<String> msgLabels = new ArrayList<>();
        for (final Map.Entry<String, String> labelInfo : labels.entrySet()) {
            if (msg.rules().contains(labelInfo.getKey())) {
                msgLabels.add(labelInfo.getValue());
            }
        }
        return msgLabels;
    }
}
