package ru.yandex.iex.proxy.complaints;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;

import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.client.so.shingler.ComplScheme;
import ru.yandex.client.so.shingler.ComplShingles;
import ru.yandex.client.so.shingler.GeneralShingleInfo;
import ru.yandex.client.so.shingler.ShingleType;
import ru.yandex.client.so.shingler.config.ShinglerType;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.iex.proxy.AbstractHandlersContext;
import ru.yandex.iex.proxy.IexProxy;
import ru.yandex.iex.proxy.move.UpdateDataExecutor;
import ru.yandex.iex.proxy.move.UpdateDataHolder;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.util.timesource.TimeSource;

public interface UserActionHandler {
    enum IndexDataType {
        MSG_ABUSES(100, "abuses_"),
        USER_DAY_ABUSES(150, "abuses_"),
        USER_ABUSES(250, "abuses_"),
        MESSAGE(300, ""),
        SOLOG(250, "solog_");

        private final long timeout;
        private final String keyPrefix;

        IndexDataType(final long timeout, final String keyPrefix) {
            this.timeout = timeout;
            this.keyPrefix = keyPrefix;
        }

        public long timeout() {
            return timeout;
        }

        public String keyPrefix() {
            return keyPrefix;
        }
    }

    static void requestIndex(
        final AbstractHandlersContext context,
        final String query,
        final IndexDataType dataType,
        final FutureCallback<JsonObject> callback)
    {
        final IexProxy iexProxy = context.iexProxy();
        AsyncClient client = iexProxy.searchClient().adjust(context.session().context());
        UniversalSearchProxyRequestContext requestContext =
            new PlainUniversalSearchProxyRequestContext(
                new User(iexProxy.config().factsIndexingQueueName(), new LongPrefix(context.prefix())),
                null,
                true,
                client,
                context.session().logger());
        iexProxy.sequentialRequest(
            context.session(),
            requestContext,
            new BasicAsyncRequestProducerGenerator(query),
            dataType.timeout(),
            true,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.session().listener().createContextGeneratorFor(client),
            callback);
    }

    static String getIndexRequest(
        final AbstractUserActionContext context,
        final IndexDataType dataType,
        final String filterExpr)
        throws BadRequestException
    {
        Set<String> ids = new HashSet<>();
        if (dataType == IndexDataType.SOLOG || dataType == IndexDataType.MESSAGE) {
            for (final UpdateDataHolder data : context.messages().values()) {
                final MailMessageContext msgInfo = data.context();
                if (msgInfo.queueId() == null && (msgInfo.allSmtpIds() == null || msgInfo.allSmtpIds().size() < 1)) {
                    continue;
                }
                if (msgInfo.allSmtpIds() == null || msgInfo.allSmtpIds().size() < 1) {
                    ids.add(msgInfo.queueId());
                } else {
                    ids.addAll(msgInfo.allSmtpIds());
                }
            }
            if (ids.size() == 0) {
                return null;
            }
        }
        return getIndexRequest(
            context.uid(),
            context.iexProxy().config().factsIndexingQueueName(),
            ids,
            dataType,
            filterExpr,
            true);
    }

    static String getIndexRequest(
        final String uid,
        final String service,
        final Collection<String> queueIds,
        final IndexDataType dataType,
        final String filterExpr,
        final boolean dollarJson)
        throws BadRequestException
    {
        QueryConstructor query = new QueryConstructor("/search?", false);
        query.append("prefix", uid);
        query.append("service", service);
        if (dollarJson) {
            query.append("json-type", "dollar");
        }
        switch (dataType) {
            case MSG_ABUSES:
                query.append("get", "url,abuses_last_timestamp");
                query.append("text","url:" + filterExpr + "");
                break;
            case USER_DAY_ABUSES:
                final long now = Math.round(TimeSource.INSTANCE.currentTimeMillis() / 1000.0);
                query.append("get", "url,abuses_cnt");
                query.append("text","abuses_uid:(" + uid + ") AND abuses_cmpl_date_hour_p:["
                    + (now - 86400) + " TO " + now + "]");
                break;
            case USER_ABUSES:
                query.append(
                    "get",
                    "url,abuses_cnt,abuses_smtp_id,abuses_type,abuses_source,abuses_msg_date,abuses_cmpl_date,"
                        + "abuses_first_timestamp,abuses_last_timestamp");
                query.append("text","abuses_uid:" + uid);
                break;
            case MESSAGE:
                query.append("get", "url,uid,mid,stid");
                query.append("text","all_smtp_ids:(" + String.join(" OR ", queueIds) + ")");
                break;
            case SOLOG:
                query.append("get", "url,solog_data,solog_smtp_id,solog_route,solog_so_res");
                query.append("text","solog_smtp_id_p:(" + String.join(" OR ", queueIds) + ")");
                break;
        }
        return query.toString();
    }

    static void requestRulesAndShingles(
        final AbstractUserActionContext context,
        final FutureCallback<JsonObject> callback)
    {
        String query;
        try {
            query = getIndexRequest(context, IndexDataType.SOLOG, null);
        } catch (BadRequestException e) {
            context.session().logger().log(Level.SEVERE, "requestRulesAndShingles failed: " + e, e);
            query = null;
        }
        if (query == null) {
            callback.completed(null);
            return;
        }
        requestIndex(context, query, IndexDataType.SOLOG, callback);
    }

    static void requestMessageInfo(
        final AbstractUserActionContext context,
        final FutureCallback<JsonObject> callback)
    {
        String query;
        try {
            query = getIndexRequest(context, IndexDataType.MESSAGE, null);
        } catch (BadRequestException e) {
            context.session().logger().log(Level.SEVERE, "requestMessageInfo failed: " + e, e);
            query = null;
        }
        if (query == null) {
            callback.completed(null);
            return;
        }
        requestIndex(context, query, IndexDataType.MESSAGE, callback);
    }

    static void requestMsgAbuses(
        final AbstractUserActionContext context,
        final List<Long> mids,
        final boolean isSpam,
        final FutureCallback<JsonObject> callback)
    {
        String query;
        try {
            List<String> urls = new ArrayList<>();
            for (final Long mid : mids) {
                final MailMessageContext msgInfo = context.messages().get(mid).context();
                if (msgInfo.stid() != null && !msgInfo.stid().isEmpty()) {
                    urls.add(UpdateDataExecutor.abusesUrl(
                        context.uid(),
                        msgInfo.stid().replace(":", "\\:"),
                        isSpam ? "spam" : "ham"));
                }
            }
            query = getIndexRequest(
                context,
                IndexDataType.MSG_ABUSES,
                "(" + String.join(" OR ", urls) + ")");
        } catch (BadRequestException e) {
            context.session().logger().log(Level.SEVERE, "requestMsgAbuses failed: " + e, e);
            query = null;
        }
        if (query == null) {
            callback.completed(null);
            return;
        }
        requestIndex(context, query, IndexDataType.MSG_ABUSES, callback);
    }

    static void requestUserDayAbuses(final AbstractUserActionContext context, final FutureCallback<JsonObject> callback)
    {
        String query;
        try {
            query = getIndexRequest(context, IndexDataType.USER_DAY_ABUSES, null);
        } catch (BadRequestException e) {
            context.session().logger().log(Level.SEVERE, "requestUserDayAbuses failed: " + e, e);
            query = null;
        }
        if (query == null) {
            callback.completed(null);
            return;
        }
        requestIndex(context, query, IndexDataType.USER_DAY_ABUSES, callback);
    }

    static void requestSologger(
        final AbstractUserActionContext context,
        final String query,
        final FutureCallback<String> callback)
    {
        IexProxy iexProxy = context.iexProxy();
        AsyncClient client = iexProxy.sologgerClient();
        BasicAsyncRequestProducerGenerator request = new BasicAsyncRequestProducerGenerator(query);
        client = client.adjust(context.session().context());
        context.session().logger().info("Send request to Sologger: " + query);
        client.execute(
            iexProxy.config().sologgerConfig().host(),
            request,
            AsyncStringConsumerFactory.OK,
            context.session().listener().createContextGeneratorFor(client),
            callback);
    }

    static void requestTikaite(
        final AbstractHandlersContext context,
        final String uriPath,
        final List<String> params,
        final FutureCallback<JsonObject> callback)
    {
        final StringBuilder uri = new StringBuilder();
        uri.append(uriPath).append("?json-type=dollar");
        for (final String param : params) {
            uri.append('&').append(param);
        }
        BasicAsyncRequestProducerGenerator request = new BasicAsyncRequestProducerGenerator(new String(uri));
        request.addHeader(YandexHeaders.X_YA_SERVICE_TICKET, context.iexProxy().tikaiteTvm2Ticket());
        request.addHeader(YandexHeaders.X_SRW_SERVICE_TICKET, context.iexProxy().unistorageTvm2Ticket());
        AsyncClient client;
        HttpHost host;
        if (uriPath.startsWith("/extract")) {
            client = context.iexProxy().tikaiteMlClient();
            host = context.iexProxy().tikaiteMlHost();
        } else {
            client = context.iexProxy().tikaiteClient();
            host = context.iexProxy().tikaiteHost();
        }

        client = client.adjust(context.session().context());
        client.execute(
            host,
            request,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.session().listener().createContextGeneratorFor(client),
            callback);
    }

    static void doFinalComplaintProcessing(final AbstractUserActionContext context, final MailMessageContext msgInfo) {
        if (msgInfo.isRetry()) {
            msgInfo.shinglersData().setIsRetry(true);
        } else {
            if (!msgInfo.setUpDefaultActivityShingles()) {
                msgInfo.shinglersData().remove(ShinglerType.ACTIVITY);
            }
            if (msgInfo.route() == Route.OUT && context.login() != null && !context.login().isEmpty()
                    && msgInfo.returnPath() != null && !msgInfo.returnPath().isEmpty()
                    && context.source() != Sources.FBL)
            {
                String sender = !context.login().contains("@") && msgInfo.returnPath().contains("@")
                    ? msgInfo.returnPath().substring(0, msgInfo.returnPath().indexOf('@')) : msgInfo.returnPath();
                if (context.login().equals(sender)) {
                    msgInfo.skipReasons().add(SkipReason.ON_OWN_LETTERS);
                    context.session().logger().info("UserActionHandler.doFinalProcessing: complaint ON_OWN_LETTERS, "
                        + "login='" + context.login() + "', sender='" + sender + "'");
                }
            }
            msgInfo.setUpFreemailShingles();
            msgInfo.fillFlags();
            context.iexProxy().complLogger().fine(msgInfo.getComplLogRow());
            if (msgInfo.flags().contains(Flags._XX) || msgInfo.rules().contains("SEO_ABUSE")) {
                if (msgInfo.flags().contains(Flags._XX)) {
                    context.session().logger().info("UserActionHandler.doFinalProcessing: complaint WAS");
                } else if (msgInfo.rules().contains("SEO_ABUSE")) {
                    context.session().logger().info("UserActionHandler.doFinalProcessing: complaint from SEO-abuses");
                }
                for (final ShinglerType shinglerType
                        : Set.of(ShinglerType.COMPL, ShinglerType.FREEMAIL, ShinglerType.SENDER, ShinglerType.URL))
                {
                    if (msgInfo.shinglersData().containsKey(shinglerType)) {
                        context.session().logger().info("UserActionHandler.doFinalProcessing: removing of data for "
                            + shinglerType.name() + "-shingler");
                        msgInfo.shinglersData().remove(shinglerType);
                    }
                }
                if (context.source() == Sources.FBL) {
                    for (final ShinglerType shinglerType : msgInfo.shinglersData().keySet()) {
                        context.session().logger().info("UserActionHandler.doFinalProcessing: removing of data for "
                            + shinglerType.name() + "-shingler");
                        msgInfo.shinglersData().remove(shinglerType);
                    }
                }
            } else {
                try {
                    if (msgInfo.shinglersData().containsKey(ShinglerType.COMPL)) {
                        ComplShingles complShingles = (ComplShingles) msgInfo.shinglersData().get(ShinglerType.COMPL);
                        List<ComplScheme> nonRelevantSchemes = List.of(
                            ComplScheme.USER_WEIGHTS_ALL,
                            ComplScheme.USER_WEIGHTS,
                            ComplScheme.HISTORY_VIRUS_DAYS,
                            ComplScheme.SHINGLE,
                            ComplScheme.OLDER);
                        for (Map.Entry<ShingleType, Map<Long, GeneralShingleInfo<ComplScheme>>> shingleTypeData
                                : complShingles.entrySet())
                        {
                            for (GeneralShingleInfo<ComplScheme> shingleInfo : shingleTypeData.getValue().values()) {
                                for (ComplScheme scheme : nonRelevantSchemes) {
                                    shingleInfo.remove(scheme);
                                }
                            }
                        }
                        context.session().logger().info("UserActionHandler.doFinalProcessing: complShingles = "
                            + complShingles);
                    }
                    if (msgInfo.shinglersData().containsKey(ShinglerType.FREEMAIL)
                            && (msgInfo.route() != Route.OUT
                            || !MailMessageContext.isYandexEmail(msgInfo.senderEmail())))
                    {
                        msgInfo.shinglersData().remove(ShinglerType.FREEMAIL);
                    }
                } catch (Exception e) {
                    context.session().logger().log(
                        Level.SEVERE,
                        "UserActionHandler.doFinalProcessing failed to prepare shinglers' counters",
                        e);
                }
            }
        }
    }

    static void doFinalComplaintProcessing(
        final AbstractUserActionContext context,
        final FutureCallback<? super List<UpdateDataHolder>> callback)
    {
        for (final UpdateDataHolder data : context.messages().values()) {
            doFinalComplaintProcessing(context, data.context());
        }
        doFinalUserActionProcessing(context, callback);
    }

    static void doFinalUserActionProcessing(
        final AbstractUserActionContext context,
        final FutureCallback<? super List<UpdateDataHolder>> callback)
    {
        List<UpdateDataHolder> docs = new ArrayList<>();
        try {
            ImmutableComplaintsConfig complaintsConfig = context.iexProxy().config().complaintsConfig();
            for (final UpdateDataHolder data : context.messages().values()) {
                for (Flags flag : data.context().flags()) {
                    complaintsConfig.flagStater(flag, UserAction.OTHER);
                    if (context.action() != UserAction.OTHER) {
                        complaintsConfig.flagStater(flag, context.action());
                    }
                }
                writeComplYtLogRow(context, data.context());
                docs.add(data);
            }
        } catch (Exception e) {
            context.session().logger().log(Level.SEVERE, "UserActionHandler.doFinalUserActionProcessing failed", e);
        }
        callback.completed(docs);
    }

    static void writeComplYtLogRow(final AbstractUserActionContext context, final MailMessageContext msgInfo) {
        try {
            if (!msgInfo.isRetry()) {
                msgInfo.iexProxy().complYtLogger().fine(
                    msgInfo.getYtLogRow(context.suid(), context.karma(), context.showTabs()));
            }
        } catch (Exception e) {
            msgInfo.session().logger().log(Level.SEVERE, "UserActionHandler.writeComplYtLogRow failed: " + e, e);
        }
    }
}
