package ru.yandex.iex.proxy.complaints;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.blackbox.BlackboxDbfield;
import ru.yandex.blackbox.BlackboxUserIdType;
import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.blackbox.BlackboxUserinfoRequest;
import ru.yandex.blackbox.SingleUserinfoBlackboxCallback;
import ru.yandex.http.config.ImmutableURIConfig;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.iex.proxy.move.UpdateDataHolder;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.stater.RequestInfo;
import ru.yandex.util.timesource.TimeSource;

public class UserActionProcessCallback
    extends AbstractFilterFutureCallback<List<UpdateDataHolder>, List<UpdateDataHolder>>
{
    protected static final int BATCH_QUEUEIDS_SIZE = 24;
    protected static final String COMPL = "compl";
    protected static final String RULES = "rules";
    protected static final String ROUTE = "route";
    protected static final String SO_RES = "so_res";
    protected static final String MAIL_BACKEND = "backend";
    protected static final String GEO = "geo";
    protected static final String SENDER_IP = "ip";

    protected final AbstractUserActionContext context;
    protected final ImmutableComplaintsConfig complaintsConfig;

    public enum BlackBoxRequestType {
        COMPLAINANT, SENDER, RECIPIENT
    }

    public UserActionProcessCallback(
        final AbstractUserActionContext context,
        final FutureCallback<List<UpdateDataHolder>> callback)
    {
        super(callback);
        this.context = context;
        complaintsConfig = context.iexProxy().config().complaintsConfig();
    }

    @Override
    public void completed(final List<UpdateDataHolder> resultList) {
        if (resultList == null || resultList.size() < 1) {
            callback.completed(null);
            return;
        }
        context.messages(resultList);
        MultiFutureCallback<Void> multiCallback =
            new MultiFutureCallback<>(
                TraceFutureCallback.wrap(new AbstractFilterFutureCallback<List<Void>, List<UpdateDataHolder>>(callback)
                {
                    @Override
                    public void completed(List<Void> unused) {
                        for (final UpdateDataHolder data : context.messages().values()) {
                            final MailMessageContext msgContext = data.context();
                            for (final String recipient : msgContext.recipients().keySet()) {
                                Matcher m = MailMessageContext.RE_YANDEX_EMAIL.matcher(recipient);
                                if (m.find() && m.group(1).equals(context.login())
                                        && context.corp() == m.group(2).contains("yandex-team"))
                                {
                                    msgContext.setRecipientEmail(recipient);
                                    msgContext.recipients().put(recipient, context.prefix());
                                    context.recipients().put(recipient, context.prefix());
                                }
                            }
                        }
                        finalizing();
                    }

                    @Override
                    public void cancelled() {
                        context.session().logger().warning(
                            "UserActionProcessCallback.completed.MultiFutureCallback request cancelled: "
                                + context.session().listener().details());
                        finalProcessing();
                    }

                    @Override
                    public void failed(final Exception e) {
                        context.session().logger().log(
                            Level.WARNING,
                            "UserActionProcessCallback.completed.MultiFutureCallback failed: "
                                + context.session().listener().details() + " because of exception: " + e.toString(),
                            e);
                        finalProcessing();
                    }
                }, context, "AbstractFilterFuture for UserActionProcessCallback.completed.MultiFutureCallback"));
        try {
            for (final Map.Entry<UserAction, List<Long>> actionInfo : context.actions().entrySet()) {
                if (complaintsConfig.useSolog()) {
                    UserActionHandler.requestRulesAndShingles(
                        context,
                        sologCallback(
                            actionInfo.getKey(),
                            actionInfo.getValue(),
                            TraceFutureCallback.wrap(multiCallback.newCallback(), context, "SologInfo_multi")));
                }
                if (complaintsConfig.useSologger()) {
                    MailMessageContext msgContext;
                    StringBuilder queueIdList = new StringBuilder();
                    Map<List<String>, Long> queueIdMids = new HashMap<>();
                    List<String> queueIds = new ArrayList<>();
                    List<String> msgQueueIds;
                    String route = null;
                    for (Long mid : actionInfo.getValue()) {
                        msgContext = context.messages().get(mid).context();
                        if (route == null) {
                            route = msgContext.route().lowerName();
                        }
                        msgQueueIds = new ArrayList<>();
                        if (msgContext.allSmtpIds() != null && msgContext.allSmtpIds().size() > 0) {
                            msgQueueIds.addAll(msgContext.allSmtpIds());
                        } else {
                            msgQueueIds.add(msgContext.queueId());
                        }
                        queueIds.addAll(msgQueueIds);
                        queueIdMids.put(msgQueueIds, mid);
                        for (String queueId : msgQueueIds) {
                            if (queueIdList.length() > 0) {
                                queueIdList.append(',');
                            }
                            queueIdList.append(queueId);
                        }
                        if (queueIds.size() > BATCH_QUEUEIDS_SIZE) {
                            safeSologgerRequest(route, queueIdList.toString(), queueIdMids, multiCallback);
                            queueIdList = new StringBuilder();
                            queueIds = new ArrayList<>();
                            queueIdMids = new HashMap<>();
                        }
                    }
                    if (queueIds.size() > 0) {
                        safeSologgerRequest(route, queueIdList.toString(), queueIdMids, multiCallback);
                    }
                } else {
                    context.session().logger().info(
                        "UserActionProcessCallback: sologger is not used");
                }
            }
            String login = context.login() == null ? null : context.login().trim();
            FutureCallback<Void> callback2 =
                TraceFutureCallback.wrap(multiCallback.newCallback(), context, "UserLogin_multi");
            if (login == null || login.isEmpty() || login.contains("\n")) {
                if (context.source() == Sources.FBL) {
                    callback2.completed(null);
                } else {
                    requestUserLogin(context, callback2);
                }
            } else {
                final int i = login.lastIndexOf('@');
                if (i > -1 && login.substring(i + 1).startsWith("yandex.")) {
                    context.setLogin(login.substring(0, i));
                }
                callback2.completed(null);
            }
            auxiliaryRequests(multiCallback);
        } catch (Exception e) {
            context.session().logger().info(
                "UserActionProcessCallback failed to process requests to auxiliary sources");
        }
        multiCallback.done();
    }

    protected void finalizing() {
        finalProcessing();
    }

    protected void auxiliaryRequests(final MultiFutureCallback<Void> multiCallback) {
    }

    @Override
    public void cancelled() {
        context.session().logger().warning(
            "UserActionProcessCallback request cancelled: " + context.session().listener().details());
        callback.cancelled();
    }

    @Override
    public void failed(final Exception e) {
        context.session().logger().log(Level.WARNING, "UserActionProcessCallback failed: "
            + context.session().listener().details() + " because of exception: " + e.toString(), e);
        callback.failed(e);
    }

    protected void finalProcessing() {
        UserActionHandler.doFinalUserActionProcessing(context, callback);
    }

    protected FutureCallback<JsonObject> sologCallback(
        final UserAction action,
        final List<Long> mids,
        final FutureCallback<Void> callback)
    {
        return TraceFutureCallback.wrap(new SologInfoCallback(context, action, mids, callback), context, "SologInfo");
    }

    protected static class SologInfoCallback extends AbstractFilterFutureCallback<JsonObject, Void>
    {
        protected final AbstractUserActionContext context;
        protected final ImmutableComplaintsConfig complaintsConfig;
        protected final Map<Route, Map<Integer, String>> rulesDictionaries;
        protected final UserAction action;
        protected final Map<String, Long> queueIdMids;      // Map: QueueID -> mid
        protected final List<Long> mids;                    // List of mids
        protected final HashSet<Long> badMessages;          // Set: mid of messages with failed search of the rules
        protected final long start = TimeSource.INSTANCE.currentTimeMillis();

        public SologInfoCallback(
            final AbstractUserActionContext context,
            final UserAction action,
            final List<Long> mids,
            final FutureCallback<Void> callback)
        {
            super(callback);
            this.context = context;
            this.action = action;
            this.mids = mids;
            queueIdMids = new HashMap<>();
            for (final Map.Entry<Long, UpdateDataHolder> entry : context.messages().entrySet()) {
                if (entry.getValue().queueId() != null) {
                    queueIdMids.put(entry.getValue().queueId(), entry.getKey());
                }
            }
            badMessages = new HashSet<>(mids);
            complaintsConfig = context.iexProxy().config().complaintsConfig();
            rulesDictionaries = complaintsConfig.rulesDictionaries();
        }

        protected void stat(final int status) {
            final long now = TimeSource.INSTANCE.currentTimeMillis();
            complaintsConfig.soSearchRequestsStater().accept(new RequestInfo(now, status, start, start, 0L, 0L));
        }

        protected void lag() {
            final long now = TimeSource.INSTANCE.currentTimeMillis();
            final long lag = Math.round((now - context.actionDate() * 1000L) / 1000.0);
            complaintsConfig.complaintsLagStater().accept(lag);
            context.session().logger().info("SologInfoCallback: lag=" + lag + "; actionInfo: "
                + context.action().name());
            if (lag > 3600 || lag < -1800) {
                context.session().logger().warning("SologInfoCallback: wrong date for move detected: "
                    + context.actionDate());
            }
        }

        @Override
        public void completed(final JsonObject result) {
            if (result == null) {
                callback.completed(null);
                return;
            }
            stat(YandexHttpStatus.SC_OK);
            lag();
            try {
                JsonList hits = result.asMap().getList("hitsArray");
                for (final JsonObject hit : hits) {
                    String url = hit.get("url").asString();
                    String sologDataStr = hit.get("solog_data").asString();
                    String sologSmtpId = hit.get("solog_smtp_id").asString();
                    context.session().logger().info("SologInfoCallback.completed: url=" + url + ", queueid="
                        + sologSmtpId + ", data=" + sologDataStr);
                    if (sologSmtpId == null || sologSmtpId.isEmpty()) {
                        continue;
                    }
                    for (final Long mid : mids) {
                        if (context.messages().get(mid).queueId() != null
                                && context.messages().get(mid).queueId().equals(sologSmtpId)
                                || context.messages().get(mid).context().allSmtpIds().contains(sologSmtpId))
                        {
                            queueIdMids.put(sologSmtpId, mid);
                        }
                    }
                    processSologData(sologSmtpId, TypesafeValueContentHandler.parse(sologDataStr).asMap());
                    if (queueIdMids.containsKey(sologSmtpId)) {
                        badMessages.remove(queueIdMids.get(sologSmtpId));
                    }
                }
                for (final Long ignored : badMessages) {
                    complaintsConfig.complaintsError(ComplaintProcessingError.ParsingRulesFalied, context.action());
                }
                callback.completed(null);
            } catch (JsonException e) {
                complaintsConfig.complaintsError(ComplaintProcessingError.ParsingRulesFalied, context.action());
                context.session().logger().log(Level.SEVERE, "SologInfoCallback failed to parse response", e);
                callback.failed(e);
            }
        }

        @Override
        public void cancelled() {
            stat(YandexHttpStatus.SC_CLIENT_CLOSED_REQUEST);
            context.session().logger().warning(
                "SologInfoCallback request cancelled: " + context.session().listener().details());
            callback.completed(null);
        }

        @Override
        public void failed(final Exception e) {
            if (e instanceof ServerException) {
                stat(((ServerException) e).statusCode());
            } else {
                stat(YandexHttpStatus.SC_REMOTE_CLOSED_REQUEST);
            }
            context.session().logger().log(Level.WARNING, "SologInfoCallback failed to find message's info: "
                + context.session().listener().details() + " because of exception: ", e);
            callback.completed(null);
        }

        protected void processSologData(final String queueId, final JsonMap sologData) throws JsonException {
            final MailMessageContext msgContext = context.messages().get(queueIdMids.get(queueId)).context();
            if (sologData.containsKey(SO_RES) && sologData.get(SO_RES) != null) {
                msgContext.setSoRes(sologData.get(SO_RES).asString());
            }
            if (sologData.containsKey(MAIL_BACKEND) && sologData.get(MAIL_BACKEND) != null) {
                msgContext.setMailBackend(sologData.get(MAIL_BACKEND).asString());
            }
            if (sologData.containsKey(GEO) && sologData.get(GEO) != null) {
                msgContext.setGeoZone(sologData.get(GEO).asString());
            }
            if (sologData.containsKey(SENDER_IP) && sologData.get(SENDER_IP) != null) {
                msgContext.setSenderIp(sologData.get(SENDER_IP).asString());
            }
            if (sologData.containsKey(ROUTE) && sologData.get(ROUTE) != null) {
                msgContext.setRoute(sologData.get(ROUTE).asString());
            }
            boolean rulesLoaded = false;
            if (sologData.containsKey(COMPL) && sologData.get(COMPL) != null) {
                try {
                    if (sologData.get(COMPL) instanceof JsonMap) {
                        JsonMap compl = sologData.get(COMPL).asMap();
                        rulesLoaded = loadRules(queueId, compl);
                    }
                } catch (Exception e) {
                    context.session().logger().log(Level.SEVERE, "SologInfoCallback: extracting of rules failed", e);
                }
            }
            if (!rulesLoaded) {
                rulesLoaded = loadRules(queueId, sologData);
            }
            if (!rulesLoaded) {
                complaintsConfig.complaintsError(ComplaintProcessingError.ParsingRulesFalied, context.action());
            }
        }

        protected boolean loadRules(final String queueId, final JsonMap jsonMap) throws JsonException {
            final MailMessageContext msgContext = context.messages().get(queueIdMids.get(queueId)).context();
            if (jsonMap.containsKey(RULES) && jsonMap.get(RULES) != null) {
                Object rulesObj = jsonMap.get(RULES);
                if (rulesObj instanceof JsonList) {
                    JsonList rulesInfo = ((JsonList) rulesObj).asList();
                    for (JsonObject item : rulesInfo) {
                        String rule;
                        if (item.type() == JsonObject.Type.LONG) {
                            rule = rulesDictionaries.get(msgContext.route()).get((int) item.asLong());
                            if (rule == null) {
                                context.session().logger().info("SologInfoCallback.loadRules: unable find rule "
                                    + "with index=" + item.asLong() + ", route=" + msgContext.route());
                                rule = item.asString();
                            }
                        } else {
                            rule = item.asString();
                        }
                        if (rule != null) {
                            msgContext.rules().add(rule);
                        }
                    }
                    return true;
                } else if (rulesObj instanceof JsonMap) {
                    JsonMap rulesInfo = ((JsonMap) rulesObj).asMap();
                    for (String rule : rulesInfo.keySet()) {
                        if (rule != null) {
                            msgContext.rules().add(rule);
                        }
                    }
                    return true;
                } else {
                    context.session().logger().info("SologInfoCallback: rules object has another type!");
                }
            } else {
                context.session().logger().info("SologInfoCallback: can not find rules in sologData!");
            }
            return false;
        }
    }

    protected void safeSologgerRequest(
        final String route,
        final String queueIds,
        final Map<List<String>, Long> queueIdMids,
        final MultiFutureCallback<Void> multiCallback)
    {
        try {
            if (route == null) {
                for (String possibleRoute
                        : List.of(Route.IN.lowerName(), Route.OUT.lowerName(), Route.CORP.lowerName()))
                {
                    sologgerRequest(possibleRoute, queueIds, queueIdMids, multiCallback.newCallback());
                }
            } else {
                sologgerRequest(route, queueIds, queueIdMids, multiCallback.newCallback());
            }
        } catch (BadRequestException e) {
            context.session().logger().log(Level.SEVERE, "Sologger request building error", e);
        }
    }

    protected void sologgerRequest(
        final String route,
        final String queueId,
        final Map<List<String>, Long> queueIdMids,
        final FutureCallback<Void> callback)
        throws BadRequestException
    {
        ImmutableURIConfig sologgerConfig = context.iexProxy().config().sologgerConfig();
        QueryConstructor query = new QueryConstructor(
            new StringBuilder(sologgerConfig.uri().toASCIIString()).append(sologgerConfig.firstCgiSeparator()));
        query.append("queueid", queueId);
        query.append("rcpt_uid", context.uid());
        query.append("route", route);
        UserActionHandler.requestSologger(
            context,
            query.toString(),
            sologgerCallback(queueIdMids, TraceFutureCallback.wrap(callback, context, "Sologger_multi_" + route)));
    }

    protected FutureCallback<String> sologgerCallback(
        final Map<List<String>, Long> queueIdMids,
        final FutureCallback<Void> callback)
    {
        return TraceFutureCallback.wrap(new SologgerCallback(context, queueIdMids, callback), context, "Sologger");
    }

    protected static class SologgerCallback extends AbstractFilterFutureCallback<String, Void>
    {
        protected final static Pattern RCPT_UID = Pattern.compile("\\bUid = (\\d+)");
        protected final static Pattern ORCP_UID = Pattern.compile("\\buid=(\\d+)");
        protected final static String X_YANDEX_QUEUEID = "x-yandex-queueid";
        protected final static String X_YANDEX_SO_FRONT = "x-yandex-so-front";
        protected final static String IY_GEOZONE = "iy-geozone";
        protected final static String ORCP = "orcp";
        protected final static String RCVD = "rcvd";
        protected final static String RCPT = "rcpt";
        protected final static String R_SP = "r_sp";
        protected final static String R_DL = "r_dl";
        protected final static String R_NL = "r_nl";
        protected final static String SPAM = "spam";
        protected final static Pattern RE_SOURCE_IP = Pattern.compile("source ip = (\\S+)");

        protected final Map<String, Long> queueIdMids;
        protected final Map<String, Map<String, List<String>>> queueIdHeaders;
        protected final Map<List<String>, String> queueIdRcptUids;
        protected final Map<Long, List<String>> midQueueIds;
        protected final ImmutableComplaintsConfig complaintsConfig;
        protected final AbstractUserActionContext context;

        public SologgerCallback(
            final AbstractUserActionContext context,
            final Map<List<String>, Long> queueIdMids,
            final FutureCallback<Void> callback)
        {
            super(callback);
            this.context = context;
            this.queueIdMids = new HashMap<>();
            complaintsConfig = context.iexProxy().config().complaintsConfig();
            queueIdHeaders = new HashMap<>();
            midQueueIds = new HashMap<>();
            queueIdRcptUids = new HashMap<>();
            if (queueIdMids != null && queueIdMids.size() > 0) {
                for (Map.Entry<List<String>, Long> msgQueueIdMids : queueIdMids.entrySet()) {
                    midQueueIds.put(msgQueueIdMids.getValue(), msgQueueIdMids.getKey());
                    queueIdRcptUids.put(msgQueueIdMids.getKey(), null);
                    for (String msgQueueId : msgQueueIdMids.getKey()) {
                        this.queueIdMids.put(msgQueueId, msgQueueIdMids.getValue());
                        queueIdHeaders.put(msgQueueId, null);
                    }
                }
            }
            if (queueIdHeaders.size() == 0) {
                context.session().logger().info("SologgerCallback: queueIds is empty!");
            }
        }

        @Override
        @SuppressWarnings("StringSplitter")
        public void completed(final String result) {
            if (result == null || result.trim().isEmpty()) {
                //context.session().logger().info("SologgerCallback: result=null for rcpt_uid=" + context.uid()
                //    + ", queueIds=" + queueIdHeaders);
                context.session().logger().info("SologgerCallback: result=null");
                complaintsConfig.complaintsError(ComplaintProcessingError.DlvLogAbsent, context.action());
                callback.completed(null);
                return;
            }
            for (String row : result.split("\n")) {
                //context.session().logger().info("SologgerCallback: row=" + row);
                if (row.trim().isEmpty()) {
                    continue;
                }
                try {
                    Map<String, List<String>> headers = new HashMap<>();
                    try {
                        JsonList list = TypesafeValueContentHandler.parse(row).asList();
                        for (JsonObject item : list) {
                            JsonList headerItem = item.asList();
                            headers.computeIfAbsent(headerItem.get(0).asString(), x -> new ArrayList<>())
                                .add(headerItem.get(1).asString());
                        }
                    } catch (JsonException e) {
                        context.session().logger().log(
                            Level.SEVERE,
                            "SologgerCallback failed to parse response row: " + row + ", rcpt_uid=" + context.uid()
                                + ", queueIds=" + queueIdHeaders,
                            e);
                    }
                    Matcher m;
                    String uid = null;
                    try {
                        if (headers.containsKey(ORCP)) {
                            for (String orcp : headers.get(ORCP).get(0).split(";")) {
                                if ((m = ORCP_UID.matcher(orcp)).find() && context.uid().equals(m.group(1))) {
                                    uid = context.uid();
                                    break;
                                }
                            }
                        }
                        if (uid == null && headers.containsKey(RCPT)) {
                            for (String rcpt : headers.get(RCPT)) {
                                if ((m = RCPT_UID.matcher(rcpt)).find() && context.uid().equals(m.group(1))) {
                                    uid = context.uid();
                                    break;
                                }
                            }
                        }
                        if (uid == null) {
                            complaintsConfig.complaintsError(
                                ComplaintProcessingError.ParsingRcptUidFailed,
                                context.action());
                            context.session().logger().info("SologgerCallback unable to find rcpt UID for dlvLog="
                                + row);
                        }
                    } catch (RuntimeException e) {
                        context.session().logger().log(
                            Level.SEVERE,
                            "SologgerCallback failed to parse uid for rcpt_uid=" + context.uid()
                                + ", queueIds=" + queueIdHeaders.keySet(),
                            e);
                    }
                    if (headers.containsKey(X_YANDEX_QUEUEID)
                            && queueIdHeaders.containsKey(headers.get(X_YANDEX_QUEUEID).get(0)))
                    {
                        String queueId = headers.get(X_YANDEX_QUEUEID).get(0);
                        queueIdHeaders.put(queueId, headers);
                        if (uid != null && queueIdRcptUids.get(midQueueIds.get(queueIdMids.get(queueId))) == null) {
                            queueIdRcptUids.put(midQueueIds.get(queueIdMids.get(queueId)), uid);
                            processDeliveryLog(queueId, headers);
                        }
                    } else {
                        complaintsConfig.complaintsError(
                            ComplaintProcessingError.ParsingQueueIdFailed,
                            context.action());
                        context.session().logger().info("SologgerCallback: unable to find queueid for uid="
                            + context.uid() + " (parsed uid=" + uid + ") among queueId=" + queueIdHeaders.keySet()
                            + " (" + headers.get(X_YANDEX_QUEUEID) + "), dlvLog=" + row);
                    }
                } catch (RuntimeException e) {
                    complaintsConfig.complaintsError(ComplaintProcessingError.ParsingDlvLogFailed, context.action());
                    context.session().logger().log(
                        Level.SEVERE,
                        "SologgerCallback failed to parse delivery-log: " + row + ", queueId="
                            + queueIdHeaders.keySet(),
                        e);
                }
            }
            try {
                for (Map.Entry<List<String>, String> queueIdRcptUid : queueIdRcptUids.entrySet()) {
                    if (queueIdRcptUid.getValue() == null && queueIdRcptUid.getKey().size() > 0) {
                        String queueId = queueIdRcptUid.getKey().get(0);
                        if (queueIdHeaders.containsKey(queueId) && queueIdHeaders.get(queueId) != null) {
                            processDeliveryLog(queueId, queueIdHeaders.get(queueId));
                        }
                    }
                }
            } catch (RuntimeException e) {
                context.session().logger().log(
                    Level.SEVERE,
                    "SologgerCallback failed to process delivery-log: " + e + ", queueIdRcptUids=" + queueIdRcptUids
                        + ", queueIdHeaders=" + queueIdHeaders.keySet(),
                    e);
            }
            callback.completed(null);
        }

        protected void processDeliveryLog(final String queueId, final Map<String, List<String>> headers) {
            MailMessageContext msgContext = context.messages().get(queueIdMids.get(queueId)).context();
            if (msgContext.rules().size() < 1) {
                parseRulesInfo(msgContext, headers, R_SP);
                parseRulesInfo(msgContext, headers, R_DL);
                parseRulesInfo(msgContext, headers, R_NL);
            }
            if (msgContext.rules().size() < 1) {
                complaintsConfig.complaintsError(ComplaintProcessingError.ParsingRulesFalied, context.action());
            }
            if ((msgContext.mailBackend() == null || msgContext.mailBackend().isEmpty())
                    && headers.containsKey(X_YANDEX_SO_FRONT)
                    && !headers.get(X_YANDEX_SO_FRONT).get(0).isEmpty())
            {
                msgContext.setMailBackend(headers.get(X_YANDEX_SO_FRONT).get(0));
            } else {
                complaintsConfig.complaintsError(ComplaintProcessingError.ParsingMailBackendFailed, context.action());
            }
            if ((msgContext.geoZone() == null || msgContext.geoZone().isEmpty())
                    && headers.containsKey(IY_GEOZONE) && !headers.get(IY_GEOZONE).get(0).isEmpty())
            {
                msgContext.setGeoZone(headers.get(IY_GEOZONE).get(0));
            } else {
                complaintsConfig.complaintsError(ComplaintProcessingError.ParsingGeoZoneFailed, context.action());
            }
            if ((msgContext.soRes() == null || msgContext.soRes() == MailMessageContext.SoResolution.SKIP)
                    && headers.containsKey(SPAM) && !headers.get(SPAM).get(0).isEmpty()) {
                if (headers.get(SPAM).get(0).equals("yes")) {
                    msgContext.setSoRes(MailMessageContext.SoResolution.SPAM);
                } else if (headers.get(SPAM).get(0).equals("no")) {
                    msgContext.setSoRes(MailMessageContext.SoResolution.HAM);
                } else {
                    complaintsConfig.complaintsError(ComplaintProcessingError.ParsingSoResFailed, context.action());
                }
            } else {
                complaintsConfig.complaintsError(ComplaintProcessingError.ParsingSoResFailed, context.action());
            }
            if ((msgContext.senderIp() == null || msgContext.senderIp().isEmpty()) && headers.containsKey(RCVD)) {
                for (String item : headers.get(RCVD)) {
                    if (item.startsWith("source ip = ")) {
                        Matcher m = RE_SOURCE_IP.matcher(item);
                        if (m.find()) {
                            msgContext.setSenderIp(m.group(1));
                        }
                        break;
                    }
                }
            }
            if (msgContext.senderIp() == null || msgContext.senderIp().isEmpty()) {
                complaintsConfig.complaintsError(ComplaintProcessingError.ParsingSenderIpFailed, context.action());
            }
        }

        private void parseRulesInfo(
            final MailMessageContext msgContext,
            final Map<String, List<String>> headers,
            final String headerName)
        {
            if (headers.containsKey(headerName)) {
                String rule;
                for (String ruleInfo : headers.get(headerName).get(0).split("[,;]\\s*")) {
                    if (!ruleInfo.isEmpty()) {
                        if (ruleInfo.contains(" ")) {
                            rule = ruleInfo.substring(0, ruleInfo.indexOf(" "));
                        } else {
                            rule = ruleInfo;
                        }
                        testRules(rule);
                        msgContext.rules().add(rule);
                    }
                }
            }
        }

        public void testRules(final String rule) {
        }

        @Override
        public void cancelled() {
            context.session().logger().warning(
                "SologgerCallback request cancelled: " + context.session().listener().details());
            callback.completed(null);
        }

        @Override
        public void failed(final Exception e) {
            context.session().logger().log(Level.WARNING, "SologgerCallback failed: "
                + context.session().listener().details() + " because of exception: " + e.toString(), e);
            complaintsConfig.complaintsError(ComplaintProcessingError.DlvLogAbsent, context.action());
            callback.completed(null);
        }
    }

    public static void requestUserLogin(final AbstractUserActionContext context, final FutureCallback<Void> callback) {
        if (context.uid() == null || context.uid().isEmpty()) {
            callback.completed(null);
            return;
        }
        requestBlackbox(context, callback, BlackBoxRequestType.COMPLAINANT, context.uid());
    }

    public static void requestUserUid(
        final AbstractUserActionContext context,
        final FutureCallback<Void> callback,
        final BlackBoxRequestType requestType,
        final String email)
    {
        if (email == null || email.isEmpty()) {
            callback.completed(null);
            return;
        }
        requestBlackbox(context, callback, requestType, email);
    }

    protected static void requestBlackbox(
        final AbstractUserActionContext context,
        final FutureCallback<Void> callback,
        final BlackBoxRequestType requestType,
        final Object value)
    {
        BlackboxClient client;
        BlackboxUserIdType bbRequestType = requestType == BlackBoxRequestType.COMPLAINANT
            ? BlackboxUserIdType.UID : BlackboxUserIdType.LOGIN;
        BlackboxUserinfoRequest request = new BlackboxUserinfoRequest(bbRequestType, value);
        if (context.corp()) {
            client = context.iexProxy().corpBlackboxClient();
            request.addHeader(YandexHeaders.X_YA_SERVICE_TICKET, context.iexProxy().corpBlackboxTvm2Ticket());
        } else {
            client = context.iexProxy().blackboxDirectClient();
            request.addHeader(YandexHeaders.X_YA_SERVICE_TICKET, context.iexProxy().blackboxTvm2Ticket());
        }
        client = client.adjust(context.session().context());
        client.userinfo(
            request.sid("smtp").dbfields(BlackboxDbfield.SUID),
            context.session().listener().createContextGeneratorFor(client),
            TraceFutureCallback.wrap(
                new SingleUserinfoBlackboxCallback(
                    TraceFutureCallback.wrap(
                        new BlackboxCallback(context, requestType, value, callback),
                        context,
                        "BB for UserActionProcess")),
                context,
                "SingleUserBB for UserActionProcess"));
    }

    protected static class BlackboxCallback extends AbstractFilterFutureCallback<BlackboxUserinfo, Void>
    {
        private final AbstractUserActionContext context;
        private final BlackBoxRequestType requestType;
        private final Object value;

        BlackboxCallback(
            final AbstractUserActionContext context,
            final BlackBoxRequestType requestType,
            final Object value,
            final FutureCallback<Void> callback)
        {
            super(callback);
            this.context = context;
            this.requestType = requestType;
            this.value = value;
        }

        @Override
        public void completed(final BlackboxUserinfo userinfo) {
            try {
                if (requestType == BlackBoxRequestType.COMPLAINANT) {
                    context.setLogin(userinfo.login());
                    if (context.suid() == null && userinfo.dbfields().containsKey(BlackboxDbfield.SUID)) {
                        context.setSuid(userinfo.dbfields().get(BlackboxDbfield.SUID));
                    }
                    context.setKarma((long) userinfo.karma());
                } else if (requestType == BlackBoxRequestType.SENDER) {
                    context.senderUids().put((String) value, userinfo.uid());
                } else if (requestType == BlackBoxRequestType.RECIPIENT) {
                    context.recipients().put((String) value, userinfo.uid());
                }
            } catch (Exception e) {
                context.session().logger().log(
                    Level.SEVERE,
                    "UserActionProcessCallback.BlackboxCallback failed: " + e,
                    e);
            }
            callback.completed(null);
        }

        @Override
        public void cancelled() {
            callback.completed(null);
        }

        @Override
        public void failed(final Exception e) {
            if (requestType == BlackBoxRequestType.COMPLAINANT) {
                context.iexProxy().config().complaintsConfig()
                    .complaintsError(ComplaintProcessingError.RcptLoginUnknown, context.action());
            }
            if (context.login() != null && context.login().contains("\n")) {
                context.setLogin(String.join(",", context.login().split("\n")));
            }
            callback.completed(null);
        }
    }
}
