package ru.yandex.iex.proxy.eshophandler;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;

import ru.yandex.client.producer.ProducerClient;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.iex.proxy.AbstractEntityHandler;
import ru.yandex.iex.proxy.IexProxy;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.mail.search.MailSearchDefaults;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.proxy.SearchResultConsumerFactory;
import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;
import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;

public class EshopHandler extends AbstractEntityHandler<EshopContext> {
    private static final String FACT_DOMAIN = "fact_domain";
    private static final String ESHOP = "eshop";
    private static final String LOG_PREFIX = "EshopHandler: ";
    private static final String ORDER_DELIMITER = "\\|";
    private static final String MARKET_ANSWER = "market_answer";
    private static final String REGEX_QUOTE = "[\"\']";
    private static final String MARKET_CATEGORY_FOUND = "market_category_found";
    private static final String MARKET_CATEGORY_ID = "market_category_id";
    private static final String MARKET_CATEGORY_NAME = "market_category_name";
    private static final String MARKET_CATEGORY_FULLNAME =
        "market_category_fullname";
    private static final String MARKET_MODEL_ID = "market_model_id";
    private static final String MARKET_MODEL_NAME = "market_model_name";
    private static final String ORDER_ITEM = "order_item";
    private static final String FULL_NAME = "fullName";
    private static final String NAME = "name";
    private static final String ID = "id";
    private static final String IEX = "iex";
    private static final String YUIDS = "yuids";
    private static final String CATEGORY = "category";

    public EshopHandler(final IexProxy iexProxy) {
        super(iexProxy, ESHOP);
    }

    @Override
    protected EshopContext createContext(
        final IexProxy iexProxy,
        final ProxySession session,
        final Map<?, ?> json)
        throws HttpException, JsonUnexpectedTokenException
    {
        return new EshopContext(iexProxy, session, json);
    }

    @Override
    protected void handle(final EshopContext context)
        throws JsonUnexpectedTokenException, BadRequestException
    {
        Object order = context.getResult().get("order");
        Object tableColumns = context.getResult().get("table_columns");

        context.session().logger().info(LOG_PREFIX + "order = " + order);
        context.session().logger().info(LOG_PREFIX + "table_columns = "
            + tableColumns);

        final MultiFutureCallback<ResultForEshop> multiCallback =
            new MultiFutureCallback<>(new EshopFinishMultiCallback(context));

        List<RequestAndOrigin> requests = new ArrayList<RequestAndOrigin>();
        addOrderToRequests(order, requests);
        addTableColumnsItemsToRequests(tableColumns, requests);

        long prefix = context.prefix();
        User user = new User(IEX, new LongPrefix(prefix));
        String yuidQueryString = makeYuidQueryString(prefix);
        YuidResultCallback yuidCallback =
            addYuidResultCallback(context, multiCallback, yuidQueryString);
        BasicAsyncRequestProducerGenerator yuidGenerator =
            new BasicAsyncRequestProducerGenerator(yuidQueryString);
        PlainUniversalSearchProxyRequestContext requestContext =
            getPlainUniversalSearchProxyRequestContent(context, user);

        context.session().logger().info(LOG_PREFIX + "order_number = "
            + context.getOrderNumber());
        // add 'is_new_order" only if 'order_number' is presents and not empty
        NewOrderCallback newOrderCallback = null;
        if (context.getOrderNumber() != null
            && !context.getOrderNumber().isEmpty())
        {
            newOrderCallback = addNewOrderCallback(context, multiCallback);
        }

        if (!requests.isEmpty()) {
            final List<MarketResultCallback> subMarketCallbacks =
                addMarketSubCallbacks(requests, multiCallback);

            final BkResultCallback subBkCallback =
                addBkSubCallback(
                    requests,
                    context,
                    multiCallback);

            multiCallback.done();

            executeMarketRequests(requests, subMarketCallbacks, context);
            executeBkRequest(requests, subBkCallback, context);
        } else {
            context.session().logger().
                info("EshopHandler: order request list is empty");
            multiCallback.done();
        }

        context.iexProxy().parallelRequest(
            context.session(),
            requestContext,
            yuidGenerator,
            SearchResultConsumerFactory.OK,
            context.session().listener().createContextGeneratorFor(
                requestContext.client()),
            yuidCallback);

        if (newOrderCallback != null) {
            executeNewOrderRequest(newOrderCallback, context);
        }
    }

    private void addOrderToRequests(
        final Object order,
        final List<RequestAndOrigin> requests)
    {
        if (order instanceof String) {
            for (String item : ((String) order).split(ORDER_DELIMITER)) {
                requests.add(
                    new RequestAndOrigin(trimString(item), Origin.ORDER));
            }
        }
    }

    private void addTableColumnsItemsToRequests(
        final Object tableColumns,
        final List<RequestAndOrigin> requests)
    {
        if (tableColumns instanceof List) {
            List<?> items = (List<?>) tableColumns;
            for (Object item: items) {
                String tableColumnItem = trimString((String) item);
                if (!tableColumnItem.isEmpty()) {
                    requests.add(
                        new RequestAndOrigin(
                            tableColumnItem,
                            Origin.TABLE_COLUMNS));
                }
            }
        } else if (tableColumns instanceof String) {
            String tableColumnItem = trimString((String) tableColumns);
            if (!tableColumnItem.isEmpty()) {
                requests.add(
                    new RequestAndOrigin(
                        tableColumnItem,
                        Origin.TABLE_COLUMNS));
            }
        }
    }

    private String trimString(final String str) {
        return str.replaceAll("\u00A0", " ").
            replaceAll(REGEX_QUOTE, "").
            trim();
    }

    private List<MarketResultCallback> addMarketSubCallbacks(
        final List<RequestAndOrigin> requests,
        final MultiFutureCallback<ResultForEshop> multiCallback)
    {
        final List<MarketResultCallback> subMarketCallbacks =
            new ArrayList<>(requests.size());
        for (final RequestAndOrigin requestAndOrigin : requests) {
            subMarketCallbacks.add(
                new MarketResultCallback(
                    multiCallback.newCallback(),
                    requestAndOrigin
                ));
        }
        return subMarketCallbacks;
    }

    private BkResultCallback addBkSubCallback(
        final List<RequestAndOrigin> requests,
        final EshopContext context,
        final MultiFutureCallback<ResultForEshop> multiCallback)
    {
        HttpHost bkHost =
            context.iexProxy().config().bkConfig().host();

        StringBuilderWriter sbw =
            new StringBuilderWriter(new StringBuilder(""));
        try (JsonWriter writer = new JsonWriter(sbw)) {
            writer.startObject();
            writer.key("requests");
            writer.startArray();
            for (RequestAndOrigin requestAndOrigin: requests) {
                writer.value(requestAndOrigin.getRequest());
            }
            writer.endArray();
            writer.endObject();

            BasicAsyncRequestProducerGenerator generator =
                new BasicAsyncRequestProducerGenerator("" + '/', sbw.toString());
            generator.addHeader(
                HttpHeaders.CONTENT_TYPE,
                ContentType.APPLICATION_JSON.toString());
            generator.addHeader(
                HttpHeaders.ACCEPT,
                ContentType.APPLICATION_JSON.toString());
            return new BkResultCallback(
                multiCallback.newCallback(),
                generator,
                bkHost);
        } catch (IOException e) {
            context.session().logger().
                info("BK: exception while constructing BK request");
            return null;
        }
    }

    private NewOrderCallback addNewOrderCallback(
        final EshopContext context,
        final MultiFutureCallback<ResultForEshop> multiCallback)
    {
        return new NewOrderCallback(
            multiCallback.newCallback(),
            context.session());
    }

    private YuidResultCallback addYuidResultCallback(
        final EshopContext context,
        final MultiFutureCallback<ResultForEshop> multiCallback,
        final String yuidQueryString)
    {
        return new YuidResultCallback(
            multiCallback.newCallback(),
            context.session(),
            yuidQueryString);
    }

    private PlainUniversalSearchProxyRequestContext
    getPlainUniversalSearchProxyRequestContent(
        final EshopContext context,
        final User user)
        throws JsonUnexpectedTokenException
    {
        return new PlainUniversalSearchProxyRequestContext(
            user,
            null,
            true,
            context.iexProxy().searchClient(),
            context.session().logger());
    }

    private String makeYuidQueryString(final long prefix)
        throws BadRequestException
    {
        QueryConstructor yuidQuery = new QueryConstructor("/search?");
        yuidQuery.append("prefix", prefix);
        yuidQuery.append("service", MailSearchDefaults.BP_CHANGE_LOG);
        yuidQuery.append("text", "url:user_gbl_" + prefix);
        yuidQuery.append("get", "*");
        return yuidQuery.toString();
    }

    private void executeMarketRequests(
        final List<RequestAndOrigin> requests,
        final List<MarketResultCallback> subMarketCallbacks,
        final EshopContext context)
    {
        AsyncClient marketClient =
            context.iexProxy().marketClient().adjust(
                context.session().context());
        HttpHost host =
            context.iexProxy().config().marketConfig().host();
        int i = 0;
        for (final RequestAndOrigin requestAndOrigin : requests) {
            QueryConstructor marketCgiQueryConstructor =
                new QueryConstructor("/v2.1/models/match?");
            try {
                marketCgiQueryConstructor.append(
                    NAME,
                    requestAndOrigin.getRequest());
            } catch (BadRequestException e) {
                context.session().logger().
                    info("EshopHandler: BadRequestException to market");
            }
            String uri = marketCgiQueryConstructor.toString();
            BasicAsyncRequestProducerGenerator generator =
                new BasicAsyncRequestProducerGenerator(uri);
            generator.addHeader(
                "Authorization",
                "elUed5AolqqKfVYlb0PUSkwLTnb6oz");
            marketClient.execute(
                host,
                generator,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                context.session().listener()
                    .createContextGeneratorFor(marketClient),
                new ErrorSuppressingFutureCallback<>(
                    subMarketCallbacks.get(i++),
                    y -> RequestErrorType.HTTP,
                    JsonMap.EMPTY)
            );
        }
    }

    private void executeBkRequest(
        final List<RequestAndOrigin> requests,
        final BkResultCallback subBkCallback,
        final EshopContext context)
    {
        AsyncClient bkClient =
            context.iexProxy().bkClient().adjust(
                context.session().context());
        bkClient.execute(
            subBkCallback.host(),
            subBkCallback.generator(),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.session().listener().
                createContextGeneratorFor(bkClient),
            new ErrorSuppressingFutureCallback<>(
                subBkCallback,
                y -> RequestErrorType.HTTP,
                JsonMap.EMPTY)
        );
    }

    private void executeNewOrderRequest(
        final NewOrderCallback newOrderCallback,
        final EshopContext context)
    {
        context.session().logger().
            info("check whether the order is new or updated");
        Prefix prefix = new LongPrefix(context.prefix());
        String serviceName = context.iexProxy().config().
            factsIndexingQueueName();
        User user = new User(serviceName, prefix);
        ProducerClient producerClient = context.iexProxy().producerClient();
        producerClient.execute(
            user,
            context.session().listener().
                createContextGeneratorFor(producerClient),
            new EshopHostsCallback(
                context,
                newOrderCallback));
    }

    private abstract static class ResultForEshop {
        protected final RequestAndOrigin request;

        ResultForEshop(final RequestAndOrigin request) {
            this.request = request;
        }

        public RequestAndOrigin getRequestAndOrigin() {
            return request;
        }

        public abstract ResultForEshopType type();
    }

    private static class MarketResult extends ResultForEshop {
        private final JsonMap result;

        MarketResult(final JsonMap result, final RequestAndOrigin request) {
            super(request);
            this.result = result;
        }

        public JsonMap result() {
            return result;
        }

        public ResultForEshopType type() {
            return ResultForEshopType.MARKET;
        }
    }

    private static class BkResult extends ResultForEshop {
        private final JsonMap result;

        BkResult(final JsonMap result, final RequestAndOrigin request) {
            super(request);
            this.result = result;
        }

        public JsonMap result() {
            return result;
        }

        public ResultForEshopType type() {
            return ResultForEshopType.BK;
        }
    }

    private static class YuidResult extends ResultForEshop {
        private final SearchResult result;

        YuidResult(final SearchResult result, final RequestAndOrigin request) {
            super(request);
            this.result = result;
        }

        public SearchResult result() {
            return result;
        }

        public ResultForEshopType type() {
            return ResultForEshopType.YUID;
        }
    }

    private static class NewOrderResult extends ResultForEshop {
        private final SearchResult result;

        NewOrderResult(
            final SearchResult result,
            final RequestAndOrigin request)
        {
            super(request);
            this.result = result;
        }

        public SearchResult result() {
            return result;
        }

        public ResultForEshopType type() {
            return ResultForEshopType.IS_NEW_ORDER;
        }
    }

    private static class EshopFinishMultiCallback implements
        FutureCallback<List<ResultForEshop>>
    {
        private static final String LOG_MULTI_PREFIX =
            "EshopHandler.EshopFinishMultiCallback: ";
        private final EshopContext context;

        EshopFinishMultiCallback(final EshopContext context) {
            this.context = context;
        }

        @Override
        public void completed(final
                              List<ResultForEshop> listResults)
        {
            List<MarketResult> marketResults = new ArrayList<>();
            BkResult bkResult = null;
            YuidResult yuidResult = null;
            NewOrderResult isNewOrderResult = null;
            String isNewOrder = "new";
            for (ResultForEshop res : listResults) {
                if (res.type() == ResultForEshopType.BK) {
                    bkResult = ((BkResult) res);
                }
                if (res.type() == ResultForEshopType.MARKET) {
                    marketResults.add((MarketResult) res);
                }
                if (res.type() == ResultForEshopType.YUID) {
                    yuidResult = (YuidResult) res;
                }
                if (res.type() == ResultForEshopType.IS_NEW_ORDER) {
                    context.session().logger().
                        info(LOG_MULTI_PREFIX + "res type is IS_NEW_ORDER");
                    isNewOrderResult = (NewOrderResult) res;
                }
            }

            List<ItemInfoWithOrigin> orderItems =
                marketOrderItems(marketResults, context);

            Map<String, String> orderItemsToBkIds = new HashMap<>();
            if (bkResult != null && bkResult.result() instanceof JsonMap) {
                JsonMap bkResultMap = ((JsonMap) bkResult.result());
                try {
                    for (Map.Entry<String, JsonObject> entry: bkResultMap.entrySet()) {
                        orderItemsToBkIds.put(
                            entry.getKey(),
                            entry.getValue().asString());
                    }
                } catch (JsonBadCastException e) {
                    context.session().logger().info("JsonBadCastException "
                        + "in parsing bkResult");
                }
            }

            if (bkResult != null) {
                context.session().logger().info(LOG_MULTI_PREFIX + "BkResult = "
                    + JsonType.NORMAL.toString(bkResult.result()));
            }

            for (ItemInfoWithOrigin itemInfo : orderItems) {
                String itemName =
                    String.valueOf(itemInfo.getData().get(ORDER_ITEM));
                String bkId = orderItemsToBkIds.get(itemName);
                if (bkId != null && bkId.length() > 0) {
                    itemInfo.setData("bk_category_id", bkId);
                }
            }

            if (yuidResult != null && yuidResult.result() != null) {
                List<SearchDocument> searchDocs =
                    yuidResult.result().hitsArray();
                if (searchDocs.size() != 1) {
                    context.session().logger().
                        info(LOG_MULTI_PREFIX + "search docs size != 1");
                } else {
                    SearchDocument searchDoc = searchDocs.get(0);
                    if (searchDoc.attrs().containsKey(YUIDS)) {
                        String yuids = searchDoc.attrs().get(YUIDS).
                            replaceAll("\n", ",");
                        context.session().logger().
                            info(LOG_MULTI_PREFIX + "yuids added");
                        context.addYuids(yuids);
                    } else {
                        context.session().logger().
                            info(LOG_MULTI_PREFIX
                                + "hitsArray doesn't contain yuids key");
                    }
                }
            } else {
                context.session().logger().
                    info(LOG_MULTI_PREFIX + "yuid is null");
            }

            List<Map<String, Object>> orderItemsToResponse = new ArrayList<>();
            for (ItemInfoWithOrigin itemInfo : orderItems) {
                if (itemInfo.getData().size() > 1
                    || itemInfo.getOrigin().equals(Origin.ORDER))
                {
                    orderItemsToResponse.add(itemInfo.getData());
                }
            }
            if (!orderItemsToResponse.isEmpty()) {
                context.session().logger().
                    info(LOG_MULTI_PREFIX + "adding market and bk responses");
                context.addMarketInfo(orderItemsToResponse);
            }

            if (isNewOrderResult != null && isNewOrderResult.result() != null) {
                for (SearchDocument searchDoc
                    : isNewOrderResult.result().hitsArray())
                {
                    if (searchDoc.attrs().containsKey(FACT_DOMAIN)
                        && searchDoc.attrs().get(FACT_DOMAIN).
                        equals(context.getDomain()))
                    {
                        context.session().logger().
                            info(LOG_MULTI_PREFIX + "is updated order");
                        isNewOrder = "updated";
                    }
                }
            }

            context.addIsNewOrder(isNewOrder);

            context.session().logger().
                info(LOG_MULTI_PREFIX + "EshopHandler response submitted");
            context.response();
        }

        @Override
        public void cancelled() {
            context.session().logger().
                info(LOG_MULTI_PREFIX + "cancelled");
            context.response();
        }

        @Override
        public void failed(final Exception e) {
            context.session().logger().
                info(LOG_MULTI_PREFIX + "failed");
            context.response();
        }
    }

    public static List<ItemInfoWithOrigin> marketOrderItems(
        final List<MarketResult> results,
        final EshopContext context)
    {
        List<ItemInfoWithOrigin> orderItems = new ArrayList<>();
        for (final MarketResult x : results) {
            Map<String, Object> itemInfo = new HashMap<>();
            itemInfo.put(ORDER_ITEM, x.getRequestAndOrigin().getRequest());
            JsonMap result = x.result();
            if (result == null) {
                context.session().logger().
                    info("EshopHandler: null result from market for "
                        + x.getRequestAndOrigin());
                orderItems.add(
                    new ItemInfoWithOrigin(
                        itemInfo,
                        x.getRequestAndOrigin().getOrigin()));
                continue;
            }
            try {
                JsonObject modelsObj = result.get("models");
                List<JsonObject> models =
                    (modelsObj == null || modelsObj.type() == JsonObject.Type.NULL) ? null : modelsObj.asList();
                if (models == null || models.isEmpty()) {
                    context.session().logger().
                        info("EshopHandler: models is null or empty for "
                            + x.getRequestAndOrigin());
                    orderItems.add(
                        new ItemInfoWithOrigin(
                            itemInfo,
                            x.getRequestAndOrigin().getOrigin()));
                    continue;
                }
                itemInfo.put(
                    "market_models_found",
                    models.size());

                if (models.size() == 1) {
                    context.session().logger().
                        info("EshopHandler: models size is 1 for "
                            + x.getRequestAndOrigin().getRequest());
                    JsonMap model = models.get(0).asMap();
                    itemInfo.put(
                        MARKET_MODEL_NAME,
                        model.get(NAME));
                    itemInfo.put(
                        MARKET_MODEL_ID,
                        model.get(ID));
                    JsonMap category = model.get(CATEGORY).asMap();
                    itemInfo.put(
                        MARKET_CATEGORY_NAME,
                        category.get(NAME));
                    itemInfo.put(
                        MARKET_CATEGORY_ID,
                        category.get(ID));
                    itemInfo.put(
                        MARKET_CATEGORY_FULLNAME,
                        category.get(FULL_NAME));
                    itemInfo.put(MARKET_CATEGORY_FOUND, 1);
                }
                if (models.size() > 1) {
                    context.session().logger().
                        info("EshopHandler: models size > 1 for "
                            + x.getRequestAndOrigin().getRequest());
                    Set<JsonObject> categories = new HashSet<JsonObject>();
                    for (JsonObject model : models) {
                        categories.add(model.get(CATEGORY).
                            asMap().get(ID));
                    }
                    if (categories.size() == 1) {
                        context.session().logger().
                            info("EshopHandler: categories size is 1 for "
                                + x.getRequestAndOrigin().getRequest());
                        JsonMap category = models.get(0).asMap().
                            get(CATEGORY).asMap();
                        itemInfo.put(MARKET_CATEGORY_FOUND, 1);
                        itemInfo.put(
                            MARKET_CATEGORY_NAME,
                            category.get(NAME));
                        itemInfo.put(
                            MARKET_CATEGORY_ID,
                            category.get(ID));
                        itemInfo.put(
                            MARKET_CATEGORY_FULLNAME,
                            category.get(FULL_NAME));
                    }
                    if (categories.size() > 1) {
                        context.session().logger().
                            info("EshopHandler: categories size > 1 for "
                                + x.getRequestAndOrigin().getRequest());
                        itemInfo.put(MARKET_CATEGORY_FOUND, categories.size());
                    }
                }
                orderItems.add(new ItemInfoWithOrigin(itemInfo, x.getRequestAndOrigin().getOrigin()));
            } catch (JsonException e) {
                context.session().logger().log(
                    Level.SEVERE,
                    "EshopHandler: MarketFinishMultiCallbacks JsonException for "
                        + x.getRequestAndOrigin().getRequest(),
                    e);
                orderItems.add(new ItemInfoWithOrigin(itemInfo, x.getRequestAndOrigin().getOrigin()));
                continue;
            }
        }
        return orderItems;
    }

    private static class BkResultCallback implements FutureCallback<JsonObject> {
        private final FutureCallback<ResultForEshop> callback;
        private BasicAsyncRequestProducerGenerator generator;
        private HttpHost host;

        BkResultCallback(
            final FutureCallback<ResultForEshop> callback,
            final BasicAsyncRequestProducerGenerator generator,
            final HttpHost host)
        {
            this.generator = generator;
            this.callback = callback;
            this.host = host;
        }

        public BasicAsyncRequestProducerGenerator generator() {
            return generator;
        }

        public HttpHost host() {
            return host;
        }

        @Override
        public void cancelled() {
            callback.completed(new BkResult(null, null));
        }

        @Override
        public void failed(final Exception e) {
            callback.completed(new BkResult(null, null));
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                callback.completed(new BkResult(result.asMap(), null));
            } catch (JsonException e) {
                callback.completed(new BkResult(null, null));
            }
        }
    }

    private static class MarketResultCallback
        implements FutureCallback<JsonObject>
    {
        private final FutureCallback<ResultForEshop> callback;
        private final RequestAndOrigin request;

        MarketResultCallback(
            final FutureCallback<ResultForEshop> callback,
            final RequestAndOrigin request)
        {
            this.request = request;
            this.callback = callback;
        }

        @Override
        public void cancelled() {
            callback.completed(new MarketResult(null, request));
        }

        @Override
        public void failed(final Exception e) {
            callback.completed(new MarketResult(null, request));
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                callback.completed(new MarketResult(result.asMap(), request));
            } catch (JsonException e) {
                callback.completed(new MarketResult(null, request));
            }
        }
    }

    private static class YuidResultCallback
        extends AbstractProxySessionCallback<SearchResult>
    {
        private final FutureCallback<ResultForEshop> callback;
        private final String request;

        YuidResultCallback(
            final FutureCallback<ResultForEshop> callback,
            final ProxySession session,
            final String request)
        {
            super(session);
            this.request = request;
            this.callback = callback;
        }

        @Override
        public void cancelled() {
            callback.completed(
                new YuidResult(null, new RequestAndOrigin(request, null)));
        }

        @Override
        public void failed(final Exception e) {
            callback.completed(
                new YuidResult(null, new RequestAndOrigin(request, null)));
        }

        @Override
        public void completed(final SearchResult result) {
            callback.completed(
                new YuidResult(result, new RequestAndOrigin(request, null)));
        }
    }

    private static class NewOrderCallback
        extends AbstractProxySessionCallback<SearchResult>
    {
        private final FutureCallback<ResultForEshop> callback;

        NewOrderCallback(
            final FutureCallback<ResultForEshop> callback,
            final ProxySession session)
        {
            super(session);
            this.callback = callback;
        }

        @Override
        public void cancelled() {
            session.logger().info("EshopHandler: NewOrderCallback cancelled");
            callback.completed(
                new NewOrderResult(
                    null,
                    new RequestAndOrigin("", null)));
        }

        @Override
        public void failed(final Exception e) {
            session.logger().info("EshopHandler: NewOrderCallback failed");
            callback.completed(
                new NewOrderResult(
                    null,
                    new RequestAndOrigin("", null)));
        }

        @Override
        public void completed(final SearchResult result) {
            session.logger().info("EshopHandler: NewOrderCallback completed");
            callback.completed(
                new NewOrderResult(
                    result,
                    new RequestAndOrigin("", null)));
        }
    }

    private enum Origin {
        ORDER,
        TABLE_COLUMNS
    }

    private static class RequestAndOrigin {
        private String request;
        private Origin origin;

        RequestAndOrigin(final String request, final Origin origin) {
            this.request = request;
            this.origin = origin;
        }

        public String getRequest() {
            return request;
        }

        public Origin getOrigin() {
            return origin;
        }
    }

    private static class ItemInfoWithOrigin {
        private Map<String, Object> data;
        private Origin origin;

        ItemInfoWithOrigin(
            final Map<String, Object> data,
            final Origin origin)
        {
            this.data = data;
            this.origin = origin;
        }

        public Map<String, Object> getData() {
            return data;
        }

        public Origin getOrigin() {
            return origin;
        }

        public void setData(final String key, final Object value) {
            data.put(key, value);
        }
    }
}
