package ru.yandex.iex.proxy.eshophandler;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.iex.proxy.XJsonUtils;
import ru.yandex.iex.proxy.XRegexpUtils;
import ru.yandex.iex.proxy.XStrAlgo;
import ru.yandex.iex.proxy.XTools;
import ru.yandex.iex.proxy.strconsts.HConsts;
import ru.yandex.iex.proxy.xutils.PriceFormat;
import ru.yandex.iex.proxy.xutils.SanitizeUtils;
import ru.yandex.iex.proxy.xutils.TrimUtils;
import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;

public class EshopStructureMaker {
    public static final String HTTP = "http://";
    public static final String HTTPS = "https://";
    public static final String NUM_ORDER_MBO = "num_order";
    public static final String ESHOP = "eshop";
    public static final String ORDER_NUMBER_MICRO = "orderNumber";
    public static final String ORDER_NUMBER = "order_number";
    public static final String PRICE = "price";
    public static final String CURRENCY = "currency";
    public static final String ITEM_OFFERED = "itemOffered";
    public static final String URL = "url";
    public static final String URLS = "urls";
    public static final String STATUS_ORDER = "status_order";
    public static final String ORDER_STATUS_TXT = "order_status_txt";
    public static final String ORDER_NUMBER_URL_NUMBER
        = "order_number_url_number";
    public static final String CAB = "cab";
    public static final String ORDER_NUMBER_URL_URL = "order_number_url_url";
    public static final String REGEXP = "regexp";
    public static final String ORDER_STATUS_MICRO = "order_status";
    public static final String PRICE_CURRENCY = "priceCurrency";
    public static final String ORDER_URLS = "order_urls";
    public static final String ORDER = "order";
    public static final String PART_OF_ORDER = "partOfOrder";
    public static final String NAME = "name";
    public static final String PAYMENT_METHOD = "payment_method";
    public static final String XPATH_LIST_DELIMITER = "\\|";
    private static final Character[] TRIM_CHARS =
        new Character[] {'\"', '\\', '\''};
    private static final Set<Character> SET_TRIM_CHARS =
        new HashSet<Character>(Arrays.asList(TRIM_CHARS));
    private String subject = null;
    private HashMap<String, Object> result = new HashMap<>();
    private String orderNumber = null;

    public HashMap<String, Object> make(
        final Map<?, ?> json,
        final String subject)
        throws JsonUnexpectedTokenException
    {
        this.subject = subject;
        parsePatterns(json);
        parseMicroData(json);

        Map<String, List<String>> rgRes = parseRegexpEshopResult(json, result);
        addXpathEshopResult(json, result);
        postParseEshopAction(result, rgRes);
        TrimUtils.trimValueOfFlatMap(result);
        SanitizeUtils.sanitizeHtmlValueOfFlatMap(result);
        orderNumberNormalization(result);
        completePriceWithCurrency(result);
        return result;
    }

    private void parseMicroData(final Map<?, ?> json)
        throws JsonUnexpectedTokenException
    {
        String microFormat = HConsts.JSON;
        Object jsonMicro =
            XJsonUtils.getNodeByPathOrNull(
                json,
                HConsts.MICRO,
                HConsts.UNKNOWN);
        if (jsonMicro == null) {
            jsonMicro = XJsonUtils.getNodeByPathOrNull(
                json,
                HConsts.MICROHTML,
                HConsts.UNKNOWN);
        }
        if (jsonMicro instanceof List && !((List) jsonMicro).isEmpty()) {
            jsonMicro = ((List) jsonMicro).get(0);
            microFormat = HConsts.MICRODATA;
        }
        if (jsonMicro instanceof Map) {
            Map<?, ?> mc = ValueUtils.asMap(jsonMicro);
            result.put(HConsts.WIDGET_SUBTYPE, ESHOP);
            result.put(HConsts.ORIGIN, HConsts.MICRO);
            result.put(HConsts.MICRO_FORMAT, microFormat);
            parseOrderNumber(mc, result);
            parseOrderStatus(mc, result);
            parseOrderPrice(mc, result);
            parseUrl(mc, result);
            parseExpectedArrivalUntil(mc, result);
            parseTrackingNumber(mc, result);
            parseTrackingUrl(mc, result);
            parseCurrency(mc, result);
        }
    }

    private void parsePatterns(final Map<?, ?> json) {
        Object eshopJson = json.get(ESHOP);
        if (eshopJson != null) {
            XJsonUtils.flattenJson("", eshopJson, result);
            result.put(HConsts.WIDGET_SUBTYPE, ESHOP);
            result.put(HConsts.ORIGIN, "patterns");
            XJsonUtils.renameKey(result, "cost_total", PRICE);
            XJsonUtils.renameKey(result, NUM_ORDER_MBO, ORDER_NUMBER);
            result.putIfAbsent(CURRENCY, "");
            orderNumberNormalization(result);
        }
    }

    private void completePriceWithCurrency(final Map<String, Object> json) {
        String currency = (String) json.get(CURRENCY);
        Object price = json.get(PRICE);
        if (price != null && currency != null) {
            if (price instanceof String) {
                String txtPrice = (String) price;
                String formattedPrice;
                ArrayList<String> prices =
                    XRegexpUtils.getPrices(txtPrice);
                ArrayList<String> curr =
                    XRegexpUtils.getCurrencies(txtPrice);
                if (!prices.isEmpty()) {
                    formattedPrice = PriceFormat.format(prices.get(0));
                } else {
                    formattedPrice = PriceFormat.format(txtPrice);
                }
                if (!curr.isEmpty()) {
                    currency = curr.get(0);
                }
                double val = PriceFormat.getPriceValue(formattedPrice);
                if (val > 0.0 && val <= PriceFormat.MAX_ESHOP_PRICE) {
                    StringBuilder sb = new StringBuilder();
                    sb.append(formattedPrice.replaceAll("\\.00[0-9]*", ""));
                    if (!currency.isEmpty()) {
                        sb.append(' ');
                        sb.append(PriceFormat.normalizeCurrency(currency));
                    }
                    json.put(PRICE, new String(sb));
                } else {
                    json.remove(PRICE);
                }
            }
            json.remove(CURRENCY);
        }
    }

    private void orderNumberNormalization(final Map<String, Object> json) {
        Object orderNumberFromJson = json.get(ORDER_NUMBER);
        if (orderNumberFromJson != null) {
            String orderNumberCutLetters =
                orderNumberFromJson instanceof Number ? orderNumberFromJson.toString() : (String) orderNumberFromJson;
            orderNumberCutLetters =
                orderNumberCutLetters.replaceAll("[№#а-яА-Яa-zA-Z ]", "");
            json.put(ORDER_NUMBER, orderNumberCutLetters);
            orderNumber = orderNumberCutLetters;
        }
    }

    private Map<String, List<String>> parseRegexpEshopResult(
        final Map<?, ?> json,
        final HashMap<String, Object> output)
    {
        Object jsonMicro =
            XJsonUtils.getNodeByPathOrNullEless(
                json,
                ESHOP + "_regexp");
        Map<String, List<String>> eshopRegexp = new HashMap<>();
        if (jsonMicro instanceof Map) {
            Map<?, ?> mp = (Map<?, ?>) jsonMicro;
            Object orderUrl = mp.get("order_url");
            GrepOrderUrl orderu = new GrepOrderUrl();
            ArrayList<String> orderUrls = orderu.grepFromList(orderUrl);
            putNotEmptyObject(eshopRegexp, ORDER_URLS, orderUrls);
            Object cab = mp.get("cabinet");
            GrepEshopData cabinet = new GrepCabinet();
            ArrayList<String> cabinetHrefs = cabinet.grepFromList(cab);
            putNotEmptyObject(eshopRegexp, CAB, cabinetHrefs);
            Object ordernumonly = mp.get("order_number_only");
            GrepEshopData orderNumOnly = new GrepOrderNumber();
            ArrayList<String> orderNumbers =
                orderNumOnly.grepFromList(ordernumonly);
            putNotEmptyObject(eshopRegexp, ORDER_NUMBER, orderNumbers);
            Object ordernumandurl = mp.get("order_number_url");
            GrepOrderNumberWithUrls orderNumAndUrl
                = new GrepOrderNumberWithUrls();
            ArrayList<String> orderNumbersurl =
                orderNumAndUrl.grepFromList(ordernumandurl);
            ArrayList<String> urls = orderNumAndUrl.getUrl();
            putNotEmptyObject(
                eshopRegexp,
                ORDER_NUMBER_URL_NUMBER,
                orderNumbersurl);
            putNotEmptyObject(eshopRegexp, ORDER_NUMBER_URL_URL, urls);
            Object paymentMethod = mp.get(PAYMENT_METHOD);
            String value = getFirstNode(paymentMethod);
            if (value != null && output.get(PAYMENT_METHOD) == null) {
                output.put(PAYMENT_METHOD, value);
            }
            if (!eshopRegexp.isEmpty()) {
                output.put(REGEXP, eshopRegexp);
            }
        }
        return eshopRegexp;
    }

    private void addXpathEshopResult(
        final Map<?, ?> json,
        final HashMap<String, Object> result)
        throws JsonUnexpectedTokenException
    {
        Map<?, ?> eshopXpath = ValueUtils.asMapOrNull(json.get("eshop_xpath"));
        if (eshopXpath != null) {
            Map<?, ?> xpathXpath = ValueUtils.asMapOrNull(
                    eshopXpath.get("xpath_xpath"));
            if (xpathXpath != null) {
                for (Map.Entry<?, ?> entry: xpathXpath.entrySet()) {
                    String value = ValueUtils.asString(entry.getValue());
                    if (value.contains("|")) {
                        result.put(
                            ValueUtils.asString(entry.getKey()),
                            Arrays.asList(value.split(XPATH_LIST_DELIMITER)));
                    } else {
                        result.put(
                            ValueUtils.asString(entry.getKey()),
                            value);
                    }
                }
            }
        }
    }

    private void putNotEmptyObject(
        final Map<String, List<String>> to,
        final String key,
        final ArrayList<String> value)
    {
        if (!value.isEmpty()) {
            to.put(key, XTools.uniq(value));
        }
    }

    private String getFirstNode(final Object list) {
        if (list instanceof List) {
            List<?> l = (List) list;
            if (!l.isEmpty()) {
                return (String) l.get(0);
            }
        }
        return null;
    }

    abstract class GrepEshopData {
        public ArrayList<String> grepFromList(final Object o) {
            ArrayList<String> res = new ArrayList<>();
            if (o instanceof List) {
                for (Object x : (List) o) {
                    String grepres = grep(x);
                    if (grepres != null && !grepres.isEmpty()) {
                        res.add(XStrAlgo.trimmer(grepres, SET_TRIM_CHARS));
                    }
                }
            }
            return res;
        }

        protected abstract String grep(Object x);
    }

    class GrepCabinet extends GrepEshopData {
        protected String grep(final Object x) {
            String res = null;
            if (x instanceof String) {
                res = XRegexpUtils.getHrefUrl((String) x);
            }
            return res;
        }
    }

    class GrepOrderNumber extends GrepEshopData {
        protected String grep(final Object x) {
            String res = null;
            if (x instanceof String) {
                res = XRegexpUtils.getOrderNumberLastSeq((String) x);
            }
            return res;
        }
    }

    class GrepOrderNumberWithUrls extends GrepEshopData {
        private ArrayList<String> urls = new ArrayList<>();

        protected String grep(final Object x) {
            String res = null;
            if (x instanceof String) {
                String url = XRegexpUtils.getHrefUrl((String) x);
                if (!url.isEmpty()) {
                    urls.add(XStrAlgo.trimmer(url, SET_TRIM_CHARS));
                }
                res = XRegexpUtils.getOrderNumberAfterTag((String) x);
                if (res.isEmpty()) {
                    res = XRegexpUtils.getOrderNumberLastSeq((String) x);
                }
            }
            return res;
        }

        public ArrayList<String> getUrl() {
            return urls;
        }
    }

    class GrepOrderUrl extends GrepEshopData {
        protected String grep(final Object x) {
            String res = null;
            if (x instanceof String) {
                res = XRegexpUtils.getHrefUrl((String) x);
            }
            return res;
        }
    }

    private void postParseEshopAction(
        final HashMap<String, Object> r,
        final Map<String, List<String>> regexpRes)
    {
        XJsonUtils.renameKey(r, NUM_ORDER_MBO, ORDER_NUMBER);
        XJsonUtils.renameKey(r, STATUS_ORDER, ORDER_STATUS_TXT);
        changeStatusOrder(r);
        updateUrl(r);
        updateOrderNumberAndUrlWithRegexpres(r, regexpRes);
    }

    private void updateUrl(final Map<String, Object> r) {
        Object url = r.get(URL);
        if (url instanceof String) {
            String urlt = (String) url;
            String[] t = urlt.split("\\s+");
            putValidUrls(r, Arrays.asList(t));
        }
    }

    private void updateOrderNumberAndUrlWithRegexpres(
        final HashMap<String, Object> r,
        final Map<String, List<String>> regexpRes)
    {
        boolean keysUpdated = false;
        Object orNum = r.get(ORDER_NUMBER);
        if (orNum == null || orNum.equals("")) {
            Object value = null;
            String numberWithJunk =
                XRegexpUtils.getOrderNumberWithJunk(subject);
            if (!numberWithJunk.isEmpty()) {
                GrepEshopData orderNumOnly = new GrepOrderNumber();
                ArrayList<String> orderNumbers =
                    orderNumOnly.grepFromList(numberWithJunk);
                if (!orderNumbers.isEmpty()) {
                    value = orderNumbers.get(0);
                }
            }
            if (value == null || value.equals("")) {
                List<String> lregNumber = regexpRes.get(ORDER_NUMBER);
                if (lregNumber == null) {
                    lregNumber = regexpRes.get(ORDER_NUMBER_URL_NUMBER);
                }
                if (lregNumber != null && !lregNumber.isEmpty()) {
                    value = lregNumber.get(0);
                    keysUpdated = true;
                }
            }
            if (value != null) {
                r.put(ORDER_NUMBER, value);
            }
        }

        if (!r.containsKey(URL) && !r.containsKey(URLS)) {
            List<String> lregUrl = regexpRes.get(ORDER_URLS);
            if (lregUrl == null) {
                lregUrl = regexpRes.get(CAB);
            }
            if (lregUrl == null) {
                lregUrl = regexpRes.get(ORDER_NUMBER_URL_URL);
            }
            if (lregUrl != null) {
                putValidUrls(r, lregUrl);
                keysUpdated = true;
            }
        }
        if (keysUpdated) {
            r.put(HConsts.WIDGET_SUBTYPE, ESHOP);
            r.put(HConsts.ORIGIN, REGEXP);
        }
    }

    private void putValidUrls(
        final Map<String, Object> r,
        final List<String> urls)
    {
        List<String> validUrls = urls.stream()
            .filter(s -> s.startsWith(HTTP) || s.startsWith(HTTPS))
            .collect(Collectors.toList());
        if (validUrls.size() == 1) {
            r.put(URL, validUrls.get(0));
        } else {
            if (validUrls.size() > 1) {
                r.put(URLS, validUrls);
            }
            r.remove(URL);
        }
    }

    private void changeStatusOrder(final HashMap<String, Object> r) {
        String rawStatusOrder = (String) r.get(ORDER_STATUS_TXT);
        if (rawStatusOrder == null || rawStatusOrder.isEmpty()) {
            rawStatusOrder = (String) r.get(ORDER_STATUS_MICRO);
        }
        if (rawStatusOrder == null || rawStatusOrder.isEmpty()) {
            if (subject != null) {
                rawStatusOrder = subject;
            } else {
                rawStatusOrder = "";
            }
        }
        String finalStatus = StatusDefinition.getShortStatus(rawStatusOrder);
        if (!finalStatus.isEmpty()) {
            r.put(ORDER_STATUS_MICRO, finalStatus);
        }
    }

    private void parseOrderNumber(
        final Map<?, ?> inputJson,
        final HashMap<String, Object> output)
    {
        Map<?, ?> subjson = inputJson;
        if (inputJson.containsKey(PART_OF_ORDER)) {
            subjson = (Map<?, ?>) inputJson.get(PART_OF_ORDER);
        }
        if (subjson.containsKey(ORDER_NUMBER_MICRO)) {
            output.put(ORDER_NUMBER, subjson.get(ORDER_NUMBER_MICRO));
        }
    }

    private void parseOrderStatus(
        final Map<?, ?> inputJson,
        final HashMap<String, Object> output)
    {
        Map<?, ?> subjson = inputJson;
        if (inputJson.containsKey(PART_OF_ORDER)) {
            subjson = (Map<?, ?>) inputJson.get(PART_OF_ORDER);
        }
        XJsonUtils.copyEntryAs(
            "orderStatus",
            ORDER_STATUS_MICRO,
            subjson,
            output);
    }

    private void parseOrderPrice(
        final Map<?, ?> inputJson,
        final HashMap<String, Object> output)
    {
        Object acl =
            XJsonUtils.getNodeByPathOrNullEless(inputJson, "acceptedOffer");
        StringBuilder sb = new StringBuilder();
        String currency = "";
        int priceInteger = 0;
        if (acl instanceof List) {
            for (Object x : (List) acl) {
                if (x instanceof Map) {
                    String nameOrder = parseOrderNameFromItemOffered(x);
                    if (!nameOrder.isEmpty()) {
                        sb.append(nameOrder);
                        sb.append('|');
                    }
                    if (currency.isEmpty()) {
                        Object cur =
                            XJsonUtils.getNodeByPathOrNullEless(
                                x,
                                ITEM_OFFERED,
                                PRICE_CURRENCY);
                        if (cur instanceof String) {
                            currency = (String) cur;
                        }
                    }
                    priceInteger += parsePriceFromOffer(x);
                }
            }
        } else if (acl instanceof Map) {
            priceInteger += parsePriceFromOffer(acl);
            Object rawCurrency = ((Map) acl).get(PRICE_CURRENCY);
            if (rawCurrency instanceof String) {
                currency = (String) rawCurrency;
            }
            String nameOrder = parseOrderNameFromItemOffered(acl);
            if (!nameOrder.isEmpty()) {
                sb.append(nameOrder);
                sb.append('|');
            }
        } else {
            acl = XJsonUtils.getNodeByPathOrNullEless(
                inputJson,
                "itemShipped",
                NAME);
            if (acl instanceof String) {
                sb.append(acl);
            }
        }
        if (priceInteger != 0) {
            output.put(PRICE, String.valueOf(priceInteger));
        }
        if (!currency.isEmpty()) {
            output.put(CURRENCY, currency);
        }
        if (!sb.toString().isEmpty()) {
            output.put(ORDER, sb.toString());
        }
        output.putIfAbsent(PRICE, String.valueOf(0));
        output.putIfAbsent(CURRENCY, "");
        output.putIfAbsent(ORDER, "");
    }

    private String parseOrderNameFromItemOffered(final Object x) {
        Object orderName =
            XJsonUtils.getNodeByPathOrNullEless(
                x,
                ITEM_OFFERED,
                NAME);
        if (orderName instanceof String) {
            return (String) orderName;
        }
        return "";
    }

    private int parsePriceFromOffer(final Object x) {
        int res = 0;
        Object price =
            XJsonUtils.getNodeByPathOrNullEless(
                x,
                PRICE);
        if (price instanceof String) {
            String prValue = (String) price;
            prValue = prValue.replaceAll("[,.][0-9]+", "");
            prValue = prValue.replaceAll("[\\s, ]+", "");
            try {
                res = Integer.parseInt(prValue);
            } catch (NumberFormatException e) {
            }
        }
        return res;
    }

    private void parseUrl(
        final Map<?, ?> inputJson,
        final HashMap<String, Object> output)
    {
        XJsonUtils.copyEntryAs(URL, URL, inputJson, output);
    }

    private void parseExpectedArrivalUntil(
        final Map<?, ?> inputJson,
        final HashMap<String, Object> output)
    {
        XJsonUtils.copyEntryAs(
            "expectedArrivalUntil",
            "date_delivery_rfc",
            inputJson,
            output);
    }

    private void parseTrackingNumber(
        final Map<?, ?> inputJson,
        final HashMap<String, Object> output)
    {
        XJsonUtils.copyEntryAs(
            "trackingNumber",
            "tracking_number",
            inputJson,
            output);
    }

    private void parseTrackingUrl(
        final Map<?, ?> inputJson,
        final HashMap<String, Object> output)
    {
        XJsonUtils.copyEntryAs(
            "trackingUrl",
            "tracking_url",
            inputJson,
            output);
    }

    private void parseCurrency(
        final Map<?, ?> inputJson,
        final HashMap<String, Object> output)
    {
        XJsonUtils.copyEntryAs(
            PRICE_CURRENCY,
            CURRENCY,
            inputJson,
            output);
    }

    public String getOrderNumber() {
        return orderNumber;
    }
}
